/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Marco "Cuc" Cuccato
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.util.filters.predicates;

import java.io.Serializable;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.node.security.Role;
import org.pepstock.jem.util.TimeUtils;
import org.pepstock.jem.util.filters.Filter;
import org.pepstock.jem.util.filters.FilterToken;
import org.pepstock.jem.util.filters.ParseException;
import org.pepstock.jem.util.filters.fields.RoleFilterFields;

import com.hazelcast.core.MapEntry;
import com.hazelcast.query.Predicate;

/**
 * The {@link Predicate} of a {@link Role}
 * @author Marco "Cuc" Cuccato
 * @version 1.0	
 *
 */
public class RolePredicate extends JemFilterPredicate<Role> implements Serializable {

	private static final long serialVersionUID = -1410063063130458355L;

	/**
	 * Empty contructor
	 */
	public RolePredicate() {
	}
	
	/**
	 * @see JemFilterPredicate
	 * @param filter 
	 */
	public RolePredicate(Filter filter) {
		super(filter);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public boolean apply(MapEntry entry) {
		Role role = (Role)entry.getValue();
		boolean includeThis = true;
		FilterToken[] tokens = getFilter().toTokenArray();
		long now = System.currentTimeMillis();
		for (int i=0; i<tokens.length && includeThis; i++) {
			FilterToken token = tokens[i];
			String tokenName = token.getName();
			String tokenValue = token.getValue();
			RoleFilterFields field = RoleFilterFields.getByName(tokenName);
			if (field == null) {
				field = RoleFilterFields.NAME;
			}
			
			switch (field) {
			case NAME:
				includeThis &= StringUtils.containsIgnoreCase(role.getName(), tokenValue);
				break;
			case REMOVABLE:
				includeThis &= StringUtils.containsIgnoreCase(String.valueOf(role.isRemovable()), tokenValue);
				break;
			case PERMISSIONS:
				// skipped permission isEmpty check
				includeThis &= StringUtils.containsIgnoreCase(role.getPermissions().toString(), tokenValue);
				break;
			case USERS:
				// skipped users isEmpty check
				includeThis &= StringUtils.containsIgnoreCase(role.getUsers().toString(), tokenValue);
				break;
			case MODIFIED:
				try {
					long inputTime = TimeUtils.parseDuration(tokenValue);
					long roleModifiedTime = now-role.getLastModified().getTime();
					
					includeThis &= roleModifiedTime >= inputTime;
				} catch (ParseException e) {
					// cannot parse the date, exclude this entry by default!
					includeThis &= false;
				}
				break;
			case MODIFIED_BY:
				includeThis &= StringUtils.containsIgnoreCase(role.getUser(), tokenValue);
				break;
			default:
				throw new RuntimeException("Unrecognized Role filter field: " + field);
			}
		}
		return includeThis;
	}
	
}