/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Marco "Cuc" Cuccato
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.util.filters.predicates;

import java.io.Serializable;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.ResourceProperty;
import org.pepstock.jem.util.TimeUtils;
import org.pepstock.jem.util.filters.Filter;
import org.pepstock.jem.util.filters.FilterToken;
import org.pepstock.jem.util.filters.ParseException;
import org.pepstock.jem.util.filters.fields.ResourceFilterFields;

import com.hazelcast.core.MapEntry;
import com.hazelcast.query.Predicate;

/**
 * The {@link Predicate} of a {@link Resource}
 * @author Marco "Cuc" Cuccato
 * @version 1.0	
 *
 */
public class ResourcePredicate extends JemFilterPredicate<Resource> implements Serializable {

	private static final long serialVersionUID = 8087227037699399624L;

	/**
	 * Empty contructor
	 */
	public ResourcePredicate() {
	}
	
	/**
	 * @see JemFilterPredicate
	 * @param filter 
	 */
	public ResourcePredicate(Filter filter) {
		super(filter);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public boolean apply(MapEntry entry) {
		Resource resource = (Resource)entry.getValue();
		boolean includeThis = true;
		FilterToken[] tokens = getFilter().toTokenArray();
		long now = System.currentTimeMillis();
		for (int i=0; i<tokens.length && includeThis; i++) {
			FilterToken token = tokens[i];
			String tokenName = token.getName();
			String tokenValue = token.getValue();
			ResourceFilterFields field = ResourceFilterFields.getByName(tokenName);
			if (field == null) {
				field = ResourceFilterFields.NAME;
			}
			
			switch (field) {
			case NAME:
				includeThis &= StringUtils.containsIgnoreCase(resource.getName(), tokenValue);
				break;
			case TYPE:
				includeThis &= StringUtils.containsIgnoreCase(resource.getType(), tokenValue);
				break;
			case PROPERTIES:
				int count = 0;
				String value = null;
				for (ResourceProperty property : resource.getProperties().values()){
					if (count == 0){
						value = property.getName() + " = " + (property.isVisible() ? property.getValue() : ResourceProperty.MASK_FOR_NO_VISIBLE_PROPERTY);
					} else {
						value = value +", " + property.getName() + " = " + (property.isVisible() ? property.getValue() : ResourceProperty.MASK_FOR_NO_VISIBLE_PROPERTY);	
					}
					count++;
				}
				includeThis &= StringUtils.containsIgnoreCase(value, tokenValue);
				break;
			case MODIFIED:
				try {
					long inputTime = TimeUtils.parseDuration(tokenValue);
					long resourceModifiedTime = now-resource.getLastModified().getTime();
					
					includeThis &= resourceModifiedTime >= inputTime;
				} catch (ParseException e) {
					// cannot parse the date, exclude this entry by default!
					includeThis &= false;
				}
				break;
			case MODIFIED_BY:
				includeThis &= StringUtils.containsIgnoreCase(resource.getUser(), tokenValue);
				break;
			default:
				throw new RuntimeException("Unrecognized Resource filter field: " + field);
			}
		}
		return includeThis;
	}

	
	
	
}