/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.util;

import java.net.URI;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.ws.rs.core.UriBuilder;

import org.apache.http.client.HttpClient;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.pepstock.jem.log.LogAppl;

import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.client.apache4.ApacheHttpClient4;

/**
 * Singleton to access to JEM by REST protocol. Uses a singleton to avoid to close the HTTP configuration, mandatory
 * to maintain the security management in web app.
 *  
 * @author Andrea "Stock" Stocchero
 *
 */
public class RestClient {

	private static RestClient CLIENT = null;
	
	private ApacheHttpClient4 client = null;
	
	private URI baseURI = null;
	
	/**
	 * Creates the object using the base URL of rest
	 */
	private RestClient(String uriString) {
		baseURI = UriBuilder.fromUri(uriString).build();
	    ClientConfig config = new DefaultClientConfig();
	    client = ApacheHttpClient4.create(config);
	    HttpClient hc = client.getClientHandler().getHttpClient();
	    if (baseURI.getScheme().equalsIgnoreCase("https")){
	    	try {
	    		configureSSLHandling(hc, baseURI);
	    	} catch (KeyManagementException e) {
	    		LogAppl.getInstance().emit(UtilMessage.JEMB008E, e);
	    	} catch (UnrecoverableKeyException e) {
	    		LogAppl.getInstance().emit(UtilMessage.JEMB008E, e);
	    	} catch (NoSuchAlgorithmException e) {
	    		LogAppl.getInstance().emit(UtilMessage.JEMB008E, e);
	    	} catch (KeyStoreException e) {
	    		LogAppl.getInstance().emit(UtilMessage.JEMB008E, e);
	    	}
	    }
	}

	/**
	 * Creates a new instance, overriding previuos instance
	 * @param uriString
	 * @return a rest instance
	 */
	public static RestClient createInstance(String uriString){
		CLIENT = new RestClient(uriString);
		return CLIENT;
	}
	
	/**
	 * Returns the REST instance. If null (because that hasn't been called previously the createInstance) an exception occurs.
	 * @return a rest instance
	 */
	public static RestClient getInstance(){
		if (CLIENT == null){
			throw new RuntimeException(UtilMessage.JEMB002E.toMessage().getMessage());
		} 
		return CLIENT;
	}
	
	/**
	 * Creates a base web resource to use to call remotely the method of REST services
	 * @return a web resource instance
	 */
	public WebResource getBaseWebResource(){
		return client.resource(baseURI);
	}
	
	
	private void configureSSLHandling(HttpClient hc, URI uri) throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
		SSLSocketFactory sf = buildSSLSocketFactory();
		Scheme https = new Scheme(uri.getScheme(), uri.getPort(), sf);
		SchemeRegistry sr = hc.getConnectionManager().getSchemeRegistry();
		sr.register(https);
	}
 
	private SSLSocketFactory buildSSLSocketFactory() throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
		TrustStrategy ts = new TrustStrategy() {
			@Override
			public boolean isTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
				// always true to avoif certicateunknow excpetion
				return true;
			}
		};

		/* build socket factory with hostname verification turned off. */
		return  new SSLSocketFactory(ts, SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
	}
 
}