/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.springbatch.tasks.utilities;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;

import javax.naming.InitialContext;

import org.pepstock.jem.node.tasks.jndi.ContextUtils;
import org.pepstock.jem.springbatch.SpringBatchMessage;
import org.pepstock.jem.springbatch.tasks.JemTasklet;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.repeat.RepeatStatus;

/**
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class CopyTasklet extends JemTasklet {

	private static final int BUF_SIZE = 8192;

	private static final String INPUT_DATA_DESCRIPTION_NAME = "INPUT";
	
	private static final String OUTPUT_DATA_DESCRIPTION_NAME = "OUTPUT";

	/**
	 * 
	 */
	public CopyTasklet() {
		
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.springbatch.tasks.JemTasklet#run(org.springframework.batch.core.StepContribution, org.springframework.batch.core.scope.context.ChunkContext)
	 */
	@SuppressWarnings("resource")
	@Override
	public RepeatStatus run(StepContribution stepContribution, ChunkContext chuckContext) throws Exception {

		// new initial context to access by JNDI to COMMAND DataDescription
		InitialContext ic = ContextUtils.getContext();

		// gets inputstream
		Object input = (Object) ic.lookup(INPUT_DATA_DESCRIPTION_NAME);
		// gets outputstream
		Object output = (Object) ic.lookup(OUTPUT_DATA_DESCRIPTION_NAME);

		InputStream istream = null;
		OutputStream ostream = null;

		// checks if object is a inputstream otherwise error
		if (input instanceof InputStream){
			istream = (InputStream) input;
		} else {
			throw new Exception(SpringBatchMessage.JEMS011E.toMessage().getFormattedMessage(INPUT_DATA_DESCRIPTION_NAME, input.getClass().getName()));
		}
		// checks if object is a outputstream otherwise error
		if (output instanceof OutputStream){
			ostream = (OutputStream) output;
		} else {
			throw new Exception(SpringBatchMessage.JEMS010E.toMessage().getFormattedMessage(OUTPUT_DATA_DESCRIPTION_NAME, output.getClass().getName()));
		}

		// if the input stream is a fileinput stream it uses the channel (faster!) 
		if (istream instanceof FileInputStream){
			FileChannel ichannel = ((FileInputStream)istream).getChannel();
			ichannel.transferTo(0, ichannel.size(), Channels.newChannel(ostream));
			ichannel.close();
		} else if (ostream instanceof FileOutputStream){ 
			// if the output stream is a fileoutput stream it uses the channel (faster!)
			FileChannel ochannel = ((FileOutputStream)ostream).getChannel();
			ochannel.transferFrom(Channels.newChannel(istream), 0, Long.MAX_VALUE);
			ochannel.close();
		} else {
			// otherwise normal read and write
			byte[] buffer = new byte[BUF_SIZE];
			int count = 0;
			do {
				ostream.write(buffer, 0, count);
				count = istream.read(buffer, 0, buffer.length);
			} while (count != -1);
			ostream.close();
			istream.close();
		}

		return  RepeatStatus.FINISHED;
	}

}