/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.springbatch.tasks;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Reference;
import javax.naming.StringRefAddr;

import org.pepstock.catalog.DataDescriptionImpl;
import org.pepstock.catalog.DataSetImpl;
import org.pepstock.catalog.DataSetType;
import org.pepstock.catalog.Disposition;
import org.pepstock.catalog.gdg.GDGManager;
import org.pepstock.jem.node.resources.FtpResource;
import org.pepstock.jem.node.resources.HttpResource;
import org.pepstock.jem.node.resources.JdbcResource;
import org.pepstock.jem.node.resources.JmsResource;
import org.pepstock.jem.node.resources.JppfResource;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.ResourceProperty;
import org.pepstock.jem.node.rmi.CommonResourcer;
import org.pepstock.jem.node.tasks.InitiatorManager;
import org.pepstock.jem.node.tasks.jndi.DataStreamReference;
import org.pepstock.jem.node.tasks.jndi.FtpReference;
import org.pepstock.jem.node.tasks.jndi.HttpReference;
import org.pepstock.jem.node.tasks.jndi.JdbcReference;
import org.pepstock.jem.node.tasks.jndi.JmsReference;
import org.pepstock.jem.node.tasks.jndi.JppfReference;
import org.pepstock.jem.node.tasks.jndi.StringRefAddrKeys;
import org.pepstock.jem.springbatch.SpringBatchMessage;
import org.pepstock.jem.springbatch.tasks.managers.ImplementationsContainer;
import org.springframework.batch.core.StepContribution;
import org.springframework.batch.core.scope.context.ChunkContext;
import org.springframework.batch.core.scope.context.StepContext;
import org.springframework.batch.core.step.tasklet.Tasklet;
import org.springframework.batch.repeat.RepeatStatus;

import com.thoughtworks.xstream.XStream;

/**
 * Is a JEM implementation of a tasklet.<br>
 * If needs to use special features of JEM (like GDG or looking), must be
 * extended, implementing abstract method.<br>
 * Loads a JNDI file resources using DataDescription so the implementation can access
 * in abstract way to resources defined in JCL.<br>
 * Loads a JNDI datasource resources using DataSource so the implementation can access
 * in abstract way to resources defined in JCL.
 * 
 * @see org.springframework.batch.core.step.tasklet.Tasklet
 * @author Andrea "Stock" Stocchero
 * 
 */
public abstract class JemTasklet implements Tasklet {

	private ArrayList<DataDescription> dataDescriptionList = new ArrayList<DataDescription>();
	
	private ArrayList<DataSource> dataSourceList = new ArrayList<DataSource>();
	
	private ArrayList<Lock> locks = new ArrayList<Lock>(); 

	/**
	 * Empty constructor
	 */
	public JemTasklet() {
	}

	/**
	 * Returns the list of data description defined for this tasklet.
	 * 
	 * @return the list of data description
	 */
	public ArrayList<DataDescription> getDataDescriptionList() {
		return dataDescriptionList;
	}

	/**
	 * Sets the list of data description
	 * 
	 * @param dataDescriptionList the list of data description
	 */
	public void setDataDescriptionList(ArrayList<DataDescription> dataDescriptionList) {
		this.dataDescriptionList = dataDescriptionList;
	}

	/**
	 * Returns the list of data sources defined for this tasklet.
	 * 
	 * @return the dataSourceList
	 */
	public ArrayList<DataSource> getDataSourceList() {
		return dataSourceList;
	}

	/**
	 * Sets the list of data sources
	 * 
	 * @param dataSourceList the dataSourceList to set
	 */
	public void setDataSourceList(ArrayList<DataSource> dataSourceList) {
		this.dataSourceList = dataSourceList;
	}

	/**
	 * @return the locks
	 */
	public ArrayList<Lock> getLocks() {
		return locks;
	}

	/**
	 * @param locks the locks to set
	 */
	public void setLocks(ArrayList<Lock> locks) {
		this.locks = locks;
	}

	/**
	 * Is called by SpringBatch framework to execute business logic.<br>
	 * Prepares datasets (and the files and resources) which could be used from
	 * implementation of this class.<br>
	 * Loads JNDI context so all resources could be used by their name, defined
	 * in JCL.
	 * 
	 * @param stepContribution step contribution, passed by SpringBatch core
	 * @param chunkContext chunk context, passed by SpringBatch core
	 * @return always the status returned by abstract method
	 *         <code>executeByJem</code>
	 * @throws Exception if a error occurs
	 */
	public final RepeatStatus execute(StepContribution stepContribution, ChunkContext chunkContext) throws Exception {
		// this boolean is necessary to understand if I have an exception 
		// before calling the main class
		boolean isExecutionStarted = false;
		
		SpringBatchSecurityManager batchSM = (SpringBatchSecurityManager)System.getSecurityManager();
		batchSM.setInternalAction(true);
		
		System.setProperty(Context.INITIAL_CONTEXT_FACTORY, "org.pepstock.jem.node.tasks.jndi.JemContextFactory");
		
		RepeatStatus status = null;

		// extract stepContext because the step name is necessary
		StepContext stepContext = chunkContext.getStepContext();

		List<DataDescriptionImpl> dataDescriptionImplList = ImplementationsContainer.getInstance().getDataDescriptionsByItem(stepContext.getStepName());

		try {
			// new initial context for JNDI
			InitialContext ic = new InitialContext();
			// scans all datasource passed
			for (DataSource source : dataSourceList){
				// checks if datasource is well defined
				if ((source.getName() == null) || (source.getResource() == null)){
					throw new Exception(SpringBatchMessage.JEMS016E.toMessage().getFormattedMessage());
				}
				// gets the RMi object to get resources
				CommonResourcer resourcer = InitiatorManager.getCommonResourcer();
				// lookups by RMI for the database 
				Resource res = resourcer.lookup(source.getResource());
				if (!batchSM.checkResource(res))
					throw new Exception(SpringBatchMessage.JEMS017E.toMessage().getFormattedMessage(res.toString()));

				// all properties create all StringRefAddrs necessary
				HashMap<String, ResourceProperty> properties = res.getProperties();
				// for testing log(properties.toString());
				// scans all properteis set by JCL
				for (Property property : source.getProperties()){
					// if a key is defined FINAL, throw an exception
					for (ResourceProperty resProperty : properties.values()){
						if (resProperty.getName().equalsIgnoreCase(property.getName())){
							if (!resProperty.isOverride())
								throw new Exception(SpringBatchMessage.JEMS018E.toMessage().getFormattedMessage(property.getName(), res));
							
						}
					}
					res.setProperty(property.getName(), property.getValue());
				}
				// creates a JNDI reference
				Reference ref = null;
				// only JBDC, JMS and FTP types are accepted 
				if (res.getType().equalsIgnoreCase(JdbcResource.TYPE)){
					// creates a JDBC reference (uses DBCP Apache)
					ref = new JdbcReference();
					
				}else if(res.getType().equalsIgnoreCase(JppfResource.TYPE)){
					// creates a JPPF reference 
					ref = new JppfReference();		
					
				}else if(res.getType().equalsIgnoreCase(JmsResource.TYPE)){
					// creates a JMS reference (uses javax.jms)
					ref = new JmsReference();
				
				}else if(res.getType().equalsIgnoreCase(HttpResource.TYPE)){
					// creates a HTTP reference (uses org.apache.http)
					ref = new HttpReference();

				} else if (res.getType().equalsIgnoreCase(FtpResource.TYPE)){
					// creates a FTP reference (uses Commons net Apache)
					ref = new FtpReference();
					
					
					// checks if I have a dataset linked to a datasource
					for (DataDescriptionImpl ddImpl : dataDescriptionImplList) {
						for (DataSetImpl ds: ddImpl.getDataSetsImpl()){
							// if has resource linked
							if (ds.getType() == DataSetType.RESOURCE){
								// checks if teh name is the same
								if (ds.getDataSource().equalsIgnoreCase(source.getName())){
									// sets file name (remote one)
									res.setProperty(FtpResource.REMOTE_FILE, ds.getName());
									// sets if wants to have a OutputStream or InputStream using
									// disposition of dataset
									if (!ddImpl.getDisposition().equalsIgnoreCase(Disposition.SHR)){
										res.setProperty(FtpResource.ACTION_MODE, FtpResource.ACTION_WRITE);
									} else {
										res.setProperty(FtpResource.ACTION_MODE, FtpResource.ACTION_READ);
									}
								}
							}
						}
					}
				} else {
					try {
						ref = resourcer.lookupCustomResource(res.getType());
						if (ref == null){
							throw new Exception(SpringBatchMessage.JEMS019E.toMessage().getFormattedMessage(res.getName(), res.getType()));
						}
					} catch (Exception e) {
						throw new Exception(SpringBatchMessage.JEMS019E.toMessage().getFormattedMessage(res.getName(), res.getType()));
					} 
				}

				// loads all properties into RefAddr
				for (ResourceProperty property : properties.values()){
					ref.add(new StringRefAddr(property.getName(), property.getValue()));
				}
				
				// for testing log(res.getType()+"/"+source.getName());
				// for testing log(properties.toString());
				// binds the object with format {type]/[name]
				//ic.rebind(res.getType()+"/"+source.getName(), ref);
				System.out.println(SpringBatchMessage.JEMS024I.toMessage().getFormattedMessage(res));
				ic.rebind(source.getName(), ref);
			}

			// check if I have resources which must be locked
			if (!dataDescriptionImplList.isEmpty()) {

				// binds all data description impl to JNDI context
				for (DataDescriptionImpl ddImpl : dataDescriptionImplList) {

					// create reference for JNDI access
					Reference reference = new DataStreamReference();

					// load GDG information, solving the real name of relative
					// position
					GDGManager.load(ddImpl);
					// serialize data description object in XML format
					XStream xstream = new XStream();
					String xml = xstream.toXML(ddImpl);
					// add string xml reference
					reference.add(new StringRefAddr(StringRefAddrKeys.DATASTREAMS_KEY, xml));

					System.out.println(SpringBatchMessage.JEMS023I.toMessage().getFormattedMessage(ddImpl));
					// bind resource using data description name
					ic.rebind(ddImpl.getName(), reference);
				}
			}
			batchSM.setInternalAction(false);
			// execute business logic
			// executes the java class defined in JCL
			// setting the boolean to TRUE
			isExecutionStarted = true;
			status = this.run(stepContribution, chunkContext);
		} catch (Exception e1) {
			throw e1;
		} finally {
			batchSM.setInternalAction(true);
			if (!dataDescriptionImplList.isEmpty()) {
				StringBuffer exceptions = new StringBuffer();
				// scans data descriptions
				for (DataDescriptionImpl ddImpl : dataDescriptionImplList) {
					try {
						// commit the GDG index in the root
						// if an exception, write on standard output of job
						// only if execution started
						if (isExecutionStarted)
							GDGManager.store(ddImpl);
					} catch (IOException e) {
						System.out.println(SpringBatchMessage.JEMS025E.toMessage().getFormattedMessage(e.getMessage()));
						if (exceptions.length() == 0)
							exceptions.append(SpringBatchMessage.JEMS025E.toMessage().getFormattedMessage(e.getMessage()));
						else 
							exceptions.append(SpringBatchMessage.JEMS025E.toMessage().getFormattedMessage(e.getMessage())).append("\n");

					}
				}
				if (exceptions.length() > 0){
					throw new Exception(exceptions.toString());
				}
			}
		}
		return status;
	}

	/**
	 * Is abstract method to implement with business logic, where it's possible
	 * to access to resources by JNDI.
	 * 
	 * @param stepContribution step contribution
	 * @param chuckContext chunk context
	 * @return status of execution
	 * @throws Exception if errors occur
	 */
	public abstract RepeatStatus run(StepContribution stepContribution, ChunkContext chuckContext) throws Exception;
}
