/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.springbatch.tasks;

import java.io.Serializable;
import java.util.ArrayList;

import org.pepstock.catalog.Disposition;

/**
 * Is a container of datasets and unit of work for all IO operation (read and
 * write).<br>
 * Can be accessed by name.
 * 
 * @see DataSet
 * @author Andrea "Stock" Stocchero
 * 
 */
public class DataDescription implements Serializable {

	private static final long serialVersionUID = 1L;

	private ArrayList<DataSet> datasets = new ArrayList<DataSet>();

	private String name = null;

	private String disposition = Disposition.SHR;

	private boolean sysout = false;

	/**
	 * Empty constructor
	 */
	public DataDescription() {
	}

	/**
	 * Returns the name. This name is used on Naming directory and then used to
	 * retrieve the streams
	 * 
	 * @return name string
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the name. This name is used on Naming directory and then used to
	 * retrieve the streams
	 * 
	 * @param name name string
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Sets a list of dataset definition.
	 * 
	 * @see DataSet
	 * @param datasets list of datasets
	 */
	public void setDatasets(ArrayList<DataSet> datasets) {
		this.datasets = datasets;
	}

	/**
	 * Gets the list of dataset related to this data description
	 * 
	 * @return list of datasets
	 */
	public ArrayList<DataSet> getDatasets() {
		return datasets;
	}

	/**
	 * Returns the disposition for data description
	 * 
	 * @see org.pepstock.catalog.Disposition
	 * @return disposition string
	 */
	public String getDisposition() {
		return disposition;
	}

	/**
	 * Sets the disposition
	 * 
	 * @see org.pepstock.catalog.Disposition
	 * @param disposition
	 */
	public void setDisposition(String disposition) {
		this.disposition = disposition;
	}

	/**
	 * Returns <code>true</code> if the data description is defined for sysout,
	 * so only to print on output path.
	 * 
	 * @return <code>true</code> if the data description is defined for sysout,
	 *         so only to print on output path
	 */
	public boolean isSysout() {
		return sysout;
	}

	/**
	 * Sets if the data description is defined for sysout.
	 * 
	 * @param sysout <code>true</code> if is a sysout
	 */
	public void setSysout(boolean sysout) {
		this.sysout = sysout;
	}

	/**
	 * Returns <code>true</code> if there is only 1 datasets in list.
	 * 
	 * @return <code>true</code> if there is only 1 datasets in list
	 */
	public boolean isSingleDataset() {
		return (datasets.size() == 1);
	}

	/**
	 * Returns <code>true</code> if there are more than 1 datasets in list. if
	 * multidatasets, it could be used only for reading.
	 * 
	 * @return <code>true</code> if there are more than 1 datasets in list
	 */
	public boolean isMultiDataset() {
		return (datasets.size() > 1);
	}

	/**
	 * Return string representation of a DataDescriptor
	 */
	@Override
	public String toString() {
		if (isSingleDataset()) {
			DataSet ds = datasets.get(0);
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", " + ds.toString() + "]";
		} else if (!datasets.isEmpty()) {
			StringBuffer sb = new StringBuffer("[dd=" + getName() + ", disp=" + getDisposition() + ",\n");
			for (DataSet ds : datasets) {
				sb.append("    ").append(ds.toString()).append(",").append('\n');
			}
			sb.append("]");
			return sb.toString();
		} else if (sysout) {
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", SYSOUT]";

		}
		return super.toString();
	}

}