/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea 'Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.springbatch;

import org.pepstock.jem.log.Description;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.log.MessageInterface;

/**
 * It is an enumeration containing all the messages related to the ANT utilities. <br>
 * It implements {@link MessageInterface}
 * 
 * @see Message
 * @author Andrea 'Stock" Stocchero
 * @version 1.0	
 */
public enum SpringBatchMessage implements MessageInterface{

	/**
	 * "Data path is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when the system property <code>jem.data</code> is null.<br>Check node log because some exception should be occurred.")
	JEMS001E(1, "Data path is null", Message.ERROR),
	
	/**
	 * "RMI port is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when the system property <code>jem.rmi.port</code> is null.<br>Check node log because some exception should be occurred.")
	JEMS002E(2, "RMI port is null", Message.ERROR),
	
	/**
	 * "Data description doesn't have 'name' attribute", Message.ERROR
	 */
	@Description(explanation = "It occurs when a data description in JCL does't have <code>name</code> attribute.<br>Check data description definition.")
	JEMS003E(3, "Data description doesn't have 'name' attribute", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : MultiDataset is not allowed in STEP-EXEC", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a data description with many datasets to write and this is not allowed."+
	 "<br>Check data description definition.")
	JEMS004E(4, "Data description \"{0}\", disposition \"{1}\" : MultiDataset is not allowed with ItemWriter", Message.ERROR),
	
	/**
	 * "Output path is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when the system property <code>jem.output</code> is null.<br>Check node log because some exception should be occurred.")
	JEMS005E(5, "Output path is null", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemReader: only SHR is allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to access to data description with ItemReader without using SHR disposition."+
	 "<br>Check data description definition.")
	JEMS006E(6, "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemReader: only SHR is allowed", Message.ERROR),	
	

	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a multi dataset: only SHR is allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to access to temporary dataset without using NEW disposition which is mandatory."+
	 "<br>Check data description definition.")
	JEMS007E(7, "Data description \"{0}\", disposition \"{1}\", \"{2}\" : Disposition wrong with a temporary file: only NEW is allowed", Message.ERROR),	
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Dataset doesn't have 'name' attribute", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a data set without <code>name</code> attribute."+
	 "<br>Check data description definition.")
	JEMS008E(8, "Data description \"{0}\", disposition \"{1}\" : Dataset doesn't have 'name' attribute", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a multi dataset: only SHR is allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to access to dataset using NEW disposition but the data set already exists."+
	 "<br>Check data description definition and file system.")
	JEMS009E(9, "Data description \"{0}\", disposition \"{1}\", \"{2}\" : Dataset already exists", Message.ERROR),	
	
	/**
	 *"Data description name \"{0}\" is not a OutputStream instance but \"{1}\"", Message.ERROR
	 */
	@Description(explanation = "It occurs when data description is not a output stream, as expect<br>Check JCL definition.")
	JEMS010E(10, "Data description name \"{0}\" is not a OutputStream instance but \"{1}\"", Message.ERROR),

	/**
	 *"Data description name \"{0}\" is not a InputStream instance but \"{1}\"", Message.ERROR
	 */
	@Description(explanation = "It occurs when data description is not a input stream, as expect<br>Check JCL definition.")
	JEMS011E(11, "Data description name \"{0}\" is not a InputStream instance but \"{1}\"", Message.ERROR),
	
	/**
	 * "Data source value is empty", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a data set using data source attribute which is empty even if defined."+
	 "<br>Check data description definition.")
	JEMS012E(12, "Data source value is empty", Message.ERROR),
	
	/**
	 * "Data source value is empty", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a data source but the name attribute is empty even if defined."+
	 "<br>Check data source definition.")
	JEMS013E(13, "Data source name attribute is empty", Message.ERROR),
	
	/**
	 * "Data source value is empty", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a data source but the resource attribute is empty even if defined."+
	 "<br>Check data source definition.")
	JEMS014E(14, "Data source resource attribute is empty", Message.ERROR),
	
	/**
	 * "Lock name attribute is empty", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a lock but the name attribute is empty even if defined."+
	 "<br>Check lock definition.")
	JEMS015E(15, "Lock name attribute is empty", Message.ERROR),
	
	/**
	 * "Data source is not well defined: 'name' or 'resource' is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when data source element doesn't contain <code>name</code> or <code>resource</code>."+
	 "<br>Check JCL data source definition.")
	JEMS016E(16, "Data source is not well defined: 'name' or 'resource' is null", Message.ERROR),
	
	/**
	 * "Datasource is not well defined: 'name' or 'resource' is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to access to a resource but you're not authorized."+
	 "<br>Check permissions on resources that job user has got.")
	JEMS017E(17, "Access to \"{0}\" is not allowed", Message.ERROR),
	
	/**
	 * "Key  \"{0}\" for resource  \"{1}\" can not be overrided", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to override a property which is defined as not overridable."+
	 "<br>Check resource definition.")
	JEMS018E(18, "Key  \"{0}\" for resource  \"{1}\" can not be overrided", Message.ERROR),
	
	/**
	 * "Resorce \"{0}\" with type \"{1}\" is not supported", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to use a resource which has go a type not supported."+
	 "<br>Check with JEM administrator which types of resources are available.")
	JEMS019E(19, "Resorce \"{0}\" with type \"{1}\" is not supported", Message.ERROR),
	
	/**
	 * "Job name is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when the job name is missing.<br>Check JCL and fill name attribute project element or use <code>jem.job.name</code> property.")
	JEMS020E(20, "Job name is null", Message.ERROR),
	
	/**
	 * "Job name is null", Message.ERROR
	 */
	@Description(explanation = "It occurs when the <code>lockingScope</code> property is wrong.<br>Check JCL using the right locking scope.")
	JEMS021E(21, "Property 'lockingScope' must be "+SpringBatchKeys.JOB_SCOPE+"' or '"+SpringBatchKeys.STEP_SCOPE+", instead of \"{0}\"", Message.ERROR),
	
	/**
	 * "Job name is null", Message.ERROR
	 */
	@Description(explanation = "It informs what kind of locking scope the job is using.")
	JEMS022I(22, "Job \"{0}\" is using \"{1}\" locking scope.", Message.INFO),
	
	/**
	 * "Data description reference: \"{0}\"", Message.INFO
	 */
	@Description(explanation = "It informs what data description is currently used.")
	JEMS023I(23, "Data description instance: \"{0}\"", Message.INFO),
	
	/**
	 * "Data source reference: \"{0}\"", Message.INFO
	 */
	@Description(explanation = "It informs what data source is currently used.")
	JEMS024I(24, "Data source instance: \"{0}\"", Message.INFO),
	
	/**
	 * "Unable to store GDG", Message.ERROR
	 */
	@Description(explanation = "It occurs when an exception is thrown during the commit phase on GDG.<br>Check with JEM adnimistrator.")
	JEMS025E(25, "Unable to complete storing GDG : {0}", Message.ERROR),
	
	/**
	 * "RMI object {0} is not binded", Message.ERROR
	 */
	JEMS026E(26, "RMI object {0} is not binded", Message.ERROR),
	
	/**
	 * "Security Manager already instantiated.", Message.ERROR
	 */
	JEMS027E(27, "Security Manager already instantiated.", Message.ERROR),
	/**
	 * "Error while starting job {0}", Message.ERROR
	 */
	JEMS028E(28, "Error while starting job {0}", Message.ERROR),
	
	/**
	 * "Dataset name attribute is empty", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to define a dataset but the name attribute is empty even if defined."+
	 "<br>Check lock definition.")
	JEMS029E(29, "Dataset name attribute is empty", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemWriter: SHR is not allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when you try to access to data description with ItemWriter using SHR disposition."+
	 "<br>Check data description definition.")
	JEMS030E(30, "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemWriter: SHR is not allowed", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemWriter: SHR is not allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when a item sets resources which are read by JCL but the container has more than 1 resource or null"+
	 "<br>Check data description definition.")
	JEMS031E(31, "Resources instance is invalid. Must 1 but is {0}", Message.ERROR),
	
	/**
	 * "Data description \"{0}\", disposition \"{1}\" : Disposition wrong with a ItemWriter: SHR is not allowed", Message.ERROR
	 */
	@Description(explanation = "It occurs when a item doesn't have any delegate."+
	 "<br>Check data description definition.")
	JEMS032E(32, "Delegate is mandatory property to set. Instance is null", Message.ERROR),
	
	/**
	 * "Input resource must not be null, must exists, must be readabl and must be a Dataset.", Message.ERROR
	 */
	@Description(explanation = "It occurs when a item is not able to read the resource for one of described cases."+
	 "<br>Check data description definition.")
	JEMS033E(33, "Input resource must not be null, must exists, must be readabl and must be a Dataset.", Message.ERROR),
	
	/**
	 * ""Reader must be open before it can be read", Message.ERROR
	 */
	@Description(explanation = "It occurs when a reader engine is null during the read phase."+
	 "<br>Check how the item reader is called.")
	JEMS034E(34, "Reader must be open before it can be read", Message.ERROR),
	
	/**
	 * "JEM bean is not defined", Message.ERROR
	 */
	@Description(explanation = "It occurs when JEM bean is not present in JCL."+
	 "<br>Check JCL and add JEM bean defining all necessary attributes.")
	JEMS035E(35, "JEM bean is not defined", Message.ERROR),
	
	/**
	 * "JEM bean is not a JemBean instance but \"{0}\"", Message.ERROR
	 */
	@Description(explanation = "It occurs when for JEM bean it's defined a different class in JCL."+
	 "<br>Check JCL and correct the JEM bean defintion using JemBean class.")
	JEMS036E(36, "JEM bean is not a JemBean instance but \"{0}\"", Message.ERROR),
	
	/**
	 * "Unable to create dataset implementation: {0}", Message.WARNING);
	 */
	@Description(explanation = "It occurs when the reader is not able to create a dataset implementation."+
	 "<br>Check JCL and correct the JCL or contact your JM administrator.")
	JEMS037W(37, "Unable to create dataset implementation: {0}", Message.WARNING);
	
	
	/**
	 * Specific message code identifier for <code>NodeMessage</code>. 
	 * It will be displayed in the log.
	 */
	private final static String MESSAGE_ID = "S";
	
	/**
	 * The {@link Message} created in the constructor corresponding to instances of ANT utilities. 
	 * @see Message
	 */
	private Message message;
	
	/**
	 * Constructor. It builds a <code>Message</code>. <br>
	 * This method uses the same parameter of the <code>Message</code> constructor.
	 * 
	 * @param code identifier ID
	 * @param msg string to display. Could contain variables, resolved at runtime
	 * @param level severity of log message
	 * @see Message
	 */
	private SpringBatchMessage(int code, String messageContent, int level){
		this.message = new Message(code, MESSAGE_ID, messageContent, level);
	}
	
	/**
	 */
	@Override
	public Message toMessage(){
		return this.message;
	}
}