/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.springbatch;

import org.pepstock.jem.Jcl;

/**
 * Contains all attributes to assign to job. it's a bean so you could add to
 * SpringBatch code.<br>
 * Only JobName is mandatory. All others are optional, using a default value.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public final class JemBean {
	
	private String jobName = null;

	private String environment = null;

	private String domain = Jcl.DEFAULT_DOMAIN;

	private String affinity = Jcl.DEFAULT_AFFINITY;
	
	private int memory = Jcl.DEFAULT_MEMORY;

	private int priority = Jcl.DEFAULT_PRIORITY;

	private boolean hold = false;

	private String classPath = null;
	
	private String lockingScope = SpringBatchKeys.JOB_SCOPE;
	
	/**
	 * Field containing the email addresses where to send the end
	 * {@link Job} notification. The addresses are semicolon separated.
	 */
	private String emailsNotification = null;
	
	private String user = null;
	
	private String options= null;
	
	private String parameters = null;
	
	/**
	 * Empty constructor
	 */
	public JemBean() {
	}

	/**
	 * Returns the name string for job, or null if none
	 * 
	 * @return the name string for job
	 */
	public String getJobName() {
		return jobName;
	}

	/**
	 * Sets the name string for job.
	 * 
	 * @param jobName the name string for job
	 */
	public void setJobName(String jobName) {
		this.jobName = jobName;
	}

	/**
	 * Returns the name of cluster (see Hazelcast "group" definition) which the
	 * job must be executed in.
	 * 
	 * @return the name of cluster
	 */
	public String getEnvironment() {
		return environment;
	}

	/**
	 * Sets the name of cluster which the job must be executed in. It must be
	 * the same of Hazelcast group name otherwise the job will move on "routing"
	 * queue for another cluster.
	 * 
	 * @param environment the name of cluster
	 */
	public void setEnvironment(String environment) {
		this.environment = environment;
	}

	/**
	 * Sets the name of affinity which job must be executed in. If none of
	 * members is defined with this domain, job will stay in "input" queue,
	 * waiting for an initiator with this domain.
	 * 
	 * @return the name of affinity
	 */
	public String getAffinity() {
		return affinity;
	}

	/**
	 * Sets the name of affinity which job must be executed in. If none of
	 * members is defined with this domain, job will stay in "input" queue,
	 * waiting for an initiator with this domain.
	 * 
	 * @param affinity the name of affinity
	 */
	public void setAffinity(String affinity) {
		this.affinity = affinity;
	}

	/**
	 * Returns the name of domain. Domain is a subset of nodes which job must be
	 * executed in.
	 * 
	 * @return the name of domain
	 */
	public String getDomain() {
		return domain;
	}

	/**
	 * Sets the name of domain which job must be executed in. If none of members
	 * is defined with this domain, job will stay in "input" queue, waiting for
	 * an initiator with this domain.
	 * 
	 * @param domain the name of domain
	 */
	public void setDomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Gets the email addresses where to send
	 * end {@link Job} notification emails.
	 * 
	 * @return the email addresses for notification
	 */
	public String getEmailsNotification() {
		return this.emailsNotification;
	}

	/**
	 * Sets the email addresses where to send the
	 * end {@link Job} notification emails.
	 * 
	 * @param addresses the email addresses for notification, semicolon separated.
	 */
	public void setEmailsNotification(String addresses) {
		this.emailsNotification = addresses;
	}
	
	/**
	 * Returns the memory estimated necessary to execute the job
	 * 
	 * @return the memory in MB
	 */
	public int getMemory() {
		return memory;
	}

	/**
	 * Sets the memory estimated necessary to execute the job
	 * 
	 * @param memory the memory in MB to set
	 */
	public void setMemory(int memory) {
		this.memory = memory;
	}

	/**
	 * Returns the priority for job, in "input" and "output" queues. Default is
	 * 10.
	 * 
	 * @return the priority of job when queued
	 */
	public int getPriority() {
		return priority;
	}

	/**
	 * Sets the priority for job, in "input" and "output" queues.
	 * 
	 * @param priority the priority of job when queued
	 */
	public void setPriority(int priority) {
		this.priority = priority;
	}

	/**
	 * Returns if job is blocked in "input" and "output" queues. Default is
	 * false.
	 * 
	 * @return true if job is hold, otherwise false
	 */
	public boolean isHold() {
		return hold;
	}

	/**
	 * Sets if job is blocked in "input" and "output" queues. Default is false.
	 * 
	 * @param hold true if job is hold, otherwise false
	 */
	public void setHold(boolean hold) {
		this.hold = hold;
	}

	/**
	 * Returns the class path necessary to Springbatch to execute job.
	 * This is not the System class path.
	 * 
	 * @return the classPath
	 */
	public String getClassPath() {
		return classPath;
	}

	/**
	 * Sets the class path necessary to Springbatch to execute job.
	 * This is not the System class path.
	 * 
	 * @param classPath the classPath to set
	 */
	public void setClassPath(String classPath) {
		this.classPath = classPath;
	}

	/**
	 * Returns the user set inside the SpringBatch 
	 * @return the user
	 */
	public String getUser() {
		return user;
	}

	/**
	 * Sets the user inside the SpringBatch 
	 * @param user the user to set
	 */
	public void setUser(String user) {
		this.user = user;
	}

	/**
	 * @return the lockingScope
	 */
	public String getLockingScope() {
		return lockingScope;
	}

	/**
	 * @param lockingScope the lockingScope to set
	 */
	public void setLockingScope(String lockingScope) {
		this.lockingScope = lockingScope;
	}
	
	/**
	 * @return the options
	 */
	public String getOptions() {
		return options;
	}

	/**
	 * @param options the options to set
	 */
	public void setOptions(String options) {
		this.options = options;
	}

	/**
	 * @return the parameters
	 */
	public String getParameters() {
		return parameters;
	}

	/**
	 * @param parameters the parameters to set
	 */
	public void setParameters(String parameters) {
		this.parameters = parameters;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "JemBean [jobName=" + jobName + ", environment=" + environment + ", domain=" + domain + ", affinity=" + affinity + ", memory=" + memory + ", priority=" + priority + ", hold=" + hold + ", classPath=" + classPath + ", lockingScope="
				+ lockingScope + ", emailsNotification=" + emailsNotification + ", user=" + user + ", options=" + options + ", parameters=" + parameters + "]";
	}

}