/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.tasks.platform;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import org.apache.commons.io.FilenameUtils;
import org.pepstock.jem.Job;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.Main;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.tasks.shell.JavaCommand;
import org.pepstock.jem.node.tasks.shell.Shell;

/**
 * @author Andrea "Stock" Stocchero
 * @version 1.3
 *
 */
public class WindowsPlatform extends AbstractPlatform {
	
	/**
	 * Default shell command for WINDOWS environment
	 */
	public static final String DEFAULT_WINDOWS_SHELL_NAME = "cmd.exe";

	/**
	 * Default shell parameters for WINDOWS environment
	 */
	public static final String DEFAULT_WINDOWS_SHELL_PARAMETERS = "/C";
	
	private final Shell SHELL = new Shell(DEFAULT_WINDOWS_SHELL_NAME, DEFAULT_WINDOWS_SHELL_PARAMETERS);
	
	/**
	 * Constant name of JOB file to execute by shell. Value is "job.sh". The file contains the JOB
	 * start command by SUDO
	 */
	public static final String JOB_FILE_CMD = "job.cmd";

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.Platform#getShell()
	 */
	@Override
	public Shell getShell() {
		return SHELL;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.Platform#getCommand(org.pepstock.jem.node.tasks.shell.JavaCommand, boolean)
	 */
	@Override
	public String getCommand(Job job, JavaCommand command, boolean sudo) throws Exception {
		//" 
		String commandToExecute = null;
		File logFile = Main.OUTPUT_SYSTEM.getMessagesLogFile(job);
		String redirect = "> "+FilenameUtils.normalize(logFile.getAbsolutePath(), true)+" 2>&1";
		if (sudo){
			File outputFolder = Main.OUTPUT_SYSTEM.getOutputPath(job);
			File scriptFile = new File(outputFolder, JOB_FILE_CMD);
			write(scriptFile, job, command);
			commandToExecute =  scriptFile.getAbsolutePath()+" "+redirect;
		} else {
			commandToExecute =  command.toCommandLine()+" "+redirect;
		}
		return commandToExecute;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.Platform#kill(int, java.lang.String, boolean)
	 */
	@Override
	public boolean kill(long pid, String user, boolean force, boolean sudo) {
		boolean isCancelled = true;
		
		Process p = null;
		try {
			// on windows you're not able to cancel a child process without force
			// so ONLY force cancel is allowed
			p = Runtime.getRuntime().exec("taskkill /T /F /PID "+pid);
			int rc = p.waitFor();
			if (rc != 0){
				isCancelled = false;
			}
		} catch (IOException e) {
			isCancelled = false;
			LogAppl.getInstance().emit(NodeMessage.JEMC017E, e);
		} catch (InterruptedException e) {
			isCancelled = false;
			LogAppl.getInstance().emit(NodeMessage.JEMC017E, e);
		} catch (Exception e) {
			isCancelled = false;
			LogAppl.getInstance().emit(NodeMessage.JEMC017E, e);
		} finally {
			if (p != null)
				p.destroy();
		}
		return isCancelled;
	}
	

	/**
	 * 
	 * @param file
	 * @param job
	 * @param jCommand
	 * @throws IOException
	 */
	private void write(File file, Job job, JavaCommand jCommand) throws IOException{
		PrintStream fos = null;
		try {
			fos = new PrintStream(new FileOutputStream(file));
			fos.println("@echo off");
			if (jCommand.getClassPath() != null){
				String pathSeparator = File.pathSeparator;
				String classPathProperty = jCommand.getClassPath();
				String[] filesNames = classPathProperty.split(pathSeparator);
				for (int i=0; i<filesNames.length; i++){
					if (i==0){
						fos.println("set CLASSPATH="+filesNames[i]);
					} else {
						fos.println("set CLASSPATH=%CLASSPATH%;"+filesNames[i]);
					}
				}
			}
			fos.println(jCommand.toCommandLine());
			fos.flush();
		} finally {
			if (fos != null)
				fos.close();
		}
		file.setExecutable(true, false);
	}
}