/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.tasks.jndi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Hashtable;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.RefAddr;
import javax.naming.Reference;
import javax.naming.spi.ObjectFactory;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;
import org.apache.commons.net.ftp.FTPSClient;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.resources.FtpResource;
import org.pepstock.jem.util.Parser;

/**
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class FtpFactory implements ObjectFactory {

	/**
	 * 
	 */
	@Override
	public Object getObjectInstance(Object obj, Name name, Context nameCtx, @SuppressWarnings("rawtypes") Hashtable environment) throws Exception {

		if ((obj == null) || !(obj instanceof Reference)) {
			return null;
		}
		Reference ref = (Reference) obj;
		Properties properties = new Properties();
		for (int i = 0; i < FtpResource.PROPERTIES_ALL.size(); i++) {
			String propertyName = FtpResource.PROPERTIES_ALL.get(i);
			RefAddr ra = ref.get(propertyName);
			if (ra != null) {
				String propertyValue = ra.getContent().toString();
				properties.setProperty(propertyName, propertyValue);
			}
		}
		return createFtpClient(properties);
	}

	/**
	 * Creates and configures a FtpClient instance based on the
	 * given properties.
	 * 
	 * @param properties the ftp client configuration properties
	 * @return remote input steam
	 * @throws Exception if an error occurs creating the ftp client
	 */
	public static Object createFtpClient(Properties properties) throws Exception {
		String ftpUrlString = properties.getProperty(FtpResource.URL);
		if (ftpUrlString == null)
			throw new Exception(NodeMessage.JEMC136E.toMessage().getFormattedMessage(FtpResource.URL));

		URL ftpUrl = new URL(ftpUrlString);
		if (!ftpUrl.getProtocol().equalsIgnoreCase("ftp") && !ftpUrl.getProtocol().equalsIgnoreCase("ftps"))
			throw new Exception(NodeMessage.JEMC137E.toMessage().getFormattedMessage(ftpUrl.getProtocol()));

		int port = ftpUrl.getPort();
		String server = ftpUrl.getHost();

		String username = properties.getProperty(FtpResource.USERID);
		if (username == null)
			throw new Exception(NodeMessage.JEMC136E.toMessage().getFormattedMessage(FtpResource.USERID));

		String password = properties.getProperty(FtpResource.PASSWORD);
		if (password == null)
			throw new Exception(NodeMessage.JEMC136E.toMessage().getFormattedMessage(FtpResource.PASSWORD));

		String remoteFile = properties.getProperty(FtpResource.REMOTE_FILE);
		if (remoteFile == null)
			throw new Exception(NodeMessage.JEMC136E.toMessage().getFormattedMessage(FtpResource.REMOTE_FILE));

		String accessMode = properties.getProperty(FtpResource.ACTION_MODE, FtpResource.ACTION_READ);

		
		FTPClient ftp = ftpUrl.getProtocol().equalsIgnoreCase("ftp") ? new FTPClient() : new FTPSClient();

//		int keepAliveTimeout = Parser.parseInt(properties.getProperty(FtpResource.PROP_KEEPALIVE_TIMEOUT, "-1"), -1);
//		if (keepAliveTimeout >= 0) {
//			ftp.setControlKeepAliveTimeout(keepAliveTimeout);
//		}
//		int controlKeepAliveReplyTimeout = Parser.parseInt(properties.getProperty(FtpResource.PROP_CONTROL_KEEPALIVE_REPLY_TIMEOUT, "-1"), -1);
//		if (controlKeepAliveReplyTimeout >= 0) {
//			ftp.setControlKeepAliveReplyTimeout(controlKeepAliveReplyTimeout);
//		}

		boolean binaryTransfer = Parser.parseBoolean(properties.getProperty(FtpResource.BINARY, "false"), false);

		// suppress login details
		//ftp.addProtocolCommandListener(new PrintCommandListener(new PrintWriter(System.out), true));

		try {
			int reply;
			if (port > 0) {
				ftp.connect(server, port);
			} else {
				ftp.connect(server);
			}
			// After connection attempt, you should check the reply code to
			// verify
			// success.
			reply = ftp.getReplyCode();

			if (!FTPReply.isPositiveCompletion(reply)) {
				ftp.disconnect();
				throw new IOException(NodeMessage.JEMC138E.toMessage().getFormattedMessage(reply));
			}
			
			if (!ftp.login(username, password)) {
				ftp.logout();
			}

			if (binaryTransfer) {
				ftp.setFileType(FTP.BINARY_FILE_TYPE);
			}

			if (accessMode.equalsIgnoreCase(FtpResource.ACTION_WRITE)){
				OutputStream os = ftp.storeFileStream(remoteFile);
				if (os == null){
					reply = ftp.getReplyCode();
					throw new IOException(NodeMessage.JEMC206E.toMessage().getFormattedMessage(remoteFile, reply));
				}
				return new FtpOutputStream(os, ftp);
			} else {
				InputStream is = ftp.retrieveFileStream(remoteFile);
				if (is == null){
					reply = ftp.getReplyCode();
					throw new IOException(NodeMessage.JEMC206E.toMessage().getFormattedMessage(remoteFile, reply));
				}
				return new FtpInputStream(is, ftp);
			}
		} catch (IOException e) {
			if (ftp.isConnected()) {
				try {
					ftp.disconnect();
				} catch (IOException f) {
					// do nothing
				}
			}
			throw e;
		}
	}
}