/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.security.keystore;

import java.security.Key;
import java.security.KeyStore;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.Date;

import org.pepstock.jem.node.NodeMessage;

/**
 * @author Andrea "Stock" Stocchero
 * @version 1.4
 */
public class KeysUtil {

	
	/**
	 * 
	 * @param keystoreInfo
	 * @return return key
	 * @throws Exception
	 */
	public static Key getSymmetricKey() throws Exception {
		return getSymmetricKey(KeyStoreUtil.getKeyStoresInfo().getClusterKeystoreInfo());
	}

	/**
	 * 
	 * @param keystoreInfo a bean containing keystore info
	 * 
	 * @return the symmetric key used in jem to make encrypt/decrypt operation
	 *         during the login phase of Jem Nodes (either web or not). The
	 *         method will try to load the symmetric key from key store
	 * @throws Exception if any exception occurs during the extraction of the
	 *             key
	 */
	public static Key getSymmetricKey(KeyStoreInfo keystoreInfo) throws Exception {
		KeyStore keystore = KeyStoreUtil.getKeystore(keystoreInfo);
		Key secretKey = keystore.getKey(keystoreInfo.getSymmetricKeyAlias(), keystoreInfo.getSymmetricKeyPwd().toCharArray());
		return secretKey;

	}

	/**
	 * The method will return the public key of the X.509 certificate inside the
	 * jem key store and relative to the alias passed as parameter. Before to
	 * extract the public key from the certificate, the method will check if the
	 * certificate is expired and if is an X.509 certificate.

	 * @param keystoreInfo 
	 * @param certAlias the alias of the X.509 certificate
	 * @return the public key relative to the certificate
	 * @throws Exception if any error occurs
	 */
	public static PublicKey getPublicKeyByAlias(KeyStoreInfo keystoreInfo, String certAlias) throws Exception {
		KeyStore keystore = KeyStoreUtil.getKeystore(keystoreInfo);
		Certificate certificate = keystore.getCertificate(certAlias);
		// check if is a X.509 certificate
		if (certificate == null || !certificate.getType().equals(CertificatesUtil.X509_CERTIFICATE_TYPE)) {
			throw new Exception(NodeMessage.JEMC200E.toMessage().getFormattedMessage(certAlias));
		}
		// verify expiration date
		Date expirationDate = ((X509Certificate) certificate).getNotAfter();
		if (expirationDate.before(new Date())) {
			throw new Exception(NodeMessage.JEMC201E.toMessage().getFormattedMessage(certAlias, expirationDate));
		}
		return certificate.getPublicKey();
	}

}
