/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.security;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.pepstock.catalog.gdg.Root;
import org.pepstock.jem.node.configuration.ConfigKeys;

/**
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class SecurityUtils {

	/**
	 * if the file name must be checked with permissions 
	 */
	public static final int TO_BE_CHECKED = 1;
	
	/**
	 * No security check
	 */
	public static final int TO_BE_IGNORED = 0;
	
	/**
	 * Not authorized! Exception!
	 */
	public static final int TO_BE_REJECTED = -1;
	
	private String dataPath = System.getProperty(ConfigKeys.JEM_DATA_PATH_NAME);

	private String outputPath = System.getProperty(ConfigKeys.JEM_OUTPUT_PATH_NAME);
	
	private String sourcePath = System.getProperty(ConfigKeys.JEM_SOURCE_PATH_NAME);
	
	private String libraryPath = System.getProperty(ConfigKeys.JEM_LIBRARY_PATH_NAME);
	
	private String binaryPath = System.getProperty(ConfigKeys.JEM_BINARY_PATH_NAME);
	
	private String classPath = System.getProperty(ConfigKeys.JEM_CLASSPATH_PATH_NAME);
	
	private String persistencePath = System.getProperty(ConfigKeys.JEM_PERSISTENCE_PATH_NAME);
	
	private String home = System.getProperty(ConfigKeys.JEM_HOME);
	
	private String temp = null;
	
	/**
	 * 
	 */
	public SecurityUtils(){
		try {
			temp = FilenameUtils.normalize(File.createTempFile("security", "tmp").getParentFile().getAbsolutePath(), true);
		} catch (IOException e) {
		}
	}
	
	/**
	 * Checks if the file name must be checked, ignored or rejected 
	 * 
	 * @param fileName to check!
	 * @return -1, 0, 1 if authorized or must be checked
	 */
	public int checkReadFileName(String fileName){
		if (fileName.startsWith(dataPath))
			return TO_BE_CHECKED;

		boolean go = fileName.startsWith(outputPath) ||
				fileName.startsWith(sourcePath) ||
				fileName.startsWith(libraryPath) ||
				fileName.startsWith(binaryPath) ||
				fileName.startsWith(classPath);
		if (go)
			return TO_BE_IGNORED;

		if (fileName.startsWith(persistencePath)){
			return TO_BE_REJECTED;
		}
		
		String ext = FilenameUtils.getExtension(fileName);
		if (ext.equalsIgnoreCase("class") || ext.equalsIgnoreCase("jar") || ext.equalsIgnoreCase("zip"))
			return TO_BE_IGNORED;

		if (fileName.startsWith(home)){
			if (FilenameUtils.wildcardMatch(fileName,home+File.separator+"*"+File.separator+"config*"))
				return TO_BE_REJECTED;
			
		}
		return TO_BE_IGNORED;
	}
	
	/**
	 * Checks if the file name must be checked, ignored or rejected 
	 * 
	 * @param fileName to check!
	 * @return -1, 0, 1 if authorized or must be checked
	 */
	public int checkWriteFileName(String fileName){
		
		if (fileName.endsWith(Root.ROOT_FILE_NAME))
			return TO_BE_REJECTED;
		
		boolean go = fileName.startsWith(dataPath) ||
				fileName.startsWith(sourcePath) ||
				fileName.startsWith(libraryPath) ||
				fileName.startsWith(binaryPath) ||
				fileName.startsWith(classPath);
		if (go)
			return TO_BE_CHECKED;
		
		if (fileName.startsWith(temp)){
				return TO_BE_IGNORED;
		}
		
		if (fileName.startsWith(outputPath)){
			/**
			 * Following code checks if the parent folder of filename is the same of output.
			 * if yes, it's not allowed. Output data set are written on folder [output]/[step]/[file]
			 * so the capabilities to write SYSOUT is guaranteed 
			 */
			if (FilenameUtils.getFullPathNoEndSeparator(fileName).equalsIgnoreCase(outputPath)){
				return TO_BE_REJECTED;
			}
			return TO_BE_IGNORED;
		}
		
		if (fileName.startsWith(home)){
			return TO_BE_REJECTED;
		}

		if (fileName.startsWith(persistencePath)){
			return TO_BE_REJECTED;
		}

		return TO_BE_REJECTED;
	}
	/**
	 * Extract the right file name, used inside the permissions
	 * @param fileName complete path
	 * @return relative file name
	 */
	public String normalizeFileName(String fileName){
		if (fileName.startsWith(dataPath)){
			String file = StringUtils.substringAfter(fileName, dataPath);
			if (FilenameUtils.separatorsToSystem(file).startsWith(File.separator)){
				return file.substring(1);
			} else {
				return file;
			}
		}
		return fileName;
	}

}