/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.resources.custom.engine.xml;

import org.pepstock.jem.util.Parser;

import com.thoughtworks.xstream.XStream;

/**
 * This class represent a generic custom resource field in the resource templates <code>xml</code> file. <br>
 * It contains all of the properties common to all types of fields.
 * @see XStream
 *  
 * @author Alessandro Zambrini
 */
public class AbstractFieldTemplate {
	
	/**
	 * Name of the key attribute. 
	 * @see XStream
	 */
	public final static String KEY_ATTRIBUTE = "key";
	
	/**
	 * Name of the label attribute. 
	 * @see XStream
	 */
	public final static String LABEL_ATTRIBUTE = "label";
	
	/**
	 * Name of the description attribute. 
	 * @see XStream
	 */
	public final static String DESCRIPTION_ATTRIBUTE = "description";
	
	/**
	 * Name of the mandatory attribute. 
	 * @see XStream
	 */
	public final static String MANDATORY_ATTRIBUTE = "mandatory";

	/**
	 * Name of the override attribute. 
	 * @see XStream
	 */
	public final static String OVERRIDE_ATTRIBUTE = "override";
	
	/**
	 * Name of the visible attribute. 
	 * @see XStream
	 */
	public final static String VISIBLE_ATTRIBUTE = "visible";
	
	/**
	 * The key of the resource field.
	 */
	private String key = null;
	
	/**
	 * The label of the resource field.
	 */
	private String label = null;
	
	/**
	 * The description of the resource field.
	 */
	private String description = null;
	
	/**
	 * The property that indicates whether the field is mandatory.
	 */
	private boolean mandatory = false;

	/**
	 * The property that indicates whether the field is visible.
	 */
	private String visible = "true";
	
	/**
	 * The property that indicates whether the field is overridable.
	 */
	private String override = "true";
	
	/**
	 * Returns the field key.
	 * @return the field key.
	 */
	public String getKey() {
		return key;
	}

	/**
	 * Sets the field key.
	 * @param key the key of the field.
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * Returns the field label.
	 * @return the label of this field
	 */
	public String getLabel() {
		return label;
	}

	/**
	 * Sets the field label.
	 * @param label the label of the field.
	 */
	public void setLabel(String label) {
		this.label = label;
	}

	/**
	 * Returns the field description.
	 * @return a description of this field.
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Sets the field description.
	 * @param description the description of the field.
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Returns the property that indicates whether the field is mandatory.
	 * @return <code>true</code> if this field needs to be filled/selected, <code>false</code> 
	 * if this field is optional and can not be filled/selected
	 */
	public boolean isMandatory() {
		return mandatory;
	}

	/**
	 * Sets if this field is mandatory or not.
	 * @param mandatory <code>true</code> if you want this field to be 
	 * mandatory, <code>false</code> otherwise.
	 */
	public void setMandatory(boolean mandatory) {
		this.mandatory = mandatory;
	}
	
	/**
	 * Returns the property that indicates whether the field is visible.
	 * @return <code>true</code> if this field is visible in the summary, 
	 * <code>false</code> otherwise.
	 */
	public boolean isVisible() {
		return Parser.parseBoolean(this.visible, true);
		//return visible;
	}

	/**
	 * Sets if this field is visible or not.
	 * @param visible <code>true</code> if you want this field to be 
	 * visible, <code>false</code> otherwise.
	 */
	public void setVisible(boolean visible) {
		this.override = Boolean.toString(visible);
	}
	
	/**
	 * Returns the property that indicates whether the field is override.
	 * @return <code>true</code> if this field is override, <code>false</code> 
	 * otherwise.
	 */
	public boolean isOverride() {
		return Parser.parseBoolean(this.override, true);
	}

	/**
	 * Sets if this field is override or not.
	 * @param override <code>true</code> if you want this field to be 
	 * override, <code>false</code> otherwise.
	 */
	public void setOverride(boolean override) {
		this.override = Boolean.toString(override);
	}

	public void override(String o){
		this.override = o;
	}
}
