/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.resources;

import java.security.Key;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;

import org.apache.shiro.authz.AuthorizationException;
import org.apache.shiro.codec.CodecSupport;
import org.apache.shiro.crypto.AesCipherService;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.security.Crypto;
import org.pepstock.jem.node.security.Role;
import org.pepstock.jem.node.security.RolesQueuePredicate;
import org.pepstock.jem.node.security.StringPermission;
import org.pepstock.jem.node.security.User;

import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.sun.org.apache.xml.internal.security.utils.Base64;

/**
 * Common resources utility for keys management, 
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class ResourcesUtil {
	
	private static ResourcesUtil INSTANCE = null;
	
	/**
	 * 
	 */
	public static final String ACTION_PARM = "action";
	
	private Key key = null;
	
	/**
	 * @param iNSTANCE
	 */
	private ResourcesUtil() {
		super();
	}
	
	/**
	 * @return instance of utility to use everywhere
	 */
	public static ResourcesUtil getInstance(){
		if (INSTANCE == null){
			INSTANCE = new ResourcesUtil();
		}
		return INSTANCE;
	}

	/**
	 * Creates a key and stores it.
	 */
	public void createKey(){
		AesCipherService aesService = new AesCipherService();
		setKey(aesService.generateNewKey());
	}
	
	/**
	 * @return the key
	 */
	public Key getKey() {
		return key;
	}

	/**
	 * @param key the key to set
	 */
	public void setKey(Key key) {
		this.key = key;
	}

	/**
	 * Encryts a secret 
	 * 
	 * @param value to encryt
	 * @return encrypted value
	 * @throws Exception 
	 */
	public CryptedValueAndHash encrypt(String value) throws Exception{
		byte[] whatBytes = CodecSupport.toBytes(value);
		byte[] cryptBytes = Crypto.crypt(whatBytes, key);
		
		CryptedValueAndHash result = new CryptedValueAndHash();
		result.setCryptedValue(Base64.encode(cryptBytes, 2 * cryptBytes.length));
		
		Sha256Hash hasher = new Sha256Hash(result.getCryptedValue());
		String valueHashed = hasher.toBase64();
		
		result.setHash(valueHashed);
		return result;
	}
	
	/**
	 * Decrypts a secret
	 * @param object oject with encrypted value and ha sh to check is correcct
	 * @return secret in clear
	 * @throws Exception if error occurs
	 */
	public String decrypt(CryptedValueAndHash object) throws Exception{
		return decrypt(object.getCryptedValue(), object.getHash());
	}
	
	/**
	 * Decrypts a secret
	 * 
	 * @param cryptedValue encrypted value
	 * @param hash hash string to check if secret is correct
	 * @return secret in clear
	 * @throws Exception if error occurs
	 */
	public String decrypt(String cryptedValue, String hash) throws Exception{
		Sha256Hash hasher = new Sha256Hash(cryptedValue);
		String valueHashed = hasher.toBase64();
		if (!valueHashed.equalsIgnoreCase(hash))
			throw new Exception(NodeMessage.JEMC118E.toMessage().getMessage());
		
		byte[] decryptBytes = Crypto.decrypt(Base64.decode(cryptedValue), key);
		String result = CodecSupport.toString(decryptBytes);
		return result;
	}
	
	/**
	 * Checks if the user has the permission.
	 * 
	 * @param userid user id must be checked
	 * @param permission permission to search
	 * @param instance Hazelcast instance to access to maps
	 * @throws AuthorizationException if any exception occurs
	 */
	public void checkPermissions(String userid, String permission, HazelcastInstance instance) throws AuthorizationException{
		User user = new User(userid);
		//
		// creates Hazelcast predicate to extract all roles and permissions
		// assigned to user
		RolesQueuePredicate predicate = new RolesQueuePredicate();
		predicate.setUser(user);

		// gets map and performs predicate!
		IMap<String, Role> roles = instance.getMap(Queues.ROLES_MAP);
		ArrayList<Role> myroles = null;
		if (roles.lockMap(10, TimeUnit.SECONDS)){ 
			try {
				myroles = new ArrayList<Role>(roles.values(predicate));
			} finally {
				roles.unlockMap();
			}
		} else {
			throw new AuthorizationException(NodeMessage.JEMC119E.toMessage().getFormattedMessage(Queues.ROLES_MAP));
		}

		StringPermission sPermission = new StringPermission(permission);
		for (Role role : myroles) {
			for (String perm: role.getPermissions()){
				StringPermission sPerm = new StringPermission(perm);
				if (sPerm.implies(sPermission))
					return;
			}
		}
		throw new AuthorizationException(NodeMessage.JEMC120E.toMessage().getFormattedMessage(userid, permission));
	}
	
}