/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.security.UserPreference;

import com.hazelcast.core.MapStore;

/**
 * Persistent manager for RoutingConfs map.<br>
 * It uses DBManager instance to perform all sqls.<br>
 * It throws RuntimeException if the database manager has errors but Hazelcast
 * is not able to catch them, so it logs all errors.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class UserPreferencesMapManager implements MapStore<String, HashMap<String, UserPreference>> {

	private UserPreferencesDBManager dbManager = null;
	
	private SQLContainer sql = null;

	/**
	 * Construct the object instantiating a new DBManager
	 */
	public UserPreferencesMapManager() {
		dbManager = UserPreferencesDBManager.getInstance();
		sql = dbManager.getPreferencesSqlContainer();
	}

	/**
	 * Loads userPreferences instance by userPreferences name passed by Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#load(java.lang.Object)
	 * @param userPreferencesName userPreferences id of userPreferences to load
	 */
	@Override
	public HashMap<String, UserPreference> load(String userId) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		HashMap<String, UserPreference> userPreferences = null;
		try {
			// load userPreferences instance from table
			userPreferences = dbManager.getUserPreferences(sql.getGetStatement(), userId);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return userPreferences;
	}

	/**
	 * Loads all keys (userPreferences names) at the starting of Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAllKeys()
	 */
	@Override
	public Set<String> loadAllKeys() {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		Set<String> set = null;
		try {
			// loadAll keys from table
			set = dbManager.getAllUserIds(sql.getGetAllKeysStatement());
			LogAppl.getInstance().emit(NodeMessage.JEMC045I, String.valueOf(set.size()), Queues.USER_PREFERENCES_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return set;
	}

	/**
	 * Deletes a userPreferences instance from queue by userPreferences name
	 * 
	 * @see com.hazelcast.core.MapStore#delete(java.lang.Object)
	 */
	@Override
	public void delete(String userId) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// deletes the userPreferences in table
			dbManager.delete(sql.getDeleteStatement(), userId);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * Stores a userPreferences instance in map. If already exists, it updates it.
	 * 
	 * @see com.hazelcast.core.MapStore#store(java.lang.Object,
	 *      java.lang.Object)
	 */
	@Override
	public void store(String userId, HashMap<String, UserPreference> userPreferences) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// inserts the userPreferences in table
			dbManager.insert(sql.getInsertStatement(), userId, userPreferences);
		} catch (SQLException e) {
			// I have an exception (it happens if the key already exists, so
			// update anyway
			try {
				// updates the userPreferences in table
				dbManager.update(sql.getUpdateStatement(), userId, userPreferences);
			} catch (SQLException e1) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
				throw new RuntimeException(e1);
			}
		}
	}

	/**
	 * used when a synchronous persistence is chosen
	 * 
	 * @see com.hazelcast.core.MapStore#storeAll(java.util.Map)
	 */
	@Override
	public void storeAll(Map<String, HashMap<String, UserPreference>> userPreferencess) {
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (Entry<String, HashMap<String, UserPreference>> userPreferences : userPreferencess.entrySet()){
			try {
				// inserts the userPreferences in table
				dbManager.insert(sql.getInsertStatement(), userPreferences.getKey(), userPreferences.getValue());
			} catch (SQLException e) {
				// I have an exception (it happens if the key already exists, so
				// update anyway
				try {
					// updates the userPreferences in table
					dbManager.update(sql.getUpdateStatement(), userPreferences.getKey(), userPreferences.getValue());
				} catch (SQLException e1) {
					LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
					throw new RuntimeException(e1);
				}
			}
		}
	}

	/**
	 * used when a synchronous persistence is chosen
	 * 
	 * @see com.hazelcast.core.MapStore#deleteAll(java.util.Collection)
	 */
	@Override
	public void deleteAll(Collection<String> ids) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (String id : ids){
			try {
				// deletes the userPreferences in table
				dbManager.delete(sql.getDeleteStatement(), id);
			} catch (SQLException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
				throw new RuntimeException(e);
			}
		}
	}

	/**
	 * Loads all userPreferencess saved, by a list of keys.
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAll(java.util.Collection)
	 * @param collaction of keys to load
	 * @return maps with all userPreferencess
	 */
	@Override
	public Map<String, HashMap<String, UserPreference>> loadAll(Collection<String> userIds) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		// use collections of keys in string format, to create SQL
		// for IN statement, put ' and , on right position
		StringBuilder sb = new StringBuilder();
		Iterator<String> iter = userIds.iterator();
		for (;;){
		    String userPreferencesName = iter.next();
		    sb.append("'").append(userPreferencesName).append("'");
		    if (!iter.hasNext()){
		    	break;
		    }
		    sb.append(", ");
		}
		// formats SQL to get all userPreferencess by keys 
		String sqlString = MessageFormat.format(sql.getGetAllStatement(), sb.toString());
		
		Map<String, HashMap<String, UserPreference>> userPreferencess = null;
		try {
			// load userPreferences instance from table
			userPreferencess = dbManager.getAllUserPreferences(sqlString);
			LogAppl.getInstance().emit(NodeMessage.JEMC055I, String.valueOf(userPreferencess.size()), Queues.USER_PREFERENCES_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return userPreferencess;
	}
}