/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.node.security.UserPreference;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the user preferences.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.4	
 *
 */
public class UserPreferencesDBManager {

	private final static UserPreferencesDBManager INSTANCE = new UserPreferencesDBManager();
	
	private XStream xs = new XStream();
	
	private SQLContainer preferencesSqlContainer = null;

	/**
	 * 
	 */
	private UserPreferencesDBManager(){
	}

	/**
	 * @return the preferencesSqlContainer
	 */
	public SQLContainer getPreferencesSqlContainer() {
		return preferencesSqlContainer;
	}

	/**
	 * @param preferencesSqlContainer the preferencesSqlContainer to set
	 */
	public void setPreferencesSqlContainer(SQLContainer preferencesSqlContainer) {
		this.preferencesSqlContainer = preferencesSqlContainer;
	}

	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of JobDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static UserPreferencesDBManager getInstance(){
		return INSTANCE;
	}

	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}
	/**
	 * Deletes userPreferences instance from queue by userPreferences name (the key of table)
	 * 
	 * @param delete SQL statement to delete
	 * @param userId userPreferences name to delete userPreferences instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, String userId) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set userPreferences name in prepared statement
			updateStmt.setString(1, userId);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new userPreferences in table, serializing userPreferences in XML
	 * 
	 * @param insert SQL statement
	 * @param userId userId
	 * @param userPreferences userPreferences instance
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, String userId, HashMap<String, UserPreference> userPreferences) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the userPreferences in XML, use a reader because necessary in clob
			StringReader reader = new StringReader(xs.toXML(userPreferences));

			PreparedStatement updateStmt = connection.prepareStatement(insert);
			// set userPreferences name to key
			updateStmt.setString(1, userId);
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the userPreferences instance by userPreferences name, serializing userPreferences in XML
	 * 
	 * @param update SQL statement
	 * @param userPreferences userPreferences instance to serialize
	 * @throws SQLException if occurs
	 */
	public void update(String update, String userId, HashMap<String, UserPreference> userPreferences) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the userPreferences in XML
			StringReader reader = new StringReader(xs.toXML(userPreferences));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			// set userPreferences name to key
			updateStmt.setString(2, userId);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all userPreferences names (keys) in a Set object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<String> getAllUserIds(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashSet<String> allIds = new HashSet<String>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(rs.getObject(1).toString());
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all userPreferencess in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<String, HashMap<String, UserPreference>> getAllUserPreferences(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashMap<String, HashMap<String, UserPreference>> allUserPreferences = new HashMap<String, HashMap<String, UserPreference>>();
			while (rs.next()) {
				// get CLOB field which contains userPreferences XML serialization
				// deserializes Role instance
				HashMap<String, UserPreference> userPreferences = (HashMap<String, UserPreference>) xs.fromXML(rs.getCharacterStream(1));

				allUserPreferences.put(rs.getObject(2).toString(), userPreferences);
			}
			rs.close();
			stmt.close();
			return allUserPreferences;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a userPreferences instance store on table, by userPreferences name.
	 * 
	 * @param query SQL statement
	 * @param userId userPreferences name (the key)
	 * @return userPreferences instance
	 * @throws SQLException if occurs
	 */
	public HashMap<String, UserPreference> getUserPreferences(String query, String userId) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets userPreferences names where condition
			stmt.setString(1, userId);

			ResultSet rs = stmt.executeQuery();
			HashMap<String, UserPreference> userPreferences = null;

			// checks if I have the result. ONLY 1 row if expected. If more, is
			// an error because the userPreferences name
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains userPreferences XML serialization
				// deserializes userPreferences instance
				userPreferences = (HashMap<String, UserPreference>) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return userPreferences;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of userPreferencess in byte present in the ROLES MAP
	 * @throws SQLException if an sql exception occurs
	 */
	public long getUserPreferencesSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(preferencesSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}