/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.configuration.SwarmConfiguration;

import com.hazelcast.core.MapStore;

/**
 * Persistent manager for RoutingConfs map.<br>
 * It uses DBManager instance to perform all sqls.<br>
 * It throws RuntimeException if the database manager has errors but Hazelcast
 * is not able to catch them, so it logs all errors.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class RoutingConfigMapManager implements MapStore<String, SwarmConfiguration> {

	private RoutingConfigDBManager dbManager = null;
	
	private SQLContainer sql = null;

	/**
	 * Construct the object instantiating a new DBManager
	 */
	public RoutingConfigMapManager() {
		dbManager = RoutingConfigDBManager.getInstance();
		sql = dbManager.getRoutingSqlContainer();
	}

	/**
	 * Loads routingConf instance by routingConf name passed by Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#load(java.lang.Object)
	 * @param routingConfName routingConf id of routingConf to load
	 */
	@Override
	public SwarmConfiguration load(String routingConfName) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		SwarmConfiguration routingConf = null;
		try {
			// load routingConf instance from table
			routingConf = dbManager.getSwarmConfiguration(sql.getGetStatement(), routingConfName);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return routingConf;
	}

	/**
	 * Loads all keys (routingConf names) at the starting of Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAllKeys()
	 */
	@Override
	public Set<String> loadAllKeys() {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		Set<String> set = null;
		try {
			// loadAll keys from table
			set = dbManager.getAllSwarmConfigurationNames(sql.getGetAllKeysStatement());
			LogAppl.getInstance().emit(NodeMessage.JEMC045I, String.valueOf(set.size()), Queues.ROUTING_CONFIG_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return set;
	}

	/**
	 * Deletes a routingConf instance from queue by routingConf name
	 * 
	 * @see com.hazelcast.core.MapStore#delete(java.lang.Object)
	 */
	@Override
	public void delete(String routingConfName) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// deletes the routingConf in table
			dbManager.delete(sql.getDeleteStatement(), routingConfName);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * Stores a routingConf instance in map. If already exists, it updates it.
	 * 
	 * @see com.hazelcast.core.MapStore#store(java.lang.Object,
	 *      java.lang.Object)
	 */
	@Override
	public void store(String routingConfName, SwarmConfiguration routingConf) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// inserts the routingConf in table
			dbManager.insert(sql.getInsertStatement(), routingConf);
		} catch (SQLException e) {
			// I have an exception (it happens if the key already exists, so
			// update anyway
			try {
				// updates the routingConf in table
				dbManager.update(sql.getUpdateStatement(), routingConf);
			} catch (SQLException e1) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
				throw new RuntimeException(e1);
			}
		}
	}

	/**
	 * used when a synchronous persistence is chosen
	 * 
	 * @see com.hazelcast.core.MapStore#storeAll(java.util.Map)
	 */
	@Override
	public void storeAll(Map<String, SwarmConfiguration> routingConfs) {
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (SwarmConfiguration routingConf : routingConfs.values()){
			try {
				// inserts the routingConf in table
				dbManager.insert(sql.getInsertStatement(), routingConf);
			} catch (SQLException e) {
				// I have an exception (it happens if the key already exists, so
				// update anyway
				try {
					// updates the routingConf in table
					dbManager.update(sql.getUpdateStatement(), routingConf);
				} catch (SQLException e1) {
					LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
					throw new RuntimeException(e1);
				}
			}
		}
	}

	/**
	 * used when a synchronous persistence is chosen
	 * 
	 * @see com.hazelcast.core.MapStore#deleteAll(java.util.Collection)
	 */
	@Override
	public void deleteAll(Collection<String> ids) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (String id : ids){
			try {
				// deletes the routingConf in table
				dbManager.delete(sql.getDeleteStatement(), id);
			} catch (SQLException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
				throw new RuntimeException(e);
			}
		}
	}

	/**
	 * Loads all routingConfs saved, by a list of keys.
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAll(java.util.Collection)
	 * @param collaction of keys to load
	 * @return maps with all routingConfs
	 */
	@Override
	public Map<String, SwarmConfiguration> loadAll(Collection<String> routingConfNames) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		// use collections of keys in string format, to create SQL
		// for IN statement, put ' and , on right position
		StringBuilder sb = new StringBuilder();
		Iterator<String> iter = routingConfNames.iterator();
		for (;;){
		    String routingConfName = iter.next();
		    sb.append("'").append(routingConfName).append("'");
		    if (!iter.hasNext()){
		    	break;
		    }
		    sb.append(", ");
		}
		// formats SQL to get all routingConfs by keys 
		String sqlString = MessageFormat.format(sql.getGetAllStatement(), sb.toString());
		
		Map<String, SwarmConfiguration> routingConfs = null;
		try {
			// load routingConf instance from table
			routingConfs = dbManager.getAllSwarmConfigurations(sqlString);
			LogAppl.getInstance().emit(NodeMessage.JEMC055I, String.valueOf(routingConfs.size()), Queues.ROUTING_CONFIG_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return routingConfs;
	}
}