/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.node.configuration.SwarmConfiguration;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the routingConfs.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class RoutingConfigDBManager {

	private final static RoutingConfigDBManager INSTANCE = new RoutingConfigDBManager();
	
	private XStream xs = new XStream();
	
	private SQLContainer routingSqlContainer = null;

	/**
	 * 
	 */
	private RoutingConfigDBManager(){
	}

	/**
	 * @return the routingSqlContainer
	 */
	public SQLContainer getRoutingSqlContainer() {
		return routingSqlContainer;
	}


	/**
	 * @param routingSqlContainer the routingSqlContainer to set
	 */
	public void setRoutingSqlContainer(SQLContainer routingSqlContainer) {
		this.routingSqlContainer = routingSqlContainer;
	}


	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of JobDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static RoutingConfigDBManager getInstance(){
		return INSTANCE;
	}

	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}
	/**
	 * Deletes routingConf instance from queue by routingConf name (the key of table)
	 * 
	 * @param delete SQL statement to delete
	 * @param routingConfName routingConf name to delete routingConf instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, String routingConfName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set routingConf name in prepared statement
			updateStmt.setString(1, routingConfName);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new routingConf in table, serializing routingConf in XML
	 * 
	 * @param insert SQL statement
	 * @param routingConf routingConf instance
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, SwarmConfiguration routingConf) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the routingConf in XML, use a reader because necessary in clob
			StringReader reader = new StringReader(xs.toXML(routingConf));

			PreparedStatement updateStmt = connection.prepareStatement(insert);
			// set routingConf name to key
			updateStmt.setString(1, routingConf.getName());
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the routingConf instance by routingConf name, serializing routingConf in XML
	 * 
	 * @param update SQL statement
	 * @param routingConf routingConf instance to serialize
	 * @throws SQLException if occurs
	 */
	public void update(String update, SwarmConfiguration routingConf) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the routingConf in XML
			StringReader reader = new StringReader(xs.toXML(routingConf));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			// set routingConf name to key
			updateStmt.setString(2, routingConf.getName());
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all routingConf names (keys) in a Set object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<String> getAllSwarmConfigurationNames(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashSet<String> allIds = new HashSet<String>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(rs.getObject(1).toString());
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all routingConfs in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<String, SwarmConfiguration> getAllSwarmConfigurations(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashMap<String, SwarmConfiguration> allSwarmConfigurations = new HashMap<String, SwarmConfiguration>();
			while (rs.next()) {
				// get CLOB field which contains routingConf XML serialization
				// deserializes Role instance
				SwarmConfiguration routingConf = (SwarmConfiguration) xs.fromXML(rs.getCharacterStream(1));

				allSwarmConfigurations.put(routingConf.getName(), routingConf);
			}
			rs.close();
			stmt.close();
			return allSwarmConfigurations;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a routingConf instance store on table, by routingConf name.
	 * 
	 * @param query SQL statement
	 * @param routingConfName routingConf name (the key)
	 * @return routingConf instance
	 * @throws SQLException if occurs
	 */
	public SwarmConfiguration getSwarmConfiguration(String query, String routingConfName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets routingConf names where condition
			stmt.setString(1, routingConfName);

			ResultSet rs = stmt.executeQuery();
			SwarmConfiguration routingConf = null;

			// checks if I have the result. ONLY 1 row if expected. If more, is
			// an error because the routingConf name
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains routingConf XML serialization
				// deserializes routingConf instance
				routingConf = (SwarmConfiguration) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return routingConf;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of routingConfs in byte present in the ROLES MAP
	 * @throws SQLException if an sql exception occurs
	 */
	public long getRoutingConfigSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(routingSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}