/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.node.security.Role;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the roles.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class RolesDBManager {

	private final static RolesDBManager INSTANCE = new RolesDBManager();
	
	private XStream xs = new XStream();
	
	private SQLContainer rolesSqlContainer = null;

	/**
	 * 
	 */
	private RolesDBManager(){
	}
	
	/**
	 * @return the rolesSqlContainer
	 */
	public SQLContainer getRolesSqlContainer() {
		return rolesSqlContainer;
	}
	/**
	 * @param rolesSqlContainer the rolesSqlContainer to set
	 */
	public void setRolesSqlContainer(SQLContainer rolesSqlContainer) {
		this.rolesSqlContainer = rolesSqlContainer;
	}
	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of JobDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static RolesDBManager getInstance(){
		return INSTANCE;
	}

	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}
	/**
	 * Deletes role instance from queue by role name (the key of table)
	 * 
	 * @param delete SQL statement to delete
	 * @param roleName role name to delete role instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, String roleName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set role name in prepared statement
			updateStmt.setString(1, roleName);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new role in table, serializing role in XML
	 * 
	 * @param insert SQL statement
	 * @param role role instance
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, Role role) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the role in XML, use a reader because necessary in clob
			StringReader reader = new StringReader(xs.toXML(role));

			PreparedStatement updateStmt = connection.prepareStatement(insert);
			// set role name to key
			updateStmt.setString(1, role.getName());
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the role instance by role name, serializing role in XML
	 * 
	 * @param update SQL statement
	 * @param role role instance to serialize
	 * @throws SQLException if occurs
	 */
	public void update(String update, Role role) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the role in XML
			StringReader reader = new StringReader(xs.toXML(role));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			// set role name to key
			updateStmt.setString(2, role.getName());
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all role names (keys) in a Set object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<String> getAllRolesNames(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashSet<String> allIds = new HashSet<String>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(rs.getObject(1).toString());
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all roles in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<String, Role> getAllRoles(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashMap<String, Role> allRoles = new HashMap<String, Role>();
			while (rs.next()) {
				// get CLOB field which contains role XML serialization
				// deserializes Role instance
				Role role = (Role) xs.fromXML(rs.getCharacterStream(1));

				allRoles.put(role.getName(), role);
			}
			rs.close();
			stmt.close();
			return allRoles;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a role instance store on table, by role name.
	 * 
	 * @param query SQL statement
	 * @param roleName role name (the key)
	 * @return role instance
	 * @throws SQLException if occurs
	 */
	public Role getRole(String query, String roleName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets role names where condition
			stmt.setString(1, roleName);

			ResultSet rs = stmt.executeQuery();
			Role role = null;

			// checks if I have the result. ONLY 1 row if expected. If more, is
			// an error because the role name
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains role XML serialization
//				Clob clob = (Clob) rs.getObject(1);
				// deserializes role instance
				role = (Role) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return role;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of roles in byte present in the ROLES MAP
	 * @throws SQLException if an sql exception occurs
	 */
	public long getRolesSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(rolesSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}