/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Simone "Busy" Businaro
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.PreJob;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the PreJob in CHECKING queues.<br>
 * 
 * @author Simone "Busy" Businaro
 * 
 */
public class PreJobDBManager {

	private static final PreJobDBManager INSTANCE = new PreJobDBManager();
	
	private XStream xs = new XStream();
	
	private SQLContainer checkingSqlContainer = null;

	/**
	 * Calls super class to create the connection
	 * 
	 * @throws Exception occurs if an error
	 */
	private PreJobDBManager(){
	}
	
	/**
	 * @return the checkingSqlContainer
	 */
	public SQLContainer getCheckingSqlContainer() {
		return checkingSqlContainer;
	}

	/**
	 * @param checkingSqlContainer the checkingSqlContainer to set
	 */
	public void setCheckingSqlContainer(SQLContainer checkingSqlContainer) {
		this.checkingSqlContainer = checkingSqlContainer;
	}

	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of JobDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static PreJobDBManager getInstance() {
		return INSTANCE;
	}

	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}
	
	/**
	 * Deletes PreJob instance from queue by prejobid that is a Long
	 * 
	 * @param delete SQL statement to delete
	 * @param prejobid (key) to delete job instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, Long prejobid) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set jobid in prepared statement
			updateStmt.setLong(1, prejobid);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new PreJob in table, serializing job in XML
	 * 
	 * @param insert SQL statement
	 * @param prejob PreJob instance
	 * @param id the id of the prejob in the map
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, PreJob prejob, Long id) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the job in XML, use a reader because necessary in clob
			StringReader reader = new StringReader(xs.toXML(prejob));
			PreparedStatement updateStmt = connection.prepareStatement(insert);
			updateStmt.setLong(1, id);
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the pre job instance by pre job id, serializing pre job in XML
	 * 
	 * @param update SQL statement
	 * @param prejob is the prejob instance
	 * @param id the id of the prejob in the map
	 * @throws SQLException if occurs
	 */
	public void update(String update, PreJob prejob, Long id) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the job in XML
			StringReader reader = new StringReader(xs.toXML(prejob));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			updateStmt.setLong(2, id);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all pre job ids (keys) in a Set object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<Long> getAllPreJobIds(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);
			// creates the set
			HashSet<Long> allIds = new HashSet<Long>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(Long.parseLong(rs.getObject(1).toString()));
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all pre jobs in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<Long, PreJob> getAllPreJobs(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);
			// creates the set
			HashMap<Long, PreJob> allPreJobs = new HashMap<Long, PreJob>();
			while (rs.next()) {
				// get CLOB field which contains JOB XML serialization
				Long prejobid = (Long) rs.getObject(1);
				// deserializes JOB instance
				PreJob job = (PreJob) xs.fromXML(rs.getCharacterStream(2));

				allPreJobs.put(prejobid, job);
			}
			rs.close();
			stmt.close();
			return allPreJobs;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a pre job instance store on table, by pre job id.
	 * 
	 * @param query SQL statement
	 * @param prejobid pre job id (the key)
	 * @return job instance
	 * @throws SQLException if occurs
	 */
	public PreJob getPreJob(String query, Long prejobid) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets job id where condition
			stmt.setLong(1, prejobid);
			ResultSet rs = stmt.executeQuery();
			PreJob prejob = null;
			// checks if I have the result. ONLY 1 row if expected. If more, is
			// an error because the pre job id
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains JOB XML serialization
				prejob = (PreJob) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return prejob;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of pre jobs in byte present in the CHECKING QUEUE
	 * @throws SQLException if an sql exception occurs
	 */
	public long getPreJobsCheckingQueueJobSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(checkingSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}