/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.Job;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the job in INPUT,
 * OUTPUT and ROUTING queues.<br>
 * Actions are the same for all queues (and then table because there is a table
 * for queue)<br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class JobDBManager {

	private final static JobDBManager INSTANCE = new JobDBManager();
	
	private XStream xs = new XStream();
	
	private SQLContainer inputSqlContainer = null;

	private SQLContainer outputSqlContainer = null;
	
	private SQLContainer routingSqlContainer = null;
	
	private SQLContainer runningSqlContainer = null;
	
	/**
	 * 
	 * 
	 * @throws Exception occurs if an error
	 */
	private JobDBManager(){
	}

	/**
	 * @return the inputSqlContainer
	 */
	public SQLContainer getInputSqlContainer() {
		return inputSqlContainer;
	}


	/**
	 * @param inputSqlContainer the inputSqlContainer to set
	 */
	public void setInputSqlContainer(SQLContainer inputSqlContainer) {
		this.inputSqlContainer = inputSqlContainer;
	}


	/**
	 * @return the outputSqlContainer
	 */
	public SQLContainer getOutputSqlContainer() {
		return outputSqlContainer;
	}


	/**
	 * @param outputSqlContainer the outputSqlContainer to set
	 */
	public void setOutputSqlContainer(SQLContainer outputSqlContainer) {
		this.outputSqlContainer = outputSqlContainer;
	}


	/**
	 * @return the routingSqlContainer
	 */
	public SQLContainer getRoutingSqlContainer() {
		return routingSqlContainer;
	}


	/**
	 * @param routingSqlContainer the routingSqlContainer to set
	 */
	public void setRoutingSqlContainer(SQLContainer routingSqlContainer) {
		this.routingSqlContainer = routingSqlContainer;
	}

	

	/**
	 * @return the runningSqlContainer
	 */
	public SQLContainer getRunningSqlContainer() {
		return runningSqlContainer;
	}

	/**
	 * @param runningSqlContainer the runningSqlContainer to set
	 */
	public void setRunningSqlContainer(SQLContainer runningSqlContainer) {
		this.runningSqlContainer = runningSqlContainer;
	}

	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of JobDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static JobDBManager getInstance(){
		return INSTANCE;
	}
	
	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}

	/**
	 * Deletes job instance from queue by jobid (the key of table)
	 * 
	 * @param delete SQL statement to delete
	 * @param jobid jobid (key) to delete job instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, String jobid) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set jobid in prepared statement
			updateStmt.setString(1, jobid);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new job in table, serializing job in XML
	 * 
	 * @param insert SQL statement
	 * @param job job instance
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, Job job) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the job in XML, use a reader because necessary in clob
			StringReader reader = new StringReader(xs.toXML(job));

			PreparedStatement updateStmt = connection.prepareStatement(insert);
			// set jobid to key
			updateStmt.setString(1, job.getId());
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the job instance by jobid, serializing job in XML
	 * 
	 * @param update SQL statement
	 * @param job job instance to s
	 * @throws SQLException if occurs
	 */
	public void update(String update, Job job) throws SQLException {
		// open connection
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the job in XML
			StringReader reader = new StringReader(xs.toXML(job));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			// set jobid to key
			updateStmt.setString(2, job.getId());
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all jobids (keys) in a Set object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<String> getAllJobIds(String query) throws SQLException {
		// open connection
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashSet<String> allIds = new HashSet<String>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(rs.getObject(1).toString());
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all jobs in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<String, Job> getAllJobs(String query) throws SQLException {
		// open connection
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashMap<String, Job> allJobs = new HashMap<String, Job>();
			while (rs.next()) {
				// get CLOB field which contains JOB XML serialization
				Job job = (Job) xs.fromXML(rs.getCharacterStream(1));

				allJobs.put(job.getId(), job);
			}
			rs.close();
			stmt.close();
			return allJobs;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a job instance store on table, by job id.
	 * 
	 * @param query SQL statement
	 * @param jobid job id (the key)
	 * @return job instance
	 * @throws SQLException if occurs
	 */
	public Job getJob(String query, String jobid) throws SQLException {
		// opens connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets job id where condition
			stmt.setString(1, jobid);

			ResultSet rs = stmt.executeQuery();
			Job job = null;

			// checks if I have the result. ONLY 1 row if expected. If more, is
			// a
			// error because the job id
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains JOB XML serialization
				//Clob clob = (Clob) rs.getObject(1);
				// deserializes JOB instance
				job = (Job) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return job;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of jobs in byte present in the INPUT QUEUE
	 * @throws SQLException if an sql exception occurs
	 */
	public long getInputQueueJobSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(inputSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}


	/**
	 * @return the size of jobs in byte present in the RUNNING QUEUE
	 * @throws SQLException if an sql exception occurs
	 */
	public long getRunningQueueJobSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(runningSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
	
	/**
	 * @return the size of jobs in byte present in the OUTPUT QUEUE
	 * @throws SQLException if an sql exception occurs
	 */
	public long getOutputQueueJobSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(outputSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of jobs in byte present in the OUTPUT QUEUE
	 * @throws SQLException if an sql exception occurs
	 */
	public long getRoutingQueueJobSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(routingSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}