/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.resources.Resource;

import com.hazelcast.core.MapStore;

/**
 * Persistent manager for CommonResources map.<br>
 * It uses DBManager instance to perform all sqls.<br>
 * It throws RuntimeException if the database manager has errors but Hazelcast
 * is not able to catch them, so it logs all errors.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class CommonResourcesMapManager implements MapStore<String, Resource> {

	private CommonResourcesDBManager dbManager = null;
	
	private SQLContainer sql = null;

	/**
	 * Construct the object instantiating a new DBManager
	 */
	public CommonResourcesMapManager() {
		dbManager = CommonResourcesDBManager.getInstance();
		sql = dbManager.getCommonResourcesSqlContainer();
	}

	/**
	 * Loads resource instance by resource name passed by Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#load(java.lang.Object)
	 * @param resourceName resource id of resource to load
	 */
	@Override
	public Resource load(String resourceName) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		Resource resource = null;
		try {
			// load resource instance from table
			resource = dbManager.getResource(sql.getGetStatement(), resourceName);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return resource;
	}

	/**
	 * Loads all keys (resource names) at the starting of Hazelcast
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAllKeys()
	 */
	@Override
	public Set<String> loadAllKeys() {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		Set<String> set = null;
		try {
			// loadAll keys from table
			set = dbManager.getAllResourceNames(sql.getGetAllKeysStatement());
			LogAppl.getInstance().emit(NodeMessage.JEMC045I, String.valueOf(set.size()), Queues.COMMON_RESOURCES_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return set;
	}

	/**
	 * Deletes a resource instance from queue by resource name
	 * 
	 * @see com.hazelcast.core.MapStore#delete(java.lang.Object)
	 */
	@Override
	public void delete(String resourceName) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// deletes the resource in table
			dbManager.delete(sql.getDeleteStatement(), resourceName);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
	}

	/**
	 * Stores a resource instance in map. If already exists, it updates it.
	 * 
	 * @see com.hazelcast.core.MapStore#store(java.lang.Object,
	 *      java.lang.Object)
	 */
	@Override
	public void store(String resourceName, Resource resource) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		try {
			// inserts the resource in table
			dbManager.insert(sql.getInsertStatement(), resource);
		} catch (SQLException e) {
			// I have an exception (it happens if the key already exists, so
			// update anyway
			try {
				// updates the resource in table
				dbManager.update(sql.getUpdateStatement(), resource);
			} catch (SQLException e1) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
				throw new RuntimeException(e1);
			}
		}
	}

	/**
	 * used if Asynchronous persistence policy is used
	 * 
	 * @see com.hazelcast.core.MapStore#storeAll(java.util.Map)
	 */
	@Override
	public void storeAll(Map<String, Resource> map) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (Resource resource : map.values()){
			try {
				// inserts the resource in table
				dbManager.insert(sql.getInsertStatement(), resource);
			} catch (SQLException e) {
				// I have an exception (it happens if the key already exists, so
				// update anyway
				try {
					// updates the resource in table
					dbManager.update(sql.getUpdateStatement(), resource);
				} catch (SQLException e1) {
					LogAppl.getInstance().emit(NodeMessage.JEMC043E, e1);
					throw new RuntimeException(e1);
				}
			}
		}
	}

	/**
	 * used if Asynchronous persistence policy is used
	 * 
	 * @see com.hazelcast.core.MapStore#deleteAll(java.util.Collection)
	 */
	@Override
	public void deleteAll(Collection<String> names) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		for (String name : names){
			try {
				// deletes the resource in table
				dbManager.delete(sql.getDeleteStatement(), name);
			} catch (SQLException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
				throw new RuntimeException(e);
			}
		}
	}

	/**
	 * Loads all resources saved, by a list of keys.
	 * 
	 * @see com.hazelcast.core.MapLoader#loadAll(java.util.Collection)
	 * @param collaction of keys to load
	 * @return maps with all resources
	 */
	@Override
	public Map<String, Resource> loadAll(Collection<String> resourceNames) {
		// check if I have the database manager, otherwise log error and
		// exception
		if (dbManager == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC044E);
			throw new RuntimeException(NodeMessage.JEMC044E.toMessage().getMessage());
		}
		// use collections of keys in string format, to create SQL
		// for IN statement, put ' and , on right position
		StringBuilder sb = new StringBuilder();
		Iterator<String> iter = resourceNames.iterator();
		for (;;){
		    String resourceName = iter.next();
		    sb.append("'").append(resourceName).append("'");
		    if (!iter.hasNext()){
		    	break;
		    }
		    sb.append(", ");
		}
		// formats SQL to get all resources by keys 
		String sqlString = MessageFormat.format(sql.getGetAllStatement(), sb.toString());
		
		Map<String, Resource> resources = null;
		try {
			// load resource instance from table
			resources = dbManager.getAllResources(sqlString);
			LogAppl.getInstance().emit(NodeMessage.JEMC055I, String.valueOf(resources.size()), Queues.COMMON_RESOURCES_MAP);
		} catch (SQLException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC043E, e);
			throw new RuntimeException(e);
		}
		return resources;
	}
}