/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.persistence;

import java.io.StringReader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.XmlUtil;

import com.thoughtworks.xstream.XStream;

/**
 * Manages all SQL statements towards the database to persist the commons
 * resources.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class CommonResourcesDBManager{

	private final static CommonResourcesDBManager INSTANCE = new CommonResourcesDBManager();
	
	private XStream xs = null;
	
	private SQLContainer commonResourcesSqlContainer = null;

	/**
	 * Calls super class to create the connection
	 * 
	 * @throws Exception occurs if an error
	 */
	private CommonResourcesDBManager() {
		xs = XmlUtil.getXStream();
	}
	
	/**
	 * @return the commonResourcesSqlContainer
	 */
	public SQLContainer getCommonResourcesSqlContainer() {
		return commonResourcesSqlContainer;
	}

	/**
	 * @param commonResourcesSqlContainer the commonResourcesSqlContainer to set
	 */
	public void setCommonResourcesSqlContainer(SQLContainer commonResourcesSqlContainer) {
		this.commonResourcesSqlContainer = commonResourcesSqlContainer;
	}

	/**
	 * Is a static method (typical of a singleton) that returns the unique
	 * instance of CommonResourcesDBManager.<br>
	 * You must ONLY one instance of this per JVM instance.<br>
	 * 
	 * @return manager instance
	 * @throws Exception
	 */
	public synchronized static CommonResourcesDBManager getInstance(){
		return INSTANCE;
	}

	/**
	 * @return <code>true</code> is is instanciated, otherwise <code>false</code>.
	 */
	public static boolean isInstanciated(){
		return INSTANCE != null;
	}
	/**
	 * Deletes resource instance from queue by resource name (the key of table)
	 * 
	 * @param delete SQL statement to delete
	 * @param resourceName resource name to delete resource instance
	 * @throws SQLException if occurs
	 */
	public void delete(String delete, String resourceName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement updateStmt = connection.prepareStatement(delete);
			// set resource name in prepared statement
			updateStmt.setString(1, resourceName);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Inserts a new resource in table, serializing resource in XML
	 * 
	 * @param insert SQL statement
	 * @param resource resource instance
	 * @throws SQLException if occurs
	 */
	public void insert(String insert, Resource resource) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the resource in XML, use a reader because necessary in
			// clob
			StringReader reader = new StringReader(xs.toXML(resource));

			PreparedStatement updateStmt = connection.prepareStatement(insert);
			// set resource name to key
			updateStmt.setString(1, resource.getName());
			// set XML to clob
			updateStmt.setCharacterStream(2, reader);
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Updates the resource instance by resource name, serializing resource in
	 * XML
	 * 
	 * @param update SQL statement
	 * @param resource resource instance to serialize
	 * @throws SQLException if occurs
	 */
	public void update(String update, Resource resource) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			// serialize the resource in XML
			StringReader reader = new StringReader(xs.toXML(resource));

			PreparedStatement updateStmt = connection.prepareStatement(update);
			// set XML to clob
			updateStmt.setCharacterStream(1, reader);
			// set resource name to key
			updateStmt.setString(2, resource.getName());
			updateStmt.executeUpdate();
			updateStmt.close();
			connection.commit();
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all resource names (keys) in a Set object (asked by Hazelcast
	 * framework).
	 * 
	 * @param query SQL statement
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public Set<String> getAllResourceNames(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashSet<String> allIds = new HashSet<String>();
			while (rs.next()) {
				// loads all keys inset
				allIds.add(rs.getObject(1).toString());
			}
			rs.close();
			stmt.close();
			return allIds;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns all resources in a HasMap object (asked by Hazelcast framework).
	 * 
	 * @param query SQL statement (well formatted previously by caller)
	 * @return set with all keys in the table
	 * @throws SQLException if occurs
	 */
	public HashMap<String, Resource> getAllResources(String query) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(query);

			// creates the set
			HashMap<String, Resource> allResources = new HashMap<String, Resource>();
			while (rs.next()) {
				// get CLOB field which contains resource XML serialization
				//Clob clob = (Clob) rs.getObject(1);
				// deserializes Resource instance
				Resource resource = (Resource) xs.fromXML(rs.getCharacterStream(1));
				allResources.put(resource.getName(), resource);
			}
			rs.close();
			stmt.close();
			return allResources;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * Returns a resource instance store on table, by resource name.
	 * 
	 * @param query SQL statement
	 * @param resourceName resource name (the key)
	 * @return resource instance
	 * @throws SQLException if occurs
	 */
	public Resource getResource(String query, String resourceName) throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			PreparedStatement stmt = connection.prepareStatement(query);
			// sets resource names where condition
			stmt.setString(1, resourceName);

			ResultSet rs = stmt.executeQuery();
			Resource resource = null;

			// checks if I have the result. ONLY 1 row if expected. If more, is
			// an error because the resource name
			// is a primary key of table
			if (rs.next()) {
				// get CLOB field which contains RESOURCE XML serialization
				//Clob clob = (Clob) rs.getObject(1);
				// deserializes RESOURCE instance
				resource = (Resource) xs.fromXML(rs.getCharacterStream(1));
			}
			rs.close();
			stmt.close();
			return resource;
		} finally{
			if (connection != null)
				connection.close();
		}
	}

	/**
	 * @return the size of resources in byte present in the COMMON RESOURCES MAP
	 * @throws SQLException if an sql exception occurs
	 */
	public long getCommonResourcesSize() throws SQLException {
		// open connection
		Connection connection = DBPoolManager.getInstance().getConnection();
		try {
			Statement stmt = connection.createStatement();
			ResultSet rs = stmt.executeQuery(commonResourcesSqlContainer.getCheckQueueSizeStatement());
			rs.next();
			long size = rs.getLong(1);
			rs.close();
			stmt.close();
			return size;
		} finally{
			if (connection != null)
				connection.close();
		}
	}
}