/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.listeners;

import java.util.Date;

import org.pepstock.jem.Job;
import org.pepstock.jem.Result;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.Main;
import org.pepstock.jem.node.NodeInfo;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.RequestLock;
import org.pepstock.jem.node.Status;
import org.pepstock.jem.node.events.JobLifecycleEvent;

import com.hazelcast.core.Cluster;
import com.hazelcast.core.IMap;
import com.hazelcast.core.ITopic;
import com.hazelcast.core.Member;
import com.hazelcast.core.MembershipEvent;
import com.hazelcast.core.MembershipListener;

/**
 * Implements the listener interface of Hazelcast cluster to listen when a
 * member is removed from cluster.<br>
 * It's necessary to check who is the coordinator and unlock all locks
 * previously asked.
 * 
 * @author Andrea "Stock" Stocchero
 * @see org.pepstock.jem.node.NodeInfo
 * 
 */
public class NodeListener implements MembershipListener {

	/**
	 * Empty constructor.
	 */
	public NodeListener() {
	}

	/**
	 * Not implement because not necessary
	 * 
	 * @param event event of hazelcast
	 */
	@Override
	public void memberAdded(MembershipEvent event) {
	}

	/**
	 * Checks who is the first member on the list will work, becoming the
	 * coordinator of group.
	 * 
	 * @see org.pepstock.jem.node.Main#IS_COORDINATOR
	 * @param event event of Hazelcast
	 */

	@Override
	public void memberRemoved(MembershipEvent event) {

		// if node is shutting down, do nothing
		if (Main.IS_SHUTTING_DOWN)
			return;
		
		// get hazelcast cluster and local node
		Cluster cluster = Main.HAZELCAST.getCluster();
		Member local = cluster.getLocalMember();

		for (Member member : cluster.getMembers()) {
			// if the member if list (it means without data) and for jem means
			// the web part
			// ignores
			if (!member.isLiteMember()) {
				// if the first in the list (excluded the lite) is not the local
				// one, return
				// because only the first must work (it could be already the
				// coordinator but
				// tried to avoid double check
				if (!local.equals(member))
					return;
				else {
					// this node is the coordinator so ONLY teh coordinator must
					// work for all group
					actionsForCoordinator(event);
					// afterwards return, stopping the cycle on FOR
					return;
				}
			}
		}
	}

	/**
	 * Performs all actions necessary when a member is removed from cluster.<br>
	 * Current node will be a coordinator, setting the status UNKNOWN to removed
	 * node. It removes all locks previously asked to GRS system.
	 * 
	 * @param event event of Hazelcast
	 */
	private void actionsForCoordinator(MembershipEvent event) {

		// set that local node is coordinator
		// doesn't matter if we override with same value
		// the attribute Main.IS_COORDINATOR
		Main.IS_COORDINATOR = true;
		// get member removed
		Member memberRemoved = event.getMember();
	
		// access to map using Uuid of member removed
		IMap<String, NodeInfo> members_map = Main.HAZELCAST.getMap(Queues.NODES_MAP);
		String key = memberRemoved.getUuid();
		// check is I have on nodes map
		if (members_map.containsKey(key)) {

			try{
				members_map.lock(key);

				// get object to set status unknown
				NodeInfo info = members_map.get(key);
				info.setStatus(Status.UNKNOWN);

				if (!memberRemoved.isLiteMember()){
					// get request for locking to check if we had some locks in place
					RequestLock request = info.getRequest();
					// checks if request is null, because for list is null 
					if (request != null){
						if (!request.getResources().isEmpty()) {
							
							info.unlockForFailover();
							// clear also the request list
							request.getResources().clear();
						}
					}
					if (info.getJob() != null){
						jobEnded(info.getJob());
					}
					
				}
				// move the node in the list again, overriding the previous info
				members_map.replace(info.getKey(), info);
			} catch (Exception ex){
				LogAppl.getInstance().emit(NodeMessage.JEMC174E, ex);		
			} finally {
				members_map.unlock(key);
			}
		} else {
			// if not found, probably this is not correct!!!
			// it means that the list NODES_MAP is not well-maintained
			LogAppl.getInstance().emit(NodeMessage.JEMC024E, memberRemoved.toString(), Queues.NODES_MAP);
		}
	}
	
	/**
	 * Moves the orphan job from running queue to output, setting CC 12.
	 * @param job to move on output
	 */
	private void jobEnded(Job job) {
		
		Result result = new Result();
		result.setReturnCode(Result.FATAL);
		result.setExceptionMessage("Node is crashed during job was executing");
		job.setResult(result);
		job.setEndedTime(new Date());
		job.setRunningStatus(Job.NONE);

		LogAppl.getInstance().emit(NodeMessage.JEMC021I, job.toString(), String.valueOf(job.getResult().getReturnCode()));

		// TODO Mettere READ/WRITE locks se server
		// moves job from running to output
		IMap<String, Job> runningQueue = Main.HAZELCAST.getMap(Queues.RUNNING_QUEUE);
		IMap<String, Job> outputQueue = Main.HAZELCAST.getMap(Queues.OUTPUT_QUEUE);
		
		try {
			runningQueue.lock(job.getId());
			outputQueue.lock(job.getId());

			runningQueue.remove(job.getId());
			outputQueue.put(job.getId(), job);
			
		} catch (Exception ex){
			LogAppl.getInstance().emit(NodeMessage.JEMC175E, ex, job.getName());				
		} finally {
			runningQueue.unlock(job.getId());
			outputQueue.unlock(job.getId());
		}

		// fires event that the job is ended
		Main.JOB_LIFECYCLE_LISTENERS_SYSTEM.addJobLifecycleEvent(new JobLifecycleEvent(Queues.OUTPUT_QUEUE, job));

		// send a topic to client which is wait for
		ITopic<Job> topic = Main.HAZELCAST.getTopic(Queues.ENDED_JOB_TOPIC);
		topic.publish(job);
	}

}