/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.executors.stats;

import java.lang.management.ManagementFactory;

import org.hyperic.sigar.Cpu;
import org.hyperic.sigar.FileSystemUsage;
import org.hyperic.sigar.Mem;
import org.hyperic.sigar.ProcCpu;
import org.hyperic.sigar.ProcMem;
import org.hyperic.sigar.Sigar;
import org.hyperic.sigar.SigarException;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.Main;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.executors.DefaultExecutor;
import org.pepstock.jem.node.stats.CpuUtilization;
import org.pepstock.jem.node.stats.FileSystemUtilization;
import org.pepstock.jem.node.stats.LightMapStats;
import org.pepstock.jem.node.stats.LightMemberSample;
import org.pepstock.jem.node.stats.MapOperationsStats;
import org.pepstock.jem.node.stats.MapStats;
import org.pepstock.jem.node.stats.MemberSample;
import org.pepstock.jem.node.stats.MemoryUtilization;
import org.pepstock.jem.node.stats.ProcessCpuUtilization;
import org.pepstock.jem.node.stats.ProcessMemoryUtilization;
import org.pepstock.jem.node.stats.QueueOperationsStats;
import org.pepstock.jem.node.stats.QueueStats;
import org.pepstock.jem.node.stats.Sample;
import org.pepstock.jem.util.TimeUtils;

import com.hazelcast.monitor.LocalMapOperationStats;
import com.hazelcast.monitor.LocalMapStats;
import com.hazelcast.monitor.LocalQueueOperationStats;
import com.hazelcast.monitor.LocalQueueStats;


/**
 * Gathers all statistics for whole JEM cluster
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.2
 *
 */
public class GetSample extends DefaultExecutor<LightMemberSample> {

	private static final long serialVersionUID = 1L;
	
	private Sample newSample = null;

	/**
	 * Constructs the object using a sample (for whole JEM cluster)
	 * @param newSample sample for whole JEM cluster
	 */
	public GetSample(Sample newSample) {
		this.newSample = newSample;
	}
	
	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.executors.DefaultExecutor#checkShutDown()
	 */
	@Override
	public void checkShutDown() throws Exception {
		// NOP here
		// is called in execute method to return null!
		// to avoid that Hazelcast distributed tasks joint fails
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.executors.DefaultExecutor#execute()
	 */
	@Override
	public LightMemberSample execute() throws Exception {
		// checks if shutdown
		// if yes, return null
		// without exception
		try{
			checkShutDown();
		} catch (Exception ex){
			return null;
		}
		
		// if node is null, skip!
		// could happen if the executor is running in a starting node 
		if (Main.NODE == null)
			return null;
		// if statistics manager is null, skip!
		// could happen if the executor is running in a starting node 
		if (Main.STATISTICS_MANAGER == null)
			return null;
		
		Sigar sigar = new Sigar();
		
		// loads all member data
		MemberSample member_sample = new MemberSample();
		member_sample.setMemberKey(Main.NODE.getKey());
		member_sample.setMemberLabel(Main.NODE.getLabel());
		member_sample.setMemberHostname(Main.NODE.getHostname());
		member_sample.setPid(sigar.getPid());
		
		// gets last sample, used to calculate increments between two  samples
		MemberSample last_member_sample = Main.STATISTICS_MANAGER.getLastMemberSample();
		
		member_sample.setNumberOfJCLCheck((last_member_sample == null) ? Main.NUMBER_OF_JCL_CHECK : Main.NUMBER_OF_JCL_CHECK - last_member_sample.getTotalNumberOfJCLCheck());
		member_sample.setNumberOfJOBSubmitted((last_member_sample == null) ? Main.NUMBER_OF_JOB_SUBMITTED : Main.NUMBER_OF_JOB_SUBMITTED - last_member_sample.getTotalNumberOfJOBSubmitted());
	
		member_sample.setTotalNumberOfJCLCheck(Main.NUMBER_OF_JCL_CHECK);
		member_sample.setTotalNumberOfJOBSubmitted(Main.NUMBER_OF_JOB_SUBMITTED);
		
		loadCpuUtilization(member_sample, sigar);
		
		loadMemoryUtilization(member_sample, sigar);
		
		loadProcessCpuUtilization(member_sample, sigar);
		
		loadProcessMemoryUtilization(member_sample, sigar);
		
		loadGFSUtilization(member_sample, sigar);
		
		loadHazelcastMapsStats(member_sample);
		
		loadHazelcastQueuesStats(member_sample);
		
		newSample.getMembers().add(member_sample);
		
		Main.STATISTICS_MANAGER.write(newSample);
		
		Main.STATISTICS_MANAGER.setLastMemberSample(member_sample);
		
		return createLightMemberSample(newSample, member_sample);
	}
	
	/**
	 * loads all info 
	 * @param sample light sample (container)
	 * @param msample set of data for specific member
	 * @return a light sample for member
	 */
	private LightMemberSample createLightMemberSample(Sample sample, MemberSample msample){
		
		LightMemberSample member_sample = new LightMemberSample();
		member_sample.setMemberKey(msample.getMemberKey());
		member_sample.setMemberLabel(msample.getMemberLabel());
		member_sample.setMemberHostname(msample.getMemberHostname());
		member_sample.setPid(msample.getPid());
		member_sample.setKey(sample.getKey());
		member_sample.setCpuPercent(msample.getCpu().getPercent());
		member_sample.setMemoryAvailable(msample.getMemory().getAvailable());
		member_sample.setMemoryFree(msample.getMemory().getFree());
		member_sample.setNumberOfJCLCheck(msample.getNumberOfJCLCheck());
		member_sample.setNumberOfJOBSubmitted(msample.getNumberOfJOBSubmitted());
		member_sample.setProcessCpuPercent(msample.getProcessCpu().getPercent());
		member_sample.setProcessTotalCpu(msample.getProcessCpu().getTotal());
		member_sample.setProcessMemoryUsed(msample.getProcessMemory().getUsed());
		member_sample.setProcessMemoryFree(msample.getProcessMemory().getFree());
		member_sample.setTime(sample.getTime());
		member_sample.setTotalNumberOfJCLCheck(msample.getTotalNumberOfJCLCheck());
		member_sample.setTotalNumberOfJOBSubmitted(msample.getTotalNumberOfJOBSubmitted());
		
		member_sample.setGfsFree(msample.getFileSystem().getFree());
		member_sample.setGfsUsed(msample.getFileSystem().getUsed());
		
		/**
		 * JOBS QUEUES
		 */
		MapStats map = msample.getMapsStats().get(Queues.INPUT_QUEUE);
		member_sample.getMapsStats().put(Queues.INPUT_QUEUE, createLightMapStats(map));

		map = msample.getMapsStats().get(Queues.OUTPUT_QUEUE);
		member_sample.getMapsStats().put(Queues.OUTPUT_QUEUE, createLightMapStats(map));

		map = msample.getMapsStats().get(Queues.RUNNING_QUEUE);
		member_sample.getMapsStats().put(Queues.RUNNING_QUEUE, createLightMapStats(map));

		map = msample.getMapsStats().get(Queues.ROUTING_QUEUE);
		member_sample.getMapsStats().put(Queues.ROUTING_QUEUE, createLightMapStats(map));
		
		return member_sample;

	}
	
	/**
	 * Load all maps data
	 * @param map Hazelcast map statistics
	 * @return a light map statistics
	 */
	private LightMapStats createLightMapStats(MapStats map){
		LightMapStats lmap = new LightMapStats();
		lmap.setHits(map.getHits());
		lmap.setName(map.getName());
		
		lmap.setNumberOfGets(map.getOperationsStats().getNumberOfGets());
		lmap.setNumberOfPuts(map.getOperationsStats().getNumberOfPuts());
		lmap.setNumberOfRemoves(map.getOperationsStats().getNumberOfRemoves());
		
		lmap.setOwnedEntryCount(map.getOwnedEntryCount());
		lmap.setOwnedEntryMemoryCost(map.getOwnedEntryMemoryCost());
		
		lmap.setLockedEntryCount(map.getLockedEntryCount());
		lmap.setLockWaitCount(map.getLockWaitCount());
		
		lmap.setTotalGetLatency(map.getOperationsStats().getTotalGetLatency());
		lmap.setTotalPutLatency(map.getOperationsStats().getTotalPutLatency());
		lmap.setTotalRemoveLatency(map.getOperationsStats().getTotalRemoveLatency());
		
		return lmap;
	}
	
	/**
	 * Calculate cpu consumption.
	 * @param sample member sample
	 * @param sigar sigar instance to get system info
	 */
	private void loadCpuUtilization(MemberSample sample, Sigar sigar){
		try {
			// loads all totals
			Cpu s_cpu = sigar.getCpu();
			CpuUtilization cpu = sample.getCpu();
			cpu.setIdle(s_cpu.getIdle());
			cpu.setSystem(s_cpu.getSys());
			cpu.setTotal(s_cpu.getTotal());
			cpu.setUser(s_cpu.getUser());

//			System.err.println("MACHINE Sys: "+s_cpu.getSys()+" Usr: "+s_cpu.getUser()+" Tot: "+s_cpu.getTotal());
			// checks if this is not the first sample
			// if not, it calculates the differences between previous sample
			MemberSample last_member_sample = Main.STATISTICS_MANAGER.getLastMemberSample();
			if (last_member_sample != null) {
				// gets consumed cpu 				
				CpuUtilization last_cpu_util = last_member_sample.getCpu();
				// gets spent time between samples 
				long diffTime = sample.getCurrentTimeMillis() - last_member_sample.getCurrentTimeMillis();
				// calculates max cpu that is consumable in time frame (equals to elapsed time multiply for processor number)
				long tot_possible_cpu = diffTime * ManagementFactory.getOperatingSystemMXBean().getAvailableProcessors();
				// calculates CPU load
				long cpu_used = (cpu.getUser() + cpu.getSystem()) - (last_cpu_util.getUser() + last_cpu_util.getSystem());
				double cpu_used_percent = cpu_used * 1D / tot_possible_cpu;
				
				cpu_used_percent = Math.min(Math.max(cpu_used_percent, 0D), 1D);
				cpu.setPercent(cpu_used_percent);
//				System.err.println("MACHINE % "+cpu_used_percent*100);
			} else {
				// uses SIGAR info without any special calculation
				long tot_possible_cpu = 1 * TimeUtils.MINUTE * ManagementFactory.getOperatingSystemMXBean().getAvailableProcessors();
				long cpu_used = (cpu.getUser() + cpu.getSystem());
				double cpu_used_percent = cpu_used * 1D / tot_possible_cpu;
				
				cpu_used_percent = Math.min(Math.max(cpu_used_percent, 0D), 1D);
				cpu.setPercent(cpu_used_percent);
//				System.err.println("MACHINE % "+cpu_used_percent*100);
			}
		} catch (SigarException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC169W, e);
		}
	}

	/**
	 * Calculate memory consumption.
	 * @param sample member sample
	 * @param sigar sigar instance to get system info
	 */
	private void loadMemoryUtilization(MemberSample sample, Sigar sigar){
		try {
			Mem s_mem = sigar.getMem();
			MemoryUtilization mem = sample.getMemory();
			mem.setAvailable(s_mem.getTotal());
			mem.setUsed(s_mem.getUsed());
			mem.setFree(s_mem.getFree());
		} catch (SigarException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC169W, e);
		}
	}

	/**
	 * Calculate JEM process cpu consumption.
	 * @param sample member sample
	 * @param sigar sigar instance to get system info
	 */
	private void loadProcessCpuUtilization(MemberSample sample, Sigar sigar){
		try {
			// loads all totals
			ProcCpu s_cpu = sigar.getProcCpu(sample.getPid());
			ProcessCpuUtilization cpu = sample.getProcessCpu();
			cpu.setSystem(s_cpu.getSys());
			cpu.setTotal(s_cpu.getTotal());
			cpu.setUser(s_cpu.getUser());
//			cpu.setPercent(s_cpu.getPercent());
//			System.err.println("Pid: "+sample.getPid());
//			System.err.println("PROCESS Sys: "+s_cpu.getSys()+" Usr: "+s_cpu.getUser()+" Tot: "+s_cpu.getTotal());
//			System.err.println(s_cpu.getPercent());

			// checks if this is not the first sample
			// if not, it calculates the differences between previous sample
			MemberSample last_member_sample = Main.STATISTICS_MANAGER.getLastMemberSample();
			if (last_member_sample != null) {
				// gets consumed cpu 				
				ProcessCpuUtilization last_cpu_util = last_member_sample.getProcessCpu();
				// gets spent time between samples 
				long diffTime = sample.getCurrentTimeMillis() - last_member_sample.getCurrentTimeMillis();
				// calculates max cpu that is consumable in time frame (equals to elapsed time multiply for processor number)
				long tot_possible_cpu = diffTime * ManagementFactory.getOperatingSystemMXBean().getAvailableProcessors();
				// calculates CPU load
				long cpu_used = cpu.getTotal() - last_cpu_util.getTotal();
				double cpu_used_percent = cpu_used * 1D / tot_possible_cpu;
				
				cpu_used_percent = Math.min(Math.max(cpu_used_percent, 0D), 1D);
				cpu.setPercent(cpu_used_percent);
//				System.err.println("PROCESS %: "+cpu_used_percent*100);
			} else {
				// uses SIGAR info without any special calculation
				long tot_possible_cpu = 1 * TimeUtils.MINUTE * ManagementFactory.getOperatingSystemMXBean().getAvailableProcessors();
				long cpu_used = cpu.getTotal();
				double cpu_used_percent = cpu_used * 1D / tot_possible_cpu;
				
				cpu_used_percent = Math.min(Math.max(cpu_used_percent, 0D), 1D);
				cpu.setPercent(cpu_used_percent);
//				System.err.println("PROCESS %: "+cpu_used_percent*100);
			}
		} catch (SigarException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC169W, e);
		}
	}

	/**
	 * Calculate JEM process memory consumption.
	 * @param sample member sample
	 * @param sigar sigar instance to get system info
	 */
	private void loadProcessMemoryUtilization(MemberSample sample, Sigar sigar){
		try {
			ProcMem s_mem = sigar.getProcMem(sample.getPid());
			ProcessMemoryUtilization mem = sample.getProcessMemory();
			mem.setAvailable(s_mem.getSize());
			
//			System.out.println(s_mem.getSize()/1024/1024+" "+s_mem.getResident()/1024/1024);
			
//			MemoryMXBean mbean = ManagementFactory.getMemoryMXBean();
//			System.out.println("Max "+mbean.getHeapMemoryUsage().getMax()/1024/1024);
//			System.out.println("init "+mbean.getHeapMemoryUsage().getInit()/1024/1024);
//			System.out.println("Used "+mbean.getHeapMemoryUsage().getUsed()/1024/1024);
			
			mem.setUsed(s_mem.getResident());
			
			long free = mem.getAvailable() - mem.getUsed();
			mem.setFree(free);
			
		} catch (SigarException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC169W, e);
		}
	}

	/**
	 * Claculates GFS usage. 
	 * @param sample member sample
	 * @param sigar sigar instance to get system info
	 */
	private void loadGFSUtilization(MemberSample sample, Sigar sigar){
		try {
			FileSystemUtilization fsUtil = sample.getFileSystem();
			FileSystemUsage usage = sigar.getFileSystemUsage(Main.OUTPUT_SYSTEM.getOutputPath().getAbsolutePath());
			long free = usage.getFree();
			long total  = usage.getTotal();
			long used  = usage.getUsed();
			
			fsUtil.setFree(free);
			fsUtil.setTotal(total);
			fsUtil.setUsed(used);

		} catch (SigarException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC169W, e);
		}
	}

	
	/**
	 * Loads all Hazelcast maps stats
	 * @param sample member sample
	 */
	private void loadHazelcastMapsStats(MemberSample sample){
		MapStats input = loadMapStats(Queues.INPUT_QUEUE);
		MapStats running = loadMapStats(Queues.RUNNING_QUEUE);
		MapStats output = loadMapStats(Queues.OUTPUT_QUEUE);
		MapStats routing = loadMapStats(Queues.ROUTING_QUEUE);
		MapStats routed = loadMapStats(Queues.ROUTED_QUEUE);
		MapStats resources = loadMapStats(Queues.COMMON_RESOURCES_MAP);
		MapStats roles = loadMapStats(Queues.ROLES_MAP);
		MapStats stats = loadMapStats(Queues.STATS_MAP);
		MapStats preferences = loadMapStats(Queues.USER_PREFERENCES_MAP);
//		MapStats grs = loadMapStats(LockStructures.COUNTER_MUTEX);
		
		sample.getMapsStats().put(input.getName(), input);
		sample.getMapsStats().put(running.getName(), running);
		sample.getMapsStats().put(output.getName(), output);
		sample.getMapsStats().put(routing.getName(), routing);
		
		sample.getMapsStats().put(routed.getName(), routed);
		sample.getMapsStats().put(resources.getName(), resources);
		sample.getMapsStats().put(roles.getName(), roles);
//		sample.getMapsStats().put(grs.getName(), grs);
		sample.getMapsStats().put(stats.getName(), stats);
		sample.getMapsStats().put(preferences.getName(), preferences);
	}
	
	/**
	 * Loads Hazelcast queue stats
	 * @param sample member sample
	 */
	private void loadHazelcastQueuesStats(MemberSample sample){
		QueueStats jclCheck = loadQueueStats(Queues.JCL_CHECKING_QUEUE);

		sample.getQueuesStats().put(jclCheck.getName(), jclCheck);
	}
	
	/**
	 * Reads statistics info of Hazelcast MAPS
	 * 
	 * @param mapName map name to access to Hazelcast
	 * @return set of statistics of map
	 */
	private MapStats loadMapStats(String mapName){
			LocalMapStats stats = Main.HAZELCAST.getMap(mapName).getLocalMapStats();
			LocalMapOperationStats ostats = stats.getOperationStats();
			
			MapStats mstats = new MapStats();
			mstats.setName(mapName);
			
			mstats.setBackupEntryCount(stats.getBackupEntryCount());
			mstats.setBackupEntryMemoryCost(stats.getBackupEntryMemoryCost());
			mstats.setDirtyEntryCount(stats.getDirtyEntryCount());
			mstats.setHits(stats.getHits());
			mstats.setLockedEntryCount(stats.getLockedEntryCount());
			mstats.setLockWaitCount(stats.getLockWaitCount());
			mstats.setOwnedEntryCount(stats.getOwnedEntryCount());
			mstats.setOwnedEntryMemoryCost(stats.getOwnedEntryMemoryCost());
			
			MapOperationsStats mostats = mstats.getOperationsStats();
			
			mostats.setNumberOfEvents(ostats.getNumberOfEvents());
			mostats.setNumberOfGets(ostats.getNumberOfGets());
			mostats.setNumberOfOtherOperations(ostats.getNumberOfOtherOperations());
			mostats.setNumberOfPuts(ostats.getNumberOfPuts());
			mostats.setNumberOfRemoves(ostats.getNumberOfRemoves());
			mostats.setTotalGetLatency(ostats.getTotalGetLatency());
			mostats.setTotalPutLatency(ostats.getTotalPutLatency());
			mostats.setTotalRemoveLatency(ostats.getTotalRemoveLatency());
			
			return mstats;
	}
	
	/**
	 * Reads statistics info of Hazelcast QUEUES
	 * 
	 * @param queueName queue name to access to Hazelcast
	 * @return set of statistics of map
	 */
	private QueueStats loadQueueStats(String queueName){
		LocalQueueStats stats = Main.HAZELCAST.getQueue(queueName).getLocalQueueStats();
		LocalQueueOperationStats ostats = stats.getOperationStats();
		
		QueueStats qstats = new QueueStats();
		qstats.setName(queueName);
		
		qstats.setAveAge(stats.getAveAge());
		qstats.setBackupItemCount(stats.getBackupItemCount());
		qstats.setMaxAge(stats.getMaxAge());
		qstats.setMinAge(stats.getMinAge());
		qstats.setOwnedItemCount(stats.getOwnedItemCount());
		
		QueueOperationsStats qostats = qstats.getOperationsStats();
		qostats.setNumberOfEmptyPolls(ostats.getNumberOfEmptyPolls());
		qostats.setNumberOfOffers(ostats.getNumberOfOffers());
		qostats.setNumberOfPolls(ostats.getNumberOfPolls());
		qostats.setNumberOfRejectedOffers(ostats.getNumberOfRejectedOffers());
		
		return qstats;
	}

}