/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.executors.jobs;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Scanner;

import org.pepstock.jem.Job;
import org.pepstock.jem.OutputFileContent;
import org.pepstock.jem.OutputListItem;
import org.pepstock.jem.node.Main;
import org.pepstock.jem.node.executors.DefaultExecutor;

/**
 * Returns the content of logs of job.<br>
 * Can get both Message log and Job log.<br>
 * Useful by Graphic User Interface
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class GetOutputFileContent extends DefaultExecutor<OutputFileContent> {

	private static final long serialVersionUID = 1L;
	
	private static final int MAX_NUMBER_OF_BYTE_READABLE = 1024 * 1024 * 5; //5MB

	@SuppressWarnings("unused")
	private Job job = null;

	private OutputListItem item = null;

	/**
	 * Constructs the object passing the job which you need logs of, and relative file by item (relative to <code>ouptut</code> path to download.
	 * 
	 * @param job job instance
	 * @param item item which represents teh file to download to UI
	 */
	public GetOutputFileContent(Job job, OutputListItem item) {
		this.job = job;
		this.item = item;
	}

	/**
	 * Checks file you need, reads it from file system and returns it.
	 * 
	 * @return instance with content of log
	 * @throws Exception occurs if errors
	 */
	@Override
	public OutputFileContent execute() throws Exception {
		// gets the file to download
		File file = new File(Main.OUTPUT_SYSTEM.getOutputPath(), item.getFileRelativePath());

		// checks if file exists otherwise exception occurs
		if (!file.exists())
			throw new FileNotFoundException(file.getAbsolutePath());

		// loads content file into a buffer
		// must be check the file size... if too big could create problems
		StringBuffer sb = load(file);

		// creates a output container and sets job file content
		OutputFileContent output = new OutputFileContent();
		output.setContent(sb.toString());
		return output;
	}

	/**
	 * Reads the content of log file and puts content in a string-buffer.
	 * 
	 * @param file log file to read
	 * @return string with content of file
	 * @throws IOException occurs it errors
	 */
	private StringBuffer load(File file) throws IOException {
		// reads using a Scanner, and \n as delimiter of rows
		Scanner sc = new Scanner(file);
		sc.useDelimiter("\n");
		StringBuffer sb = new StringBuffer();
		while (sc.hasNext() && sb.length() < MAX_NUMBER_OF_BYTE_READABLE) {
			sb.append(sc.next().toString()).append('\n');
		}
		sc.close();
		return sb;
	}
}