/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.executors.gfs;

import java.io.File;

import org.apache.commons.io.FileUtils;
import org.pepstock.jem.GfsFile;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.configuration.ConfigKeys;
import org.pepstock.jem.node.executors.DefaultExecutor;

/**
 * The executor returns the list of files and/or directories in a specific folder.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.2	
 *
 */
public class GetFile extends DefaultExecutor<String> {
	
	private static final long serialVersionUID = 1L;

	private String file = null;
	
	private int type = GfsFile.DATA;
	
	/**
	 * Saves the type of GFS to read and the folder
	 * 
	 * @param type could a integer value
	 * @see GfsFile
	 * @param file the folder (relative to type of GFS) to use to read files and directories
	 * 
	 */
	public GetFile(int type, String file) {
		this.type = type;
		this.file = file;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.executors.DefaultExecutor#execute()
	 */
	@Override
	public String execute() throws Exception {

		String parentPath = null;
		File fileToRead = null;
		
		// checks here the type of file-system to scan
		switch(type){
			case GfsFile.DATA:
				parentPath = System.getProperty(ConfigKeys.JEM_DATA_PATH_NAME);
			break;
			case GfsFile.LIBRARY:
				parentPath = System.getProperty(ConfigKeys.JEM_LIBRARY_PATH_NAME);
			break;
			case GfsFile.SOURCE:
				parentPath = System.getProperty(ConfigKeys.JEM_SOURCE_PATH_NAME);
			break;
			case GfsFile.CLASS:
				parentPath = System.getProperty(ConfigKeys.JEM_CLASSPATH_PATH_NAME);
			break;
			case GfsFile.BINARY:
				parentPath = System.getProperty(ConfigKeys.JEM_BINARY_PATH_NAME);
			break;

			default:
				parentPath = System.getProperty(ConfigKeys.JEM_DATA_PATH_NAME);
		}
		fileToRead = new File(parentPath, file);

		// checks if folder exists and must be a folder (not a file)
		if (!fileToRead.exists())
			throw new Exception(NodeMessage.JEMC186E.toMessage().getFormattedMessage(file));
		if (!fileToRead.isFile())
			throw new Exception(NodeMessage.JEMC188E.toMessage().getFormattedMessage(file));
		
		String content = FileUtils.readFileToString(fileToRead);
		return content;
	}

}