/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.executors;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.concurrent.Callable;
import java.util.jar.Attributes;
import java.util.jar.JarFile;

import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.About;
import org.pepstock.jem.node.Main;
import org.pepstock.jem.node.NodeLicense;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.configuration.ConfigKeys;

/**
 * Extracts licenses info if exist.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.3
 * 
 */
public class GetAbout extends DefaultExecutor<About>{

	private static final long serialVersionUID = 1L;
	
	/**
	 * Calls an executor to extract all licences information, only if enterprise
	 * configuration is running
	 * @return bean with all info to show on UI
	 * @throws Exception occurs if errors
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public About execute() throws Exception {
		// I need the reflection to avoid to distribute 
		// create reference on enterprise
		About about = new About();
		try {
			Class clazz = Class.forName("org.pepstock.jem.license.GetLicensesInfo");
			Callable<Collection<NodeLicense>> da = (Callable<Collection<NodeLicense>>) clazz.newInstance();
			about.getLicenses().addAll(da.call());
		} catch (ClassNotFoundException e) {
			// nothing, means the enterprise is not loaded
		} catch (Exception e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC207E, e);
		}
		loadManifest(about);
		return about;
	}
	
	/**
	 * Loads the info which are present in manifst file of JAR  
	 * @param about bean to load
	 */
	private void loadManifest(About about){
		JarFile jarFile = null;
		try {
			// gets JAR file
			jarFile = new JarFile(new File(Main.class.getProtectionDomain().getCodeSource().getLocation().toURI()));
			
			// gets attributes
			Attributes at = (Attributes) jarFile.getManifest().getAttributes(ConfigKeys.JEM_MANIFEST_SECTION);
			// gets version
			String version = at.getValue(ConfigKeys.JEM_MANIFEST_VERSION);
			if (version != null){
				about.setVersion(version);
			}
			// gets build time
			String creation = at.getValue(ConfigKeys.JEM_MANIFEST_CREATION_TIME);
			if (creation != null){
				about.setCreationTime(creation);
			}
		} catch (IOException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} catch (URISyntaxException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} finally {
			if (jarFile != null) {
				try {
					jarFile.close();
				} catch (IOException e) {
				}
			}
		}
	}
}