/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node.affinity;

import java.io.File;
import java.io.FileReader;
import java.io.Reader;
import java.io.StringReader;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.pepstock.jem.Jcl;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.util.Parser;

/**
 * Is a loader of affinity and uses a javascript code to load simply all affinities for the node.<br>
 * The policy file in JS must be passed in the properties in <code>init</code> method.<br>
 * Prepares a global variable for the JS, named <code>SYSINFO</code>. Is a instance of SystemInfo java class.<br>
 * Reads 2 variables use inside of JS:<br>
 * <ul>
 * <li><b>RESULT</b>: a set of affinities labels, comma separated</li>
 * <li><b>MEMORY</b>: memory value to use to executed the job, must be a integer</li>
 * </ul>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class JSPolicyAffinityLoader extends PolicyAffinityLoader {
	
	public static final String TYPE = "javascript";
	
	private static final String JS_RESULT_VARIABLE = "RESULT";

	private static final String JS_MEMORY_VARIABLE = "MEMORY";
	
	private static final String JS_SYSINFO_VARIABLE = "SYSINFO";

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.affinity.ScriptAffinityLoader#runScript(java.io.File, org.pepstock.jem.node.affinity.SystemInfo)
	 */
	@Override
	public Result runScript(File script, SystemInfo info) throws Exception {
		// reader of JS file
		return runScript(new FileReader(script), info);
	}


	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.affinity.ScriptAffinityLoader#getScriptType()
	 */
	@Override
	public String getScriptType() {
		return TYPE;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.affinity.ScriptAffinityLoader#testScript(java.lang.String, org.pepstock.jem.node.affinity.SystemInfo)
	 */
	@Override
	public Result testScript(String script, SystemInfo info) throws Exception {
		return runScript(new StringReader(script), info);
	}
	
	/**
	 * 
	 * @param reader
	 * @param info
	 * @return
	 * @throws Exception
	 */
	private Result runScript(Reader reader, SystemInfo info) throws Exception {		
		// creates JS context
		Result result = new Result();
		Context jsContext = Context.enter();

		try {
			//inits JS context
            ScriptableObject scope = jsContext.initStandardObjects();

            // Collect the arguments into a single string.
            String s = "";
            //            for (int i=0; i < args.length; i++) {
            //                s += args[i]+" ";
            //            }

            // Set up "SYSINFO" in the global scope to contain Sysinfo java object
            jsContext.newObject(scope);
            scope.defineProperty(JS_SYSINFO_VARIABLE, info, ScriptableObject.READONLY);
            
            // Now evaluate the string we've collected. We'll ignore the result.
            jsContext.evaluateReader(scope, reader, s, 1, null);

            // gets the value of variable "RESULT"
            Object x = scope.get(JS_RESULT_VARIABLE, scope);
            if (x == Scriptable.NOT_FOUND) {
            	LogAppl.getInstance().emit(NodeMessage.JEMC092E, JS_RESULT_VARIABLE);
            } else {
            	// gets string values
                String affinities = Context.toString(x);
                if (affinities != null){
                	if (affinities.trim().length() > 0){
                		// parses the value (comma separated)
                		String[] affinitiesValues = affinities.split(",");
                		// loads into result collection
                		for (int i=0; i<affinitiesValues.length; i++){
                			result.getAffinities().add(affinitiesValues[i].trim().toLowerCase());
                		}
                	}
                }
            }
            // Gets the value of variable "MEMORY"
            Object xm = scope.get(JS_MEMORY_VARIABLE, scope);
            if (xm == Scriptable.NOT_FOUND) {
            	LogAppl.getInstance().emit(NodeMessage.JEMC092E, JS_MEMORY_VARIABLE);
            } else {
            	// gets value string and parses in integer.
            	// if is not a integer, uses teh default 
                String memoryString = Context.toString(xm);
                int memory = Parser.parseInt(memoryString, Jcl.DEFAULT_MEMORY);
                // sets in result object
                result.setMemory(memory);
            }
        } finally {
        	// cleanup of JS context
            Context.exit();
        }
		return result;
	}

}