/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.rmi.RemoteException;
import java.util.Date;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.Job;
import org.pepstock.jem.Result;
import org.pepstock.jem.factories.JemFactory;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.events.JobLifecycleEvent;
import org.pepstock.jem.node.rmi.CommonResourcer;
import org.pepstock.jem.node.rmi.CommonResourcerImpl;
import org.pepstock.jem.node.rmi.ExternalObject;
import org.pepstock.jem.node.rmi.ResourceLocker;
import org.pepstock.jem.node.rmi.ResourceLockerImpl;
import org.pepstock.jem.node.rmi.TasksDoor;
import org.pepstock.jem.node.rmi.TasksDoorImpl;
import org.pepstock.jem.node.tasks.JobTask;
import org.pepstock.jem.util.TimeUtils;
import org.pepstock.jem.util.rmi.RegistryContainer;

import com.hazelcast.core.IMap;
import com.hazelcast.core.ITopic;

/**
 * Is a thread, responsible to execute jobs which are removed from input queue
 * to be executed here. It creates JobTask, it submits it, manages exceptions
 * which are thrown by job execution, moves the job from RUNNING to OUTPUT queue
 * and manages different status of node.
 * 
 * @see org.pepstock.jem.node.InputQueueManager#InputQueueManager()
 * @see org.pepstock.jem.node.Status#Status(int, String)
 * @see org.pepstock.jem.node.Queues#RUNNING_QUEUE
 * @see org.pepstock.jem.node.Queues#OUTPUT_QUEUE
 * @author Andrea "Stock" Stocchero
 * 
 */
public class Submitter extends Thread implements ShutDownInterface{
	
	private static final String CLASS_FOR_EXTERNAL = "org.pepstock.jem.ant.AntUtilManager";

	private ExecutorService executor = null;
	
	private boolean isDown = false;

	/**
	 * Constructs a submitter. It creates a executor to execute jobs.
	 */
	public Submitter() {
		executor = Executors.newSingleThreadExecutor();
	}

	/**
	 * Register 2 RMI objects to RMI registry, necessary during the job
	 * execution to notify the end of steps and ask for global resource locks.<br>
	 * Then it takes job from queue and execute it.
	 * 
	 * @see org.pepstock.jem.node.rmi.ResourceLocker
	 * @see org.pepstock.jem.node.rmi.TasksDoor
	 * @see org.pepstock.jem.util.rmi.RegistryContainer#addRmiObject(String,
	 *      org.pepstock.jem.util.rmi.RmiObject)
	 * @see org.pepstock.jem.node.InputQueueManager#takeJob()
	 */
	public void run() {
		try {
			// create and load RMI object for step listeners
			RegistryContainer.getInstance().addRmiObject(TasksDoor.NAME, new TasksDoorImpl());
			LogAppl.getInstance().emit(NodeMessage.JEMC015I, TasksDoor.NAME);

			// create and load RMI object for global resource looking
			RegistryContainer.getInstance().addRmiObject(ResourceLocker.NAME, new ResourceLockerImpl());
			LogAppl.getInstance().emit(NodeMessage.JEMC015I, ResourceLocker.NAME);

			// create and load RMI object for common resources
			RegistryContainer.getInstance().addRmiObject(CommonResourcer.NAME, new CommonResourcerImpl());
			LogAppl.getInstance().emit(NodeMessage.JEMC015I, CommonResourcer.NAME);

			try {
				// Try to load internal utilities. Due to they are not free
				// and in another project, avoiding to create useless dependency they are
				// loaded dynamically, by reflection
				@SuppressWarnings("rawtypes")
				Class internals = Class.forName(CLASS_FOR_EXTERNAL);
				ExternalObject externalObject = (ExternalObject)internals.newInstance();
				// create and load RMI object for internal utilities
				RegistryContainer.getInstance().addRmiObject(externalObject.getName(), externalObject.getObject());
				LogAppl.getInstance().emit(NodeMessage.JEMC195I, externalObject.getName());

			} catch (ClassNotFoundException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC196W, CLASS_FOR_EXTERNAL);
			} catch (InstantiationException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC196W, CLASS_FOR_EXTERNAL);
			} catch (IllegalAccessException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC196W, CLASS_FOR_EXTERNAL);
			} catch (RemoteException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC196W, CLASS_FOR_EXTERNAL);
			} 
		} catch (RemoteException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC016E, e, TasksDoor.NAME);
			return;
		}

		while (!isDown) {
			try {
				// check if there is any job to submit
				Main.INPUT_QUEUE_MANAGER.checkJobToSubmit();

				// take job from internal queue or wait
				Job job = Main.INPUT_QUEUE_MANAGER.takeJob();

				// having job, if not null submit
				if (job != null){
					if (job instanceof ShutDownJob){
						isDown = true;
					} else  {
						submit(job);
					}
				}
			} catch (InterruptedException e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC068W, StringUtils.substringAfterLast(Submitter.class.getName(), "."));
			}
		}
		LogAppl.getInstance().emit(NodeMessage.JEMC069I, StringUtils.substringAfterLast(Submitter.class.getName(), "."));
	}

	/**
	 * Creates job task, execute it, manages exceptions which are thrown by job
	 * execution and manages different status of node.
	 * 
	 * @param job job instance
	 */
	public synchronized void submit(Job job) {
		if (job == null)
			return;
//		Main.START = System.currentTimeMillis();
		Main.NUMBER_OF_JOB_SUBMITTED++;
		
		// sets current job which is executed
		// TODO addJob
		Main.NODE.setJob(job);

		// sets job name as requestor for GRS name
		// each node can have only one request for grs
		Main.NODE.getRequest().setRequestorName(job.getName());
		Main.NODE.getRequest().setRequestorId(job.getId());

		// store node info in NODES_QUEUE map
		NodeInfoUtility.storeNodeInfo(Main.NODE);

		// gets the JEM factory to have jobtask
		JemFactory factory = Main.FACTORIES_LIST.get(job.getJcl().getType());
		JobTask at = factory.createJobTask(job);

		// saves jobtask in a static reference so accessible everywhere
		Main.CURRENT_TASK = new CancelableTask(at);

		// fires event that job is running
		Main.JOB_LIFECYCLE_LISTENERS_SYSTEM.addJobLifecycleEvent(new JobLifecycleEvent(Queues.RUNNING_QUEUE, job));

		// executes jobtask
		executor.submit(Main.CURRENT_TASK);
		LogAppl.getInstance().emit(NodeMessage.JEMC020I, job.toString());

		Result result = null;
		StringWriter sw = new StringWriter();
		try {

			// it waits for the end of job
			result = Main.CURRENT_TASK.get();

			// if no exceptions and no canceled, prepares Result and finalizes
			// job end
			if (result.getReturnCode() != Result.CANCELED) {
				job.setResult(result);
				job.setEndedTime(new Date());
				jobEnded(job);
			}
		} catch (CancellationException e) {
			// save the complete stack trace
			e.printStackTrace(new PrintWriter(sw));

			// job has been canceled
			result = new Result();
			result.setReturnCode(Result.CANCELED);
			result.setExceptionMessage(sw.getBuffer().toString());
			job.setResult(result);
			job.setEndedTime(new Date());
			jobEnded(job);

		} catch (InterruptedException e) {
			// save the complete stack trace
			e.printStackTrace(new PrintWriter(sw));

			// interrupt occurs during waiting for execution end
			result = new Result();
			result.setReturnCode(Result.ERROR);
			result.setExceptionMessage(sw.getBuffer().toString());
			job.setResult(result);
			job.setEndedTime(new Date());
			jobEnded(job);

		} catch (ExecutionException e) {
			// save the complete stack trace
			e.printStackTrace(new PrintWriter(sw));

			// System error during executor running
			result = new Result();
			result.setReturnCode(Result.SEVERE);
			result.setExceptionMessage(sw.getBuffer().toString());
			job.setResult(result);
			job.setEndedTime(new Date());
			jobEnded(job);
		} catch (Exception e) {
			// save the complete stack trace
			e.printStackTrace(new PrintWriter(sw));

			// System error during executor running
			result = new Result();
			result.setReturnCode(Result.SEVERE);
			result.setExceptionMessage(sw.getBuffer().toString());
			job.setResult(result);
			job.setEndedTime(new Date());
			jobEnded(job);			
		}
		
		// checks id there is any resources and then locks still active.
		// could happen when the job has been cancelled
		if (!Main.NODE.getRequest().getResources().isEmpty()){
			Main.NODE.getRequest().unlock();
			Main.NODE.getRequest().getResources().clear();
		}
		
		// prints job log footer and clean jobtask static reference
		JobLogManager.printFooter(job, result.getReturnCode(), result.getExceptionMessage());
		// writes JOB
		Main.OUTPUT_SYSTEM.writeJob(job);

		Main.CURRENT_TASK = null;
		
		Main.NODE.getLock().lock();
		try {
			// checks which new status must be set for node
			if (Main.NODE.getStatus().equals(Status.DRAINING)) {
				Main.NODE.setStatus(Status.DRAINED);
			} else {
				Main.NODE.setStatus(Status.INACTIVE);
			}
			// cleans nodeinfo data
			Main.NODE.setJob(null);

			// clear information of job inside the request for grs, using the ky and
			// label of GRS node
			Main.NODE.getRequest().setRequestorName(Main.NODE.getKey());
			Main.NODE.getRequest().setRequestorName(Main.NODE.getLabel());

			// store node info in NODES_QUEUE map
			NodeInfoUtility.storeNodeInfo(Main.NODE);
			LogAppl.getInstance().emit(NodeMessage.JEMC030I, Main.NODE.getStatus());
		} finally {
			Main.NODE.getLock().unlock();
		}
	}

	/**
	 * Moves teh job instance from RUNNING queue to OUTPUT queue.<br>
	 * It notify the message for job end, using Hazelcast topic structure.
	 * 
	 * @see org.pepstock.jem.node.Queues#RUNNING_QUEUE
	 * @see org.pepstock.jem.node.Queues#OUTPUT_QUEUE
	 * @see org.pepstock.jem.node.Queues#ENDED_JOB_TOPIC
	 * @param job job instance
	 */
	private void jobEnded(Job job) {
		job.setRunningStatus(Job.NONE);
		LogAppl.getInstance().emit(NodeMessage.JEMC021I, job.toString(), String.valueOf(job.getResult().getReturnCode()));

		// moves job from running to output
		IMap<String, Job> runningQueue = Main.HAZELCAST.getMap(Queues.RUNNING_QUEUE);
		IMap<String, Job> outputQueue = Main.HAZELCAST.getMap(Queues.OUTPUT_QUEUE);
		// TODO Mettere READ/WRITE locks se server
		try {
			
			runningQueue.lock(job.getId());
			outputQueue.lock(job.getId());

			runningQueue.remove(job.getId());
			outputQueue.put(job.getId(), job);
			
		} catch (Exception ex){
			LogAppl.getInstance().emit(NodeMessage.JEMC175E, ex, job.getName());			
		} finally {
			runningQueue.unlock(job.getId());
			outputQueue.unlock(job.getId());
		}

		// fires event that the job is ended
		Main.JOB_LIFECYCLE_LISTENERS_SYSTEM.addJobLifecycleEvent(new JobLifecycleEvent(Queues.OUTPUT_QUEUE, job));

		// send a topic to client which is wait for
		ITopic<Job> topic = Main.HAZELCAST.getTopic(Queues.ENDED_JOB_TOPIC);
		topic.publish(job);
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.ShutDownInterface#shutdown()
	 */
	@Override
	public void shutdown() throws Exception {
		while (!isDown){
			Thread.sleep(1 * TimeUtils.SECOND);
		}
	}
}