/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.Key;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.TimeUnit;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.io.FilenameUtils;
import org.pepstock.jem.Jcl;
import org.pepstock.jem.Job;
import org.pepstock.jem.factories.JemFactory;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.affinity.AffinityLoader;
import org.pepstock.jem.node.affinity.Result;
import org.pepstock.jem.node.affinity.SystemInfo;
import org.pepstock.jem.node.configuration.AffinityFactory;
import org.pepstock.jem.node.configuration.ConfigKeys;
import org.pepstock.jem.node.configuration.Configuration;
import org.pepstock.jem.node.configuration.ConfigurationException;
import org.pepstock.jem.node.configuration.CustomResourceDefinition;
import org.pepstock.jem.node.configuration.Database;
import org.pepstock.jem.node.configuration.Factory;
import org.pepstock.jem.node.configuration.Listener;
import org.pepstock.jem.node.configuration.Node;
import org.pepstock.jem.node.configuration.Paths;
import org.pepstock.jem.node.configuration.StatsManager;
import org.pepstock.jem.node.configuration.SwarmConfiguration;
import org.pepstock.jem.node.events.JobLifecycleListener;
import org.pepstock.jem.node.listeners.NodeListener;
import org.pepstock.jem.node.persistence.CommonResourcesDBManager;
import org.pepstock.jem.node.persistence.DBPoolManager;
import org.pepstock.jem.node.persistence.JobDBManager;
import org.pepstock.jem.node.persistence.PreJobDBManager;
import org.pepstock.jem.node.persistence.RecoveryManager;
import org.pepstock.jem.node.persistence.RolesDBManager;
import org.pepstock.jem.node.persistence.RoutingConfigDBManager;
import org.pepstock.jem.node.persistence.SQLContainer;
import org.pepstock.jem.node.persistence.SQLContainerFactory;
import org.pepstock.jem.node.persistence.UserPreferencesDBManager;
import org.pepstock.jem.node.persistence.sql.DefaultSQLContainerFactory;
import org.pepstock.jem.node.persistence.sql.MySqlSQLContainerFactory;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.ResourcesUtil;
import org.pepstock.jem.node.resources.custom.ResourceDefinitionException;
import org.pepstock.jem.node.resources.custom.ResourceDefinitionsManager;
import org.pepstock.jem.node.security.Role;
import org.pepstock.jem.node.security.UserPreference;
import org.pepstock.jem.node.security.keystore.KeysUtil;
import org.pepstock.jem.util.VariableSubstituter;
import org.xml.sax.SAXException;

import com.hazelcast.config.FileSystemXmlConfig;
import com.hazelcast.core.Cluster;
import com.hazelcast.core.Hazelcast;
import com.hazelcast.core.ILock;
import com.hazelcast.core.IMap;
import com.hazelcast.core.ISemaphore;
import com.hazelcast.core.Member;
import com.thoughtworks.xstream.XStream;

/**
 * It starts up loading the configuration from a file, passed by a system
 * property. <br>
 * It initialize the Log4j, using LogAppl and a system property. <br>
 * It loads the execution environment, factories, paths and listeners (if they
 * are defined)<br>
 * Afterwards it initialize Hazelcast cluser checking is the configuration is
 * coherent with the Jem configuration, because the groupname of Hazelcast must
 * be the same of the defined environment in jem conf.<br>
 * 
 * 
 * @see org.pepstock.jem.log.LogAppl#getInstance()
 * @see org.pepstock.jem.node.configuration.ConfigKeys#JEM_CONFIG
 * @author Andrea "Stock" Stocchero
 * 
 */
public class StartUpSystem {

	private static final Properties PROPERTIES = new Properties();

	private static final String SEMAPHORE = "org.pepstock.jem.semaphore";

	/**
	 * Main method which calls JEM configuration loading and then Hazelcast
	 * initialization.
	 * 
	 * @throws ConfigurationException if a configuration error, exception occurs
	 */
	public static void run() throws ConfigurationException {
		// load jem-node.xml configuration from node folder
		loadConfiguration();
		// load jem-env.xml configuration from gfs
		loadEnvConfiguration();
		// initialize Hazelcast
		startHazelcast();
		// delete resources of not existing types
		deleteNotExistingTypesResources();
		LogAppl.getInstance().emit(NodeMessage.JEMC012I, ManagementFactory.getRuntimeMXBean().getName());

//
//		TODO this is just a test. Must be developed on next versions
//
//		PartitionService partitionService = Main.HAZELCAST.getPartitionService();
//		partitionService.addMigrationListener(new MigrationListener () {
//			
//			private final Object lock= new Object();
//			
//			private Thread dataLossHandlerThread = null;
//			
//			private DataLossHandler dataLossHandler = null;
//
//			@Override
//			public void migrationCompleted(MigrationEvent arg0) {
//				System.err.println("Completed "+arg0.getPartitionId()+" "+arg0.getOldOwner()+" "+arg0.getNewOwner()+" "+Main.HAZELCAST.getMap(Queues.OUTPUT_QUEUE).size());
//				if (arg0.getOldOwner() == null){
//					synchronized (lock) {
//						if (dataLossHandler == null){
//							dataLossHandler = new DataLossHandler();
//						}
//						dataLossHandler.setLastMigrationCompleted(System.currentTimeMillis());
//						
//						if (dataLossHandlerThread == null){
//							dataLossHandlerThread = new Thread(dataLossHandler);
//							dataLossHandlerThread.start();
//						} else if (dataLossHandler.isReady()){
//							dataLossHandlerThread.interrupt();
//							dataLossHandlerThread = null;
//							dataLossHandlerThread = new Thread(dataLossHandler);
//							dataLossHandlerThread.start();
//						}
//					}
//				}
//				if (dataLossHandlerThread != null){
//					System.err.println("is running? "+dataLossHandlerThread.isAlive());	
//				}
//				
//			}
//
//			@Override
//			public void migrationFailed(MigrationEvent arg0) {
//				System.err.println("Failed");
//			}
//
//			@Override
//			public void migrationStarted(MigrationEvent arg0) {
//				System.err.println("Started");
//			}
//
//
//		});
	}

	
	/**
	 * starts up of Hazelcast
	 * 
	 * @throws ConfigurationException if a configuration error, exception occurs
	 */
	private static void startHazelcast() throws ConfigurationException {
		LogAppl.getInstance().emit(NodeMessage.JEMC002I);

		// creates a key anyway, even if couldn't be necessary, to avoid synch
		// in Hazelcast
		ResourcesUtil.getInstance().createKey();

		// reads Hazecast init parameter
		String hazelcastFile = System.getProperty(ConfigKeys.HAZELCAST_CONFIG);
		if (hazelcastFile == null) {

			LogAppl.getInstance().emit(NodeMessage.JEMC005E, ConfigKeys.HAZELCAST_CONFIG);
			throw new ConfigurationException(NodeMessage.JEMC005E.toMessage().getFormattedMessage(ConfigKeys.HAZELCAST_CONFIG));
		} else {
			// loads configuration file
			FileSystemXmlConfig config;
			try {
				// XML syntax check
				// because Hazelcast continues if XMl error occurs
				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
				DocumentBuilder builder = factory.newDocumentBuilder();
				builder.parse(new FileInputStream(hazelcastFile));

				
				config = new FileSystemXmlConfig(hazelcastFile);
				
				// checks if the password defined on hazelcast is the same of
				// the
				// constant defined (to avoid to managed all password, is
				// useless)
				String pwdFromConfig = config.getGroupConfig().getPassword();
				if ((pwdFromConfig == null) || (pwdFromConfig.trim().equals(""))) {
					throw new ConfigurationException(NodeMessage.JEMC108E.toMessage().getFormattedMessage());
				}
				Main.HAZELCAST_CONFIG = config;
				
				// to avoid to loose data, sets Hazelcast shutdown hook disable
				// It must be set before to create Hazelcast Instance
				System.setProperty("hazelcast.shutdownhook.enabled", "false");
				
				Main.HAZELCAST = Hazelcast.newHazelcastInstance(config);

			} catch (FileNotFoundException e) {
				throw new ConfigurationException(e);
			} catch (ParserConfigurationException e) {
				throw new ConfigurationException(e);
			} catch (SAXException e) {
				throw new ConfigurationException(e);
			} catch (IOException e) {
				throw new ConfigurationException(e);
			}

		}

		ILock lock = Main.HAZELCAST.getLock(Queues.STARTUP_LOCK);
		try{
			lock.lock();
			// get the cluster adding a new listener and extract local member to
			// retrieve information
			Cluster cluster = Main.HAZELCAST.getCluster();
			cluster.addMembershipListener(new NodeListener());
			// Main.HAZELCAST.getPartitionService().addMigrationListener(new
			// DataMoverListener());

			Member member = cluster.getLocalMember();

			// load node information starting from member object
			try {
				NodeInfoUtility.loadNodeInfo(member, Main.NODE);
			} catch (Exception e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC147E, e);
				throw new ConfigurationException(NodeMessage.JEMC147E.toMessage().getMessage(), e);
			}
			NodeInfoUtility.storeNodeInfo(Main.NODE);
			LogAppl.getInstance().emit(NodeMessage.JEMC030I, Main.NODE.getStatus());
			LogAppl.getInstance().emit(NodeMessage.JEMC003I, Main.NODE.getKey());

			// checks if the group name is the same of enviroment attributes defined
			// in jem config file
			String group = Main.HAZELCAST.getConfig().getGroupConfig().getName();
			if (!group.equalsIgnoreCase(Main.EXECUTION_ENVIRONMENT.getEnvironment())) {
				LogAppl.getInstance().emit(NodeMessage.JEMC010E, Main.EXECUTION_ENVIRONMENT.getEnvironment(), group);
				throw new ConfigurationException(NodeMessage.JEMC010E.toMessage().getFormattedMessage(Main.EXECUTION_ENVIRONMENT.getEnvironment(), group));
			}

			// check if the local member is the first of member list. if yes, it's
			// the coordinator
			// remember that Hazelcast maintains the list the "cluster joining"
			// order
			Member local = cluster.getLocalMember();
			Member first = null;

			for (Member mem : cluster.getMembers()) {
				if (!mem.isLiteMember() && (first == null)) {
					first = mem;
				}
			}
			cluster.getMembers().iterator().next();
			// checks if is coordinator
			if (local.equals(first)) {

				Main.IS_COORDINATOR = true;

				// if there are some job listeners, then says it
				if (Main.JOB_LIFECYCLE_LISTENERS_SYSTEM.hasListeners()) {
					LogAppl.getInstance().emit(NodeMessage.JEMC034I);
				}

			} else {
				// if local member is not the first, so it's not coordinator of
				// cluster
				Main.IS_COORDINATOR = false;

				Key key;
				try {
					key = KeysUtil.getSymmetricKey();
				} catch (Exception e) {
					throw new ConfigurationException(e);
				}
				
				ResourcesUtil.getInstance().setKey(key);
			}
			checkIfEnoughMembers();
			// bring in memory the persisted queue
			loadQueues();
		} finally {
			lock.unlock();
		}
	}

	/**
	 * If NODES_MAP contain at least a node with status different from STARTING
	 * will return otherwise checks if the cluster have enough nodes to support
	 * the persisted queue in memory. If the nodes are not enough the cluster
	 * will wait for new joining nodes.
	 * 
	 * @throws ConfigurationException
	 */
	private static void checkIfEnoughMembers() throws ConfigurationException {
		// by default the number of permits are 0
		ISemaphore semaphore = Main.HAZELCAST.getSemaphore(SEMAPHORE);
		// check if exists a node of the cluster with status different from
		// STARTING if so return.
		List<Status> statusList = new ArrayList<Status>();
		statusList.add(Status.STARTING);
		List<NodeInfo> nodesInfo = NodeInfoUtility.getNodesInfoByStatus(statusList, true, false);
		if (!nodesInfo.isEmpty()) {
			return;
		}

		// times 2 because in memory there will be a replication copy
		long queueSize = calculateQueueSize() * 2;
		// calculate the number of nodes (exclude light member)
		Cluster cluster = Main.HAZELCAST.getCluster();
		int membersNumber = cluster.getMembers().size();
		for (Member member : cluster.getMembers()) {
			if (member.isLiteMember())
				membersNumber--;
		}
		MemoryMXBean bean = ManagementFactory.getMemoryMXBean();
		long freeMemoryForNode = bean.getHeapMemoryUsage().getMax() - bean.getHeapMemoryUsage().getUsed();
		// we consider each has at this point the same max and used memory
		// or we could insert this information in nodeInfo ?
		long clusterFreMemory = freeMemoryForNode * membersNumber;
		// we consider clusterFreMemory enough if is grather than the queueSize
		// + 20%
		long neededMemory = queueSize + (queueSize / 10) * 2;
		if (clusterFreMemory > neededMemory) {
			semaphore.release(membersNumber - 1);
			return;
		} else {
			LogAppl.getInstance().emit(NodeMessage.JEMC086W, clusterFreMemory / 1000, neededMemory / 1000);
			try {
				semaphore.acquire();
			} catch (Exception e) {
				throw new ConfigurationException(e);
			}
		}
	}

	/**
	 * 
	 * @return the size in byte of the persisted maps.
	 * @throws ConfigurationException
	 */
	private static long calculateQueueSize() throws ConfigurationException {
		try {
			long inputQueueSize = JobDBManager.getInstance().getInputQueueJobSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.INPUT_QUEUE, inputQueueSize / 1000);
			long runningQueueSize = JobDBManager.getInstance().getRunningQueueJobSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.RUNNING_QUEUE, runningQueueSize / 1000);
			
			long outputQueueSize = JobDBManager.getInstance().getOutputQueueJobSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.OUTPUT_QUEUE, outputQueueSize / 1000);
			long routingQueueSize = JobDBManager.getInstance().getRoutingQueueJobSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.ROUTING_QUEUE, routingQueueSize / 1000);
			long rolesSize = RolesDBManager.getInstance().getRolesSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.ROLES_MAP, rolesSize / 1000);
			long resourcesSize = CommonResourcesDBManager.getInstance().getCommonResourcesSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.COMMON_RESOURCES_MAP, resourcesSize / 1000);
			long checkingQueueSize = PreJobDBManager.getInstance().getPreJobsCheckingQueueJobSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.JCL_CHECKING_QUEUE, checkingQueueSize / 1000);
			long routingConfSize = RoutingConfigDBManager.getInstance().getRoutingConfigSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.ROUTING_CONFIG_MAP, routingConfSize / 1000);
			long userPrefSize = UserPreferencesDBManager.getInstance().getUserPreferencesSize();
			LogAppl.getInstance().emit(NodeMessage.JEMC085I, Queues.USER_PREFERENCES_MAP, userPrefSize / 1000);
			
			long totlaSize = inputQueueSize + runningQueueSize + outputQueueSize + routingQueueSize + rolesSize + resourcesSize + checkingQueueSize + routingConfSize;
			return totlaSize;
		} catch (Exception e) {
			throw new ConfigurationException(e);
		}
	}

	/**
	 * Load the persisted queues by calling them for the first time.
	 * 
	 * @throws ConfigurationException
	 */
	private static void loadQueues() throws ConfigurationException {
		// loads all keys for INPUT, RUNNING, OUTPUT, ROUTING and CommonResources. this
		// call of method
		// will schedule the call on persistent manager
		IMap<String, Job> inputQueue = Main.HAZELCAST.getMap(Queues.INPUT_QUEUE);
		inputQueue.size();
		
		IMap<String, Job> outputQueue = Main.HAZELCAST.getMap(Queues.OUTPUT_QUEUE);
		outputQueue.size();
		
		// this code removes all jobs in running queue if the node is coordinator
		// this is necessary if the cluster crashed due to a failure
		if (Main.IS_COORDINATOR){
			IMap<String, Job> runningQueue = Main.HAZELCAST.getMap(Queues.RUNNING_QUEUE);
			// TODO Mettere READ/WRITE locks se server
			try{
				runningQueue.lockMap(10, TimeUnit.SECONDS);
				if (!runningQueue.isEmpty()){
					for (Job job : runningQueue.values()){
						org.pepstock.jem.Result result = new org.pepstock.jem.Result();
						result.setReturnCode(org.pepstock.jem.Result.FATAL);
						result.setExceptionMessage("Node is crashed during job was executing");
						job.setResult(result);
						job.setEndedTime(new Date());
						job.setRunningStatus(Job.NONE);
						LogAppl.getInstance().emit(NodeMessage.JEMC190W, job.getName(), job.getId());
						runningQueue.remove(job.getId());
						outputQueue.putIfAbsent(job.getId(), job);
					}
				}
			} finally  {
				runningQueue.unlockMap();
			}
		}

		IMap<String, Job> routingQueue = Main.HAZELCAST.getMap(Queues.ROUTING_QUEUE);
		routingQueue.size();

		RecoveryManager.getInstance();
		
		IMap<String, Resource> resourceMap = Main.HAZELCAST.getMap(Queues.COMMON_RESOURCES_MAP);
		resourceMap.size();
		// per calcolare la dimensione degli oggetti job
		// per attivarlo mettere nei parametri della JVM:
		// -javaagent:./lib/classmexer/classmexer.jar
		//
		// for (Resource role : resourceMap.values()){
		// long roleSize = MemoryUtil.deepMemoryUsageOf(role);
		// System.out.println(role.getName()+": "+roleSize);
		// }

		IMap<String, Role> rolesMap = Main.HAZELCAST.getMap(Queues.ROLES_MAP);
		rolesMap.size();

		IMap<String, SwarmConfiguration> routingConfigMap = Main.HAZELCAST.getMap(Queues.ROUTING_CONFIG_MAP);
		routingConfigMap.size();
		
		// if map is emtpy, a DEFAULT routing config will be added
		if (routingConfigMap.isEmpty()){
			routingConfigMap.put(SwarmConfiguration.DEFAULT_NAME, new SwarmConfiguration());
		}
		
		IMap<String, HashMap<String, UserPreference>> userPreferencesMap = Main.HAZELCAST.getMap(Queues.USER_PREFERENCES_MAP);
		userPreferencesMap.size();
		
	}

	/**
	 * loads the environment configuration from configuration file, present in the gfs,
	 * which is defined by a system property.
	 * 
	 * @throws ConfigurationException
	 */
	private static void loadEnvConfiguration() throws ConfigurationException {
		// loads configuration file from node folder.
		// if doesn't exist, exception
		String configFile = System.getProperty(ConfigKeys.JEM_ENV_CONF);
		if (configFile == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC005E, ConfigKeys.JEM_ENV_CONF);
			throw new ConfigurationException(NodeMessage.JEMC005E.toMessage().getFormattedMessage(ConfigKeys.JEM_CONFIG));
		}
		// load Xstream and set alias to have a configuration tags user friendly
		XStream xstream = new XStream();

		xstream.alias(ConfigKeys.CONFIGURATION_TAG, Configuration.class);
		
		xstream.aliasField(ConfigKeys.STATISTICS_MANAGER_ALIAS, Configuration.class, ConfigKeys.STATISTICS_MANAGER_FIELD);
		xstream.alias(ConfigKeys.NODE_ALIAS, Node.class);
		xstream.alias(ConfigKeys.DATABASE_ELEMENT, Database.class);

		xstream.alias(ConfigKeys.FACTORY_ALIAS, Factory.class);
		xstream.aliasAttribute(Factory.class, ConfigKeys.CLASS_NAME_FIELD, ConfigKeys.CLASS_NAME_ATTRIBUTE_ALIAS);

		xstream.alias(ConfigKeys.LISTENER_ALIAS, Listener.class);
		xstream.aliasAttribute(Listener.class, ConfigKeys.CLASS_NAME_FIELD, ConfigKeys.CLASS_NAME_ATTRIBUTE_ALIAS);
		
		xstream.aliasAttribute(StatsManager.class, ConfigKeys.PATH_FIELD, ConfigKeys.PATH_ATTRIBUTE_ALIAS);
		xstream.aliasAttribute(StatsManager.class, ConfigKeys.ENABLE_FIELD, ConfigKeys.ENABLE_ATTRIBUTE_ALIAS);

		xstream.alias(ConfigKeys.RESOURCE_DEFINITION_ALIAS, CustomResourceDefinition.class);
		xstream.aliasAttribute(CustomResourceDefinition.class, ConfigKeys.CLASS_NAME_FIELD, ConfigKeys.CLASS_NAME_ATTRIBUTE_ALIAS);
		xstream.aliasField(ConfigKeys.RESOURCE_DEFINITIONS_ALIAS, Configuration.class, ConfigKeys.RESOURCE_DEFINITIONS_FIELD);		

		// load configuration fle by XStream. if object is not a instance of
		// Configuration, exception occurs
		Object object;
		try {
			object = xstream.fromXML(new FileInputStream(configFile));
		} catch (Exception e1) {
			LogAppl.getInstance().emit(NodeMessage.JEMC006E);
			throw new ConfigurationException(NodeMessage.JEMC006E.toMessage().getMessage(), e1);
		}
		if (!(object instanceof Configuration)) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.CONFIGURATION_TAG);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.CONFIGURATION_TAG));
		}
		// configuration loaded
		Configuration conf = (Configuration) object;
		LogAppl.getInstance().emit(NodeMessage.JEMC008I, configFile);

		loadDatabaseManagers(conf);
		loadNode(conf);
		loadFactories(conf);
		loadListeners(conf);
		loadStatisticsManager(conf);
		loadResourceConfigurations(conf);

	}

	/**
	 * loads the configuration from configuration file, which is defined by a
	 * system property.
	 * 
	 * @see org.pepstock.jem.node.configuration.ConfigKeys#JEM_CONFIG
	 * @see org.pepstock.jem.node.Main#EXECUTION_ENVIRONMENT
	 * @see org.pepstock.jem.node.Main#OUTPUT_SYSTEM
	 * @throws ConfigurationException if a configuration error, exception occurs
	 */
	private static void loadConfiguration() throws ConfigurationException {
		// we set this environment variable as system properties 
		handleEnvironmentVariable(ConfigKeys.JEM_NODE, ConfigKeys.JEM_HOME, ConfigKeys.JEM_ENVIRONMENT);
		LogAppl.getInstance().emit(NodeMessage.JEMC192I, System.getProperty(ConfigKeys.JAVA_VERSION), System.getProperty(ConfigKeys.JAVA_HOME), System.getProperty(ConfigKeys.JAVA_VENDOR));
		LogAppl.getInstance().emit(NodeMessage.JEMC001I);
		// loads JVM properties to Properties instance
		// to be able to substitute variables in configuration
		PROPERTIES.putAll(System.getProperties());
		PROPERTIES.putAll(System.getenv());
		
		// loads configuration file from node folder.
		// if doesn't exist, exception
		String configFile = System.getProperty(ConfigKeys.JEM_CONFIG);
		if (configFile == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC005E, ConfigKeys.JEM_CONFIG);
			throw new ConfigurationException(NodeMessage.JEMC005E.toMessage().getFormattedMessage(ConfigKeys.JEM_CONFIG));
		}

		// load Xstream and set alias to have a configuration tags user friendly
		XStream xstream = new XStream();

		xstream.alias(ConfigKeys.CONFIGURATION_TAG, Configuration.class);
		xstream.aliasField(ConfigKeys.EXECUTION_ENVIRONMENT_ALIAS, Configuration.class, ConfigKeys.EXECUTION_ENVIRONMENT_FIELD);
		xstream.alias(ConfigKeys.AFFINITY_FACTORY_ALIAS, AffinityFactory.class);
		xstream.alias(ConfigKeys.FACTORY_ALIAS, Factory.class);
		xstream.aliasAttribute(Factory.class, ConfigKeys.CLASS_NAME_FIELD, ConfigKeys.CLASS_NAME_ATTRIBUTE_ALIAS);
		
		// load configuration fle by XStream. if object is not a instance of
		// Configuration, exception occurs
		Object object;
		try {
			object = xstream.fromXML(new FileInputStream(configFile));
		} catch (Exception e1) {
			LogAppl.getInstance().emit(NodeMessage.JEMC006E);
			throw new ConfigurationException(NodeMessage.JEMC006E.toMessage().getMessage(), e1);
		}
		if (!(object instanceof Configuration)) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.CONFIGURATION_TAG);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.CONFIGURATION_TAG));
		}
		// configuration loaded
		Configuration conf = (Configuration) object;
		LogAppl.getInstance().emit(NodeMessage.JEMC008I, configFile);

		loadPaths(conf);
		PROPERTIES.putAll(System.getProperties());
		loadExecutionEnvironment(conf);		
	}

	/**
	 * load the paths of the jem configuration checking if they are configured
	 * 
	 * @param conf the jem node config
	 * @param substituter a VariableSubstituter containing all the System
	 *            properties
	 * @throws ConfigurationException if some error occurs
	 */
	private static void loadPaths(Configuration conf) throws ConfigurationException {
		// load paths, checking if they are configured. If not, exception occurs
		Paths paths = conf.getPaths();
		if (paths == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.PATHS_ELEMENT);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.PATHS_ELEMENT));
		}

		// load Paths
		String dataPath = paths.getData();
		String outputPath = paths.getOutput();
		String binaryPath = paths.getBinary();
		String classpathPath = paths.getClasspath();
		String libraryPath = paths.getLibrary();
		String persistencePath = paths.getPersistence();
		String sourcetPath = paths.getSource();

		// check if outputPath is not null. if not, exception occurs
		if (outputPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_OUTPUT_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_OUTPUT_PATH_NAME));
		}
		// check if dataPath is not null. if not, exception occurs
		if (dataPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_DATA_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_DATA_PATH_NAME));
		}
		// check if binaryPath is not null. if not, exception occurs
		if (binaryPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_BINARY_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_BINARY_PATH_NAME));
		}
		// check if classpathPath is not null. if not, exception occurs
		if (classpathPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_CLASSPATH_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_CLASSPATH_PATH_NAME));
		}
		// check if libraryPath is not null. if not, exception occurs
		if (libraryPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_LIBRARY_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_LIBRARY_PATH_NAME));
		}
		// check if persistencePath is not null. if not, exception occurs
		if (persistencePath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_PERSISTENCE_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_PERSISTENCE_PATH_NAME));
		}
		// check if persistencePath is not null. if not, exception occurs
		if (sourcetPath == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC039E, ConfigKeys.JEM_SOURCE_PATH_NAME);
			throw new ConfigurationException(NodeMessage.JEMC039E.toMessage().getFormattedMessage(ConfigKeys.JEM_SOURCE_PATH_NAME));
		}

		// substitutes variables if present, on data path
		// adds this value in properties of JVM
		dataPath = substituteVariable(dataPath);
		System.getProperties().setProperty(ConfigKeys.JEM_DATA_PATH_NAME, normalizePath(dataPath));

		// substitutes variables if present, on output path
		// adds this value in properties of JVM
		outputPath = substituteVariable(outputPath);
		System.getProperties().setProperty(ConfigKeys.JEM_OUTPUT_PATH_NAME, normalizePath(outputPath));

		// substitutes variables if present, on binary Path
		// adds this value in properties of JVM
		binaryPath = substituteVariable(binaryPath);
		System.getProperties().setProperty(ConfigKeys.JEM_BINARY_PATH_NAME, normalizePath(binaryPath));
		// substitutes variables if present, on classpathPath
		// adds this value in properties of JVM
		classpathPath = substituteVariable(classpathPath);
		System.getProperties().setProperty(ConfigKeys.JEM_CLASSPATH_PATH_NAME, normalizePath(classpathPath));

		// substitutes variables if present, on libraryPath
		// adds this value in properties of JVM, substituter);
		System.getProperties().setProperty(ConfigKeys.JEM_LIBRARY_PATH_NAME, normalizePath(libraryPath));

		// substitutes variables if present, on libraryPath
		// adds this value in properties of JVM
		persistencePath = substituteVariable(persistencePath);
		System.getProperties().setProperty(ConfigKeys.JEM_PERSISTENCE_PATH_NAME, normalizePath(persistencePath));

		// substitutes variables if present, on libraryPath
		// adds this value in properties of JVM
		sourcetPath = substituteVariable(sourcetPath);
		System.getProperties().setProperty(ConfigKeys.JEM_SOURCE_PATH_NAME, normalizePath(sourcetPath));

		// load ths path values on a static reference on Main class which
		// creates all necessary files and directories
		Main.OUTPUT_SYSTEM = new OutputSystem(outputPath, dataPath, persistencePath);
		
	}

	/**
	 * 
	 * @param string the string that may contains the variable to substitute
	 *            with the values contained in the substituter
	 * @return the value of the param string eventually substituted
	 */
	private static String substituteVariable(String string) {
		return VariableSubstituter.substitute(string, PROPERTIES);
	}

	/**
	 * load the listeners of the jem configuration checking if they are
	 * configured
	 * 
	 * @param conf the jem node config
	 * @param substituter a VariableSubstituter containing all the System
	 *            properties
	 * @throws ConfigurationException if some error occurs
	 */
	private static void loadListeners(Configuration conf) throws ConfigurationException {
		// load listeners, checking if they are configured. If not, they are
		// optional, so go ahead
		ArrayList<Listener> listeners = conf.getListeners();
		if (listeners != null) {
			if (!listeners.isEmpty()) {
				// for all listener checking which have the right className. If
				// not, execption occurs, otherwise it's loaded
				for (Listener listener : listeners) {
					if (listener.getClassName() != null) {
						String className = listener.getClassName();
						try {
							// load by Class.forName of listener
							Object objectListener = Class.forName(className).newInstance();

							// check if it's a JobLifecycleListener. if not,
							// exception occurs. if yes, it's loaded on a
							// EventListenerManager
							if (objectListener instanceof JobLifecycleListener) {
								JobLifecycleListener lister = (JobLifecycleListener) objectListener;
								Main.JOB_LIFECYCLE_LISTENERS_SYSTEM.addListener(JobLifecycleListener.class, lister);
								// gets properties defined. If not empty,
								// substitutes the value of property with
								// variables
								Properties propsOfListener = listener.getProperties();
								if (propsOfListener != null){
								if (!propsOfListener.isEmpty()) {
									// scans all properties
									for (Enumeration<Object> e = propsOfListener.keys(); e.hasMoreElements();) {
										// gets key and value
										String key = e.nextElement().toString();
										String value = propsOfListener.getProperty(key);
										// substitutes variables if present
										// and sets new value for the key
										propsOfListener.setProperty(key, substituteVariable(value));
									}
								}
								} else {
									// creates however an empty collection to avoid null pointer
									propsOfListener = new Properties();
								}
								// initialize the listener passing parameters
								// properties
								lister.init(propsOfListener);
								LogAppl.getInstance().emit(NodeMessage.JEMC037I, className);
							} else {
								LogAppl.getInstance().emit(NodeMessage.JEMC036E, className);
								throw new ConfigurationException(NodeMessage.JEMC036E.toMessage().getFormattedMessage(className));
							}
						} catch (ClassNotFoundException e) {
							LogAppl.getInstance().emit(NodeMessage.JEMC031E, e, className);
							throw new ConfigurationException(NodeMessage.JEMC031E.toMessage().getFormattedMessage(className));
						} catch (InstantiationException e) {
							LogAppl.getInstance().emit(NodeMessage.JEMC031E, e, className);
							throw new ConfigurationException(NodeMessage.JEMC031E.toMessage().getFormattedMessage(className));
						} catch (IllegalAccessException e) {
							LogAppl.getInstance().emit(NodeMessage.JEMC031E, e, className);
							throw new ConfigurationException(NodeMessage.JEMC031E.toMessage().getFormattedMessage(className));
						}
						// in this case the class name is null so ignore,
						// emitting a warning
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC038W, ConfigKeys.LISTENER_ALIAS, listener.toString());
					}
				}
			}
		}
	}

	
	/**
	 * Load the custom resource definitions of the jem configuration checking if they are
	 * configured
	 * 
	 * @param conf the jem node config
	 * @param substituter a VariableSubstituter containing all the System
	 *            properties
	 * @throws ConfigurationException if some error occurs.
	 */
	private static void loadResourceConfigurations(Configuration conf) throws ConfigurationException {
		// load custom resource definitions, checking if they are configured. If not, they are
		// optional, so go ahead
		ArrayList<CustomResourceDefinition> resourceDefinitions = conf.getResourceDefinitions();
		if (resourceDefinitions != null) {
			if (!resourceDefinitions.isEmpty()) {
				// for all resource definitions checking which have the right className. If
				// not, exception occurs, otherwise it's loaded
				for (CustomResourceDefinition resourceDefinition : resourceDefinitions) {
					if (resourceDefinition.getClassName() != null) {
						Properties propsOfListener = resourceDefinition.getProperties();
						String xmlResourceTemplateFile = null;
						if (propsOfListener != null){
							xmlResourceTemplateFile = propsOfListener.getProperty(ResourceDefinitionsManager.XML_RESOURCE_TEMPLATE_FILE_PROPERTY);
							if(null != xmlResourceTemplateFile)
								xmlResourceTemplateFile = substituteVariable(xmlResourceTemplateFile);
						}
						try{
							Main.CUSTOM_RESOURCE_DEFINITION_MANAGER.loadCustomResourceDefinition(resourceDefinition, xmlResourceTemplateFile);						
						} catch (ResourceDefinitionException e) {
							throw new ConfigurationException(e);
						}
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC038W, ConfigKeys.RESOURCE_DEFINITION_ALIAS, resourceDefinition.toString());
					}
				}
			}
		}
	}
	
	/**
	 * This method clean all common resources removing all resources
	 * whose type ({@link Resource#getType()}) is not configured or is not one of
	 * the predefined resources.
	 * 
	 * @throws ConfigurationException  if some error occurs.
	 * 
	 * @see #isValidResourceType(String)
	 * @see Resource
	 */
	private static void deleteNotExistingTypesResources() throws ConfigurationException{
		try{
			Main.CUSTOM_RESOURCE_DEFINITION_MANAGER.deleteNotExistingTypesResources();
		} catch (ResourceDefinitionException e) {
			throw new ConfigurationException(e);
		}
	}

	
	/**
	 * This method clean all common resources removing all resources
	 * whose type ({@link Resource#getType()}) is not configured or is not one of
	 * the predefined resources.
	 * 
	 * @throws ConfigurationException  if some error occurs.
	 * 
	 * @see #isValidResourceType(String)
	 * @see Resource
	 */
//	private static void deleteNotExistingTypesResources_() throws ConfigurationException{
//		IMap<String, Resource> resourceMap = Main.HAZELCAST.getMap(Queues.COMMON_RESOURCES_MAP);
//		for(Resource resource:resourceMap.values()){
//			if(!isValidResourceType(resource.getType())){
//				String name = resource.getName();
//				try {
//					// locks the key 
//					resourceMap.lock(name);
//					// remove the resource from map 
//					resource = resourceMap.remove(name);
//					LogAppl.getInstance().emit(ResourceMessage.JEMR019W, name, resource.getType());
//				} finally {
//					// unlocks always the key
//					resourceMap.unlock(name);
//				}
//			}
//		}
//	}

	/**
	 * It checks if the resource type is configured, or is
	 * one of the the predefined resources:
	 * <li>{@link FtpResource}
	 * <li>{@link HttpResource} 
	 * <li>{@link JdbcResource} 
	 * <li>{@link JmsResource} 
	 * <li>{@link JppfResource}
	 *  
	 * @param resourceType the resource type to be checked.
	 * @return <code>true</code> if the resource type is configured, or is
	 * one of the the predefined resources, <code>false</code> otherwise.
	 * @throws ConfigurationException if some error occurs.
	 */
//	private static boolean isValidResourceType(String resourceType) throws ConfigurationException{
//		try{
//			if(		resourceType.equalsIgnoreCase(FtpResource.TYPE) || 
//					resourceType.equalsIgnoreCase(HttpResource.TYPE) || 				
//					resourceType.equalsIgnoreCase(JdbcResource.TYPE) || 
//					resourceType.equalsIgnoreCase(JmsResource.TYPE) || 
//					resourceType.equalsIgnoreCase(JppfResource.TYPE) || 
//					Main.CUSTOM_RESOURCE_DEFINITION_MANAGER.hasCustomResourceDefinition(resourceType))
//				return true;
//			return false;
//		} catch (ResourceDefinitionException e) {
//			throw new ConfigurationException(e);
//		}
//	}
	
	/**
	 * 
	 * @param conf
	 * @throws Exception
	 */
	private static void loadDatabaseManagers(Configuration conf) throws ConfigurationException {
		Database database = conf.getDatabase();
		if (database == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.DATABASE_ELEMENT);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.DATABASE_ELEMENT));
		}
		// try to substitute vars in URL
		String url = substituteVariable(database.getUrl());
		database.setUrl(url);
		LogAppl.getInstance().emit(NodeMessage.JEMC193I, url);

		// try to substitute vars in user
		String user = substituteVariable(database.getUser());
		database.setUser(user);

		String dbType = null;
		try {
			URI url1 = new URI(database.getUrl());
			URI myURL = new URI(url1.getSchemeSpecificPart());
			dbType = myURL.getScheme();
		} catch (URISyntaxException e2) {
			LogAppl.getInstance().emit(NodeMessage.JEMC166E, e2, database.getUrl());
			throw new ConfigurationException(NodeMessage.JEMC166E.toMessage().getFormattedMessage(database.getUrl()));
		}

		SQLContainerFactory engine = null;
		if (dbType.equals(MySqlSQLContainerFactory.DATABASE_TYPE)) {
			engine = new MySqlSQLContainerFactory();
		} else {
			engine = new DefaultSQLContainerFactory();
		}

		// load JobManager for input, output, routing

		try {
			DBPoolManager.getInstance().setDriver(database.getDriver());
			DBPoolManager.getInstance().setUrl(database.getUrl());
			DBPoolManager.getInstance().setUser(database.getUser());
			DBPoolManager.getInstance().setPassword(database.getPassword());
			DBPoolManager.getInstance().setProperties(database.getProperties());
			DBPoolManager.getInstance().setKeepAliveConnectionSQL(engine.getKeepAliveConnectionSQL());

			DBPoolManager.getInstance().init();
			
			
			JobDBManager.getInstance().setInputSqlContainer(engine.getSQLContainerForInputQueue());
			JobDBManager.getInstance().setRunningSqlContainer(engine.getSQLContainerForRunningQueue());
			JobDBManager.getInstance().setOutputSqlContainer(engine.getSQLContainerForOutputQueue());
			JobDBManager.getInstance().setRoutingSqlContainer(engine.getSQLContainerForRoutingQueue());
			
			PreJobDBManager.getInstance().setCheckingSqlContainer(engine.getSQLContainerForCheckingQueue());

			RolesDBManager.getInstance().setRolesSqlContainer(engine.getSQLContainerForRolesMap());

			CommonResourcesDBManager.getInstance().setCommonResourcesSqlContainer(engine.getSQLContainerForCommonResourcesMap());
			
			RoutingConfigDBManager.getInstance().setRoutingSqlContainer(engine.getSQLContainerForRoutingConfigMap());
			
			UserPreferencesDBManager.getInstance().setPreferencesSqlContainer(engine.getSQLContainerForUserPreferencesMap());
			
			Connection conn = DBPoolManager.getInstance().getConnection();
			try {
				DatabaseMetaData md = conn.getMetaData();
				checkAndCreateTable(md, JobDBManager.getInstance().getInputSqlContainer());
				checkAndCreateTable(md, JobDBManager.getInstance().getRunningSqlContainer());
				checkAndCreateTable(md, JobDBManager.getInstance().getOutputSqlContainer());
				checkAndCreateTable(md, JobDBManager.getInstance().getRoutingSqlContainer());
				
				checkAndCreateTable(md, PreJobDBManager.getInstance().getCheckingSqlContainer());

				checkAndCreateTable(md, RolesDBManager.getInstance().getRolesSqlContainer());

				checkAndCreateTable(md, CommonResourcesDBManager.getInstance().getCommonResourcesSqlContainer());
				
				checkAndCreateTable(md, RoutingConfigDBManager.getInstance().getRoutingSqlContainer());
				
				checkAndCreateTable(md, UserPreferencesDBManager.getInstance().getPreferencesSqlContainer());
				
			} catch (SQLException e1) {
				LogAppl.getInstance().emit(NodeMessage.JEMC167E, e1);
				throw new ConfigurationException(NodeMessage.JEMC167E.toMessage().getFormattedMessage());
			} finally {
				if (conn != null){
					conn.close();
				}
			}
			
		} catch (SQLException e) {
			throw new ConfigurationException(NodeMessage.JEMC165E.toMessage().getFormattedMessage(JobDBManager.class.getName()), e);
		}

	}

	/**
	 * 
	 * @param md
	 * @param dbmanager
	 * @param tableName
	 * @param createStmt
	 * @throws SQLException
	 */
	private static void checkAndCreateTable(DatabaseMetaData md, SQLContainer container) throws SQLException {
		ResultSet rs = md.getTables(null, null, container.getTableName(), new String[] { "TABLE" });
		if (!rs.next()) {
			// creates table and return a empty set because if empty of
			// course
			DBPoolManager.getInstance().create(container.getCreateTableStatement());
		}
	}

	/**
	 * load the factories of the jem configuration checking if they are
	 * configured
	 * 
	 * @param conf the jem node config
	 * @param substituter a VariableSubstituter containing all the System
	 *            properties
	 * @throws ConfigurationException if some error occurs
	 */
	private static void loadFactories(Configuration conf) throws ConfigurationException {
		// load factories, checking if they are configured. If not, exception
		// occurs
		ArrayList<Factory> factories = conf.getFactories();
		if (factories == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.FACTORIES_ELEMENT);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.FACTORIES_ELEMENT));
		}
		if (factories.isEmpty()) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.FACTORY_ALIAS);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.FACTORY_ALIAS));
		}

		// for all factories checking which have the right className. If not,
		// execption occurs, otherwise it's loaded
		for (Factory factory : factories) {
			if (factory.getClassName() != null) {
				String className = factory.getClassName();
				try {
					// load by Class.forName of factory
					Object objectFactory = Class.forName(className).newInstance();

					// check if it's a JemFactory. if not, exception occurs. if
					// yes, it's loaded on a map
					// with all factory. Remember the the key of map is getType
					// result, put to lowercase to ignore case
					// during the search by key
					if (objectFactory instanceof JemFactory) {
						JemFactory jf = (JemFactory) objectFactory;

						// gets properties defined. If not empty, substitutes
						// the value of property with variables
						Properties propsOfFactory = factory.getProperties();
						if (propsOfFactory != null){
							if (!propsOfFactory.isEmpty()) {
								// scans all properties
								for (Enumeration<Object> e = propsOfFactory.keys(); e.hasMoreElements();) {
									// gets key and value
									String key = e.nextElement().toString();
									String value = propsOfFactory.getProperty(key);
									// substitutes variables if present
									// and sets new value for the key
									propsOfFactory.setProperty(key, substituteVariable(value));
								}
							}
						} else {
							// creates however an emtpy collection to avoid null pointer
							propsOfFactory = new Properties();
						}
						// initializes the factory with properties defined
						// and puts in the list if everything went good
						jf.init(propsOfFactory);

						Main.FACTORIES_LIST.put(jf.getType().toLowerCase(), jf);

						LogAppl.getInstance().emit(NodeMessage.JEMC032I, className, jf.getType());
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC040E, className);
					}

				} catch (Exception e) {
					LogAppl.getInstance().emit(NodeMessage.JEMC031E, e, className);
				}
				// in this case the class name is null so ignore, emitting a
				// warning
			} else {
				LogAppl.getInstance().emit(NodeMessage.JEMC038W, ConfigKeys.FACTORY_ALIAS, factory.toString());
			}
		}
	}

	/**
	 * load the execution-environment of the jem configuration
	 * 
	 * @param conf the jem node config
	 * @param substituter a VariableSubstituter containing all the System
	 *            properties
	 * @throws ConfigurationException if some error occurs
	 */
	private static void loadExecutionEnvironment(Configuration conf) throws ConfigurationException {
		// checks if ExecutionEnvironment is configured
		if (conf.getExecutionEnviroment() == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.EXECUTION_ENVIRONMENT_ALIAS);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.EXECUTION_ENVIRONMENT_ALIAS));
		}

		// environment is mandatory. must be not null
		if (conf.getExecutionEnviroment().getEnvironment() == null) {
			LogAppl.getInstance().emit(NodeMessage.JEMC009E, ConfigKeys.ENVIRONMENT);
			throw new ConfigurationException(NodeMessage.JEMC009E.toMessage().getFormattedMessage(ConfigKeys.ENVIRONMENT));
		}

		// domain is optional. if null set default value
		if (conf.getExecutionEnviroment().getDomain() == null) {
			conf.getExecutionEnviroment().setDomain(Jcl.DEFAULT_DOMAIN);
		}

		// affinity is optional. if null, set a defualt value
		if (conf.getExecutionEnviroment().getAffinity() == null) {
			conf.getExecutionEnviroment().setAffinity(Jcl.DEFAULT_AFFINITY);
		}

		// gets environment and substitute variables if present
		String environment = conf.getExecutionEnviroment().getEnvironment();
		conf.getExecutionEnviroment().setEnvironment(substituteVariable(environment));
		// loads environment on shared object
		Main.EXECUTION_ENVIRONMENT.setEnvironment(conf.getExecutionEnviroment().getEnvironment());

		// gets domain and substitute variables if present
		String domain = conf.getExecutionEnviroment().getDomain();
		conf.getExecutionEnviroment().setDomain(substituteVariable(domain));
		// loads domain on shared object
		Main.EXECUTION_ENVIRONMENT.setDomain(conf.getExecutionEnviroment().getDomain());

		// gets affinity and substitute variables if present
		String affinity = conf.getExecutionEnviroment().getAffinity();
		conf.getExecutionEnviroment().setAffinity(substituteVariable(affinity));
		// loads affinities on shared object
		String[] affinities = conf.getExecutionEnviroment().getAffinity().split(",");
		for (int i = 0; i < affinities.length; i++) {
			Main.EXECUTION_ENVIRONMENT.getStaticAffinities().add(affinities[i].trim().toLowerCase());
		}

		// load factories, checking if they are configured. If not, exception
		// occurs
		AffinityFactory affinityFactory = conf.getExecutionEnviroment().getAffinityFactory();
		if (affinityFactory != null) {

			// load all factories for affinity factory
			// for all factories checking which have the right className. If
			// not,
			// exception occurs, otherwise it's loaded
			if (affinityFactory.getClassName() != null) {
				String className = affinityFactory.getClassName();
				try {
					// load by Class.forName of loader
					Object objectFactory = Class.forName(className).newInstance();

					// check if it's a AffinityLoader. if not, exception occurs.
					if (objectFactory instanceof AffinityLoader) {
						AffinityLoader loader = (AffinityLoader) objectFactory;

						// gets properties defined. If not empty, substitutes
						// the value of property with variables
						Properties propsOfFactory = affinityFactory.getProperties();
						if (!propsOfFactory.isEmpty()) {
							// scans all properties
							for (Enumeration<Object> e = propsOfFactory.keys(); e.hasMoreElements();) {
								// gets key and value
								String key = e.nextElement().toString();
								String value = propsOfFactory.getProperty(key);
								// substitutes variables if present
								// and sets new value for the key
								propsOfFactory.setProperty(key, substituteVariable(value));
							}
						}
						LogAppl.getInstance().emit(NodeMessage.JEMC049I, className);
						// initializes the factory with properties defined
						// and puts in the list if everything went good
						loader.init(propsOfFactory);
						Result result = loader.load(new SystemInfo());
						if (result != null) {
							Main.EXECUTION_ENVIRONMENT.getDynamicAffinities().addAll(result.getAffinities());
							Main.EXECUTION_ENVIRONMENT.setMemory(result.getMemory());
						}
						Main.AFFINITY_LOADER = loader;
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC089E, className);
					}

				} catch (Exception e) {
					LogAppl.getInstance().emit(NodeMessage.JEMC031E, e, className);
				}
				// in this case the class name is null so ignore, emitting a
				// warning
			} else {
				LogAppl.getInstance().emit(NodeMessage.JEMC038W, ConfigKeys.FACTORY_ALIAS, affinityFactory.toString());
			}
		}
		LogAppl.getInstance().emit(NodeMessage.JEMC050I, Main.EXECUTION_ENVIRONMENT);
	}

	/**
	 * 
	 * @param conf
	 * @param substituter
	 * @throws ConfigurationException
	 */
	private static void loadStatisticsManager(Configuration conf) throws ConfigurationException {
		StatsManager statsManager = conf.getStatsManager();
		if (statsManager != null) {
			if (!statsManager.isEnable()){
				Main.STATISTICS_MANAGER = new StatisticsManager(false);
			} else {
				// gets path and substitute variables if present
				String path = statsManager.getPath();
				if (path != null)
					path = substituteVariable(path);
				Main.STATISTICS_MANAGER = new StatisticsManager(true, path);
			}
		} else {
			Main.STATISTICS_MANAGER = new StatisticsManager();
		}
	}
	
	/**
	 * 
	 * @param conf
	 * @param substituter
	 * @throws ConfigurationException
	 */
	private static void loadNode(Configuration conf) throws ConfigurationException {
		// load node class, checking if they are configured.
		Node node = conf.getNode();
		if (node != null) {

			// load all factories for affinity factory
			// for all factories checking which have the right className. If
			// not,
			// exception occurs, otherwise it's loaded
			if (node.getClassName() != null) {
				String className = node.getClassName();
				try {
					// load by Class.forName of loader
					Object objectNode = Class.forName(className).newInstance();

					// check if it's a AffinityLoader. if not, exception occurs.
					if (objectNode instanceof NodeInfo) {
						Main.NODE = (NodeInfo) objectNode;

						// gets properties defined. If not empty, substitutes
						// the value of property with variables
						Properties propsOfNode = node.getProperties();
						if (!propsOfNode.isEmpty()) {
							// scans all properties
							for (Enumeration<Object> e = propsOfNode.keys(); e.hasMoreElements();) {
								// gets key and value
								String key = e.nextElement().toString();
								String value = propsOfNode.getProperty(key);
								// substitutes variables if present
								// and sets new value for the key
								propsOfNode.setProperty(key, substituteVariable(value));
							}
						}
						// initializes the factory with properties defined
						// and puts in the list if everything went good
						Main.NODE.init(propsOfNode);
						LogAppl.getInstance().emit(NodeMessage.JEMC090I, className);
						return;
					}
					LogAppl.getInstance().emit(NodeMessage.JEMC091E);
					throw new ConfigurationException(NodeMessage.JEMC091E.toMessage().getFormattedMessage(className));
				} catch (Exception e) {
					throw new ConfigurationException(e);
				}
				// in this case the class name is null so ignore, emitting a
				// warning
			}
		}
		Main.NODE = new NodeInfo();
		LogAppl.getInstance().emit(NodeMessage.JEMC090I, NodeInfo.class.getName());
	}

	/**
	 * Jem uses 4 environment variables that are set when the node is launched: <br>
	 * 1. JEM_HOME is the root folder of the jem installation and is set in the
	 * OS system variable<br>
	 * 2. JEM_ENVIRONMENT is the root folder of the environment of the node<br>
	 * 3. JEM_NODE is the root folder of the node<br>
	 * 4. JEM_GFS is the root folder of the Jem Global File Syste<br>
	 * 
	 * @param variables lista on env variables to handle
	 * @throws ConfigurationException if some error occurs
	 */
	private static void handleEnvironmentVariable(String... variables) throws ConfigurationException {
		/*
		 * If the jem node is started with the YAJSW because of some bug of
		 * YAJSW all the environment variable name are set to lower case and
		 * JEM_ENVIRONMENT is going to be wrapper.app.env.jem_environment,
		 * JEM_NODE is going to be wrapper.app.env.jem_node (This because those
		 * variable are set in the wrapper.conf while JEM_HOME is set in the OS
		 * and it will be jem_home). So we fixed this problem here by code.
		 */

		// scans all mandatory env variables
		for (int i = 0; i < variables.length; i++) {
			// reads from enviroment
			String variable = System.getenv().get(variables[i]);
			// if not found it
			if (variable == null) {
				String key = null;
				if (variables[i].equals(ConfigKeys.JEM_HOME)) {
					// read jem_home (see above explaination)
					key = variables[i].toLowerCase();
				} else {
					// read using wrapper prefix (see above explaination)
					key = ConfigKeys.WRAPPER_APP_ENV + variables[i].toLowerCase();
				}
				// gets variable
				variable = System.getenv().get(key);
				// if not found it
				if (variable == null) {
					// writes log and throw an exception because teh variables
					// al mandatory
					LogAppl.getInstance().emit(NodeMessage.JEMC058E, variables[i]);
					throw new ConfigurationException(NodeMessage.JEMC058E.toMessage().getFormattedMessage(variables[i]));
				}

			}
			// sets the variable on system and writes log
			System.getProperties().setProperty(variables[i], normalizePath(variable));
			LogAppl.getInstance().emit(NodeMessage.JEMC057I, variables[i], variable);
		}
	}

	/**
	 * Normalize the path put in configuration, changing all separators
	 * 
	 * @param path path to normalize
	 * @return normalized path
	 */
	private static String normalizePath(String path) {
		File file = new File(path);
		// String normalized =
		// FilenameUtils.separatorsToSystem(file.getAbsolutePath());
		String normalized = FilenameUtils.normalize(file.getAbsolutePath(), true);
		return normalized;
	}
}
