/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.persistence.DBPoolManager;
import org.pepstock.jem.util.TimeUtils;

import com.hazelcast.core.ILock;

/**
 * Is the shutdown hook, called when SIGNAl is sent to JVM to close the JVM.<br>
 * Cleans up all structures like Hazelcast, database connections and job in execution. 
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class ShutDownHandler extends Thread {

	
	private JclCheckingQueueManager jclCheckManager = null;
	
	private Submitter submitter = null;

	/**
	 * Creates the handler with a list of threads to interrupt
	 * @param jclCheckManager 
	 * @param submitter 
	 * 
	 * @param threads threads to interrupt
	 * 
	 */
	public ShutDownHandler(JclCheckingQueueManager jclCheckManager, Submitter submitter) {
		super(ShutDownHandler.class.getName());
		this.jclCheckManager = jclCheckManager;
		this.submitter = submitter;
	}

	
	/**
	 * Main method, called by JVM to clean up.<br>
	 * In sequence :<br>
	 * <br>
	 * <ul>
	 * <ol>sets a static reference <code>Main.IS_SHUTTINGDOWN</code> to <code>true</code></ol>
	 * <ol>drains the initiator</ol>
	 * <ol>waits for initiator in DRAINED (if there is a job in execution, it waits)</ol>
	 * <ol>interrupts all threads passed by arguments in the constructor</ol>
	 * <ol>shuts down Hazelcast, removing itself from Hazelcast group</ol>
	 * <ol>Closes all databases connection (for Job, Roles, CommonResources)</ol>
	 * </ul>
	 */
	
	public void run() {
		ILock shutdownSynch = null;
		
		if (Main.NODE != null){
			// writes to log that is shutting down
			LogAppl.getInstance().emit(NodeMessage.JEMC071I, Main.NODE.getLabel());
			
			if (Main.HAZELCAST != null){
				shutdownSynch = Main.HAZELCAST.getLock(Queues.SHUTDOWN_LOCK);
				shutdownSynch.lock();
			}
			// sets the static reference to true
			Main.IS_SHUTTING_DOWN = true;
			// drains initiator
			NodeInfoUtility.drain();
			// waits for a second before to check the status
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {
			}

			// waits for the status is drained (means the end of eventually execution job)
			while(!Main.NODE.getStatus().equals(Status.DRAINED)){
				if (Main.CURRENT_TASK != null){
					LogAppl.getInstance().emit(NodeMessage.JEMC072I, Main.CURRENT_TASK.getJobTask().getJob().toString());
				}
				try {
					Thread.sleep(10 * TimeUtils.SECOND);
				} catch (InterruptedException e) {
				}
			}
		}
		
		
		// interrupts all threads
		if (jclCheckManager != null){
			try {
				jclCheckManager.shutdown();
			} catch (Exception e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC162E, e, StringUtils.substringAfterLast(JclCheckingQueueManager.class.getName(), "."));
			}
		}
		// interrupts all threads
		if (Main.INPUT_QUEUE_MANAGER != null){
			try {
				Main.INPUT_QUEUE_MANAGER.shutdown();
			} catch (Exception e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC162E, e, StringUtils.substringAfterLast(InputQueueManager.class.getName(), "."));
			}
		}

		// interrupts all threads
		if (submitter != null){
			try {
				submitter.shutdown();
			} catch (Exception e) {
				LogAppl.getInstance().emit(NodeMessage.JEMC162E, e, StringUtils.substringAfterLast(Submitter.class.getName(), "."));
			}
		}

		// stops teh statistics timer
		if (Main.STATISTICS_MANAGER != null){
			Main.STATISTICS_MANAGER.stop();
		}
		
		if (Main.HAZELCAST != null){
			// shuts down Hazelcast
			Main.HAZELCAST.getLifecycleService().shutdown();
		}

		// closes DB connection 
		try {
			//JobDBManager.getInstance().close();
			DBPoolManager.getInstance().close();
			
			LogAppl.getInstance().emit(NodeMessage.JEMC069I, StringUtils.substringAfterLast(DBPoolManager.class.getName(), ".") );
		} catch (Exception e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC070E, e,  StringUtils.substringAfterLast(DBPoolManager.class.getName(), "."));
		}

	}
}