/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Alessandro Zambrini
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package org.pepstock.jem.node;

import org.pepstock.jem.log.Description;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.log.MessageInterface;

/**
 * It is an enumeration containing all the messages about JEM node. <br>
 * It is a list of <code>NodeMessage</code>. <br>
 * Each <code>NodeMessage</code> in the list corresponds to a
 * <code>Message</code>. <br>
 * It implements {@link MessageInterface}
 * 
 * @see Message
 * @author Alessandro Zambrini
 * @version 1.0
 */
public enum NodeMessage implements MessageInterface {

//	JEMC198W(198, "Arguments \"{0}\" and \"{1}\" are not compatible", Message.WARNING),
//	
//	/**
//	 * ""Unable to get FTp stream for file \"{0}\" with reply code \"{1}\"", Message.ERROR
//	 */
//	@Description(explanation = "It occurs when it's not able to have FTP stream. Please check if file exists.")
//	JEMC199E(199, "Unable to get FTp stream for file \"{0}\" with reply code \"{1}\"", Message.ERROR);
	
	/**
	 * "Node is starting", Message.INFO
	 */
	JEMC001I(1, "Node is starting", Message.INFO),

	/**
	 * "Hazelcast node is starting", Message.INFO
	 */
	JEMC002I(2, "Hazelcast node is starting", Message.INFO),

	/**
	 * Hazelcast node [{0}] started", Message.INFO
	 */
	@Description(explanation = "Display the UUID of Hazelcast member")
	JEMC003I(3, "Hazelcast node [{0}] started", Message.INFO),

	/**
	 * "Unable to resolve hostname for the Hazelcast node", Message.ERROR
	 */
	JEMC004E(4, "Unable to resolve hostname for the Hazelcast node", Message.ERROR),

	/**
	 * "System property {0} not found", Message.ERROR
	 */
	@Description(explanation = "Display the system property not set but mandatory. Check command line.")
	JEMC005E(5, "System property {0} not found", Message.ERROR),

	/**
	 * "Configuration error", Message.ERROR
	 */
	@Description(explanation = "Some mistakes are present on configuration files. Check configuration files.")
	JEMC006E(6, "Configuration error", Message.ERROR),

	/**
	 * "Unable to start RMI registry", Message.ERROR
	 */
	@Description(explanation = "RMI registry is not able to start. Check the ports allocation.")
	JEMC007E(7, "Unable to start RMI registry", Message.ERROR),

	/**
	 * "Configuration file {0} loaded", Message.INFO
	 */
	@Description(explanation = "Display the config file which has been loaded")
	JEMC008I(8, "Configuration file {0} loaded", Message.INFO),

	/**
	 * "Configuration xml node {0} not found", Message.ERROR
	 */
	@Description(explanation = "Display the element of xml that is missing nonetheless is mandatory. Check configuration file.")
	JEMC009E(9, "Configuration xml node {0} not found", Message.ERROR),

	/**
	 * "Environment mismatch configuration: \"{0}\" against in Hazelcast \"{1}\""
	 * , Message.ERROR
	 */
	@Description(explanation = "Display the 2 different values found inside of JEM configuration node and Hazelcast configuration file. " + "The environment value and group one must be the same. "
			+ "Check the configuration files changing the group name or environment value.")
	JEMC010E(10, "Environment mismatch configuration: \"{0}\" against in Hazelcast \"{1}\"", Message.ERROR),

	/**
	 * "{0}", Message.INFO - used for Hazelcast log records
	 */
	@Description(explanation = "Display all messages produced by Hazelcast, errors as well.")
	JEMC011I(11, "{0}", Message.INFO),

	/**
	 * "Initiator [{0}] started", Message.INFO
	 */
	@Description(explanation = "Display the process ID and hostname of JEM node started.")
	JEMC012I(12, "Initiator [{0}] started", Message.INFO),

	/**
	 * "RMI registry is starting on port {0}", Message.INFO
	 */
	@Description(explanation = "Display the listening RMI port.")
	JEMC013I(13, "RMI registry is starting on port {0}", Message.INFO),

	/**
	 * "RMI registry started", Message.INFO
	 */
	JEMC014I(14, "RMI registry started", Message.INFO),

	/**
	 * "Remote-object \"{0}\" is added", Message.INFO
	 */
	@Description(explanation = "Display the name of RMI object, binded on registry")
	JEMC015I(15, "Remote-object \"{0}\" is added", Message.INFO),

	/**
	 * "Unable to add remote-object \"{0}\"", Message.ERROR
	 */
	@Description(explanation = "Display the name of RMI object which wasn't able to be binded on registry. See the exception")
	JEMC016E(16, "Unable to add remote-object \"{0}\"", Message.ERROR),

	/**
	 * "Interrupt waiting", Message.ERROR
	 */
	@Description(explanation = "Some interrupt unblock the wait state. Internal error.")
	JEMC017E(17, "Interrupt waiting", Message.ERROR),

	/**
	 * "Job {0} is taken for processing", Message.INFO
	 */
	@Description(explanation = "Display the job name and id of job taken from input queue to be executed.")
	JEMC018I(18, "Job {0} is taken for processing", Message.INFO),

	/**
	 * "Job {0} is routed to \"{1}\" for processing", Message.INFO
	 */
	@Description(explanation = "Display the job name and id of job which was routed to another cluster.")
	JEMC019I(19, "Job {0} is routed to \"{1}\" for processing", Message.INFO),

	/**
	 * "Job {0} is submitted for processing", Message.INFO
	 */
	@Description(explanation = "Display the job name and id of job which is now in submission phase, not executed yet.")
	JEMC020I(20, "Job {0} is submitted for processing", Message.INFO),

	/**
	 * "Job {0} is ended in return-code {1}", Message.INFO
	 */
	@Description(explanation = "Display the job name and id of job which is ended in the displayed return code.")
	JEMC021I(21, "Job {0} is ended in return-code {1}", Message.INFO),

	/**
	 * "Job {0} is running on process {1}", Message.INFO
	 */
	@Description(explanation = "Display the job name and id of job which is now running in new process with displayed process id.")
	JEMC022I(22, "Job {0} is running on process {1}", Message.INFO),

	/**
	 * "Unable to resolve hostname", Message.ERROR
	 */
	JEMC023E(23, "Unable to resolve hostname", Message.ERROR),

	/**
	 * "Member removed \"{0}\" is not in {1} queue", Message.ERROR
	 */
	@Description(explanation = "Display the member which is not present on queue as expected. Internal error.")
	JEMC024E(24, "Member removed \"{0}\" is not in {1} queue", Message.ERROR),

	/**
	 * "Process ID is null and \"cancel\" command is ignored", Message.WARNING
	 */
	@Description(explanation = "Cancel command aborted because the job is ended.")
	JEMC025W(25, "Process ID is null and \"cancel\" command is ignored", Message.WARNING),

	/**
	 * "Path \"{0}\" has been created", Message.INFO
	 */
	@Description(explanation = "Display the path that is necessary but it doesn't exist so JEM created.")
	JEMC026I(26, "Path \"{0}\" has been created", Message.INFO),

	/**
	 * "Unable to write job log record", Message.ERROR
	 */
	@Description(explanation = "Internal error.")
	JEMC027E(27, "Unable to write job log record", Message.ERROR),

	/**
	 * "Ask canceling job {0} which is not equal to job {1}, currently running",
	 * Message.WARNING
	 */
	@Description(explanation = "Cancel command aborted because the job is ended.")
	JEMC028W(28, "Ask canceling job {0} which is not equal to job {1}, currently running", Message.WARNING),

	/**
	 * "Cancel command for job {0} by {1} (force mode:{2})", Message.INFO
	 */
	@Description(explanation = "Display the job name, id of job and user who canceled it.")
	JEMC029I(29, "Cancel command for job {0} by {1} (force mode:{2})", Message.INFO),

	/**
	 * "Initiator is {0}", Message.INFO
	 */
	@Description(explanation = "Display the status of JEM node.")
	JEMC030I(30, "Initiator is {0}", Message.INFO),

	/**
	 * "Unable to load factory \"{0}\"", Message.ERROR
	 */
	@Description(explanation = "Display the class name which wasn't able to be loaded.<br>Check the class name and classpath of JEM node")
	JEMC031E(31, "Unable to load factory \"{0}\"", Message.ERROR),

	/**
	 * "Factory \"{0}\" [type=\"{1}\"] loaded", Message.INFO
	 */
	@Description(explanation = "Display the JEM factory loaded and its type.")
	JEMC032I(32, "Factory \"{0}\" [type=\"{1}\"] loaded", Message.INFO),

	/**
	 * "Unable to submit job", Message.ERROR
	 */
	@Description(explanation = "Client message.")
	JEMC033E(33, "Unable to submit job", Message.ERROR),

	/**
	 * "Job lifecycle listeners are ready", Message.INFO
	 */
	JEMC034I(34, "Job lifecycle listeners are ready", Message.INFO),

	/**
	 * "Unable to load job listener \"{0}\"", Message.ERROR
	 */
	@Description(explanation = "Display the class name which wasn't able to be loaded.<br>Check the class name and classpath of JEM node.")
	JEMC035E(35, "Unable to load job listener \"{0}\"", Message.ERROR),

	/**
	 * "Configured job listener \"{0}\" is not an instance of JobLifecycleListener"
	 * , Message.ERROR
	 */
	@Description(explanation = "Display the class name which wasn't able to be loaded.<br>" + "Check the class name and classpath of JEM node because is not a JobLifecycleListener.")
	JEMC036E(36, "Configured job listener \"{0}\" is not an instance of JobLifecycleListener", Message.ERROR),

	/**
	 * "JobLifecycle listener \"{0}\" loaded", Message.INFO
	 */
	@Description(explanation = "Display the JobLifecycle listener loaded.")
	JEMC037I(37, "JobLifecycle listener \"{0}\" loaded", Message.INFO),

	/**
	 * "{0} \"{1}\" is not well-defined", Message.WARNING
	 */
	@Description(explanation = "Display the element of xml that is not wel defined.<br>Check configuration file.")
	JEMC038W(38, "{0} \"{1}\" is not well-defined", Message.WARNING),

	/**
	 * "Value of \"{0}\" is not well-defined", Message.ERROR
	 */
	@Description(explanation = "Display the path name not defined even if is mandatory.<br>Check configuration file.")
	JEMC039E(39, "Value of path \"{0}\" is not defined", Message.ERROR),

	/**
	 * "Configured factory \"{0}\" is not an instance of JemFactory",
	 * Message.ERROR
	 */
	@Description(explanation = "Display the JEM factory that is not a JEM Factory.<br>Check configuration file and source code of factory.")
	JEMC040E(40, "Configured factory \"{0}\" is not an instance of JemFactory", Message.ERROR),

	/**
	 * "Job lifecycle listener system is not able to process the queue
	 */
	@Description(explanation = "Internal error of event manager.")
	JEMC041E(41, "Job lifecycle listener system is not able to process the queue", Message.ERROR),

	/**
	 * "Job lifecycle listener system is not able to add a event in queue,
	 * Message.ERROR
	 */
	@Description(explanation = "Internal error of event manager.")
	JEMC042E(42, "Job lifecycle listener system is not able to add a event in queue", Message.ERROR),

	/**
	 * "Unable to persist objects due to an exception on database",
	 * Message.ERROR
	 */
	@Description(explanation = "Occurs when the persistent MapManager is not able to store data on database.")
	JEMC043E(43, "Unable to persist objects due to an exception on database", Message.ERROR),

	/**
	 * "Database manager instance is null.", Message.ERROR
	 */
	@Description(explanation = "Occurs when the persistent MapManager is not able to store data on database.")
	JEMC044E(44, "Database manager instance is null.", Message.ERROR),

	/**
	 * "{0} key loaded for \"{1}\" queue", Message.INFO
	 */
	@Description(explanation = "Display the map name and keys loaded (and then the objects) for Hazelcast.")
	JEMC045I(45, "{0} keys loaded for \"{1}\" queue", Message.INFO),

	/**
	 * "GRS Status display before {0}\n {1}", Message.INFO
	 */
	// @Description(explanation =
	// "Display all contentions inside of GRS system.")
	// JEMC046I(46, "GRS Status display before {0}\n {1}", Message.INFO),

	/**
	 * "Job {0} is ended in exception:\n {1}", Message.ERROR
	 */
	JEMC047E(47, "Job {0} is ended in exception:\n {1}", Message.ERROR),

	/**
	 * "{0} jobs loaded for \"{1}\" queue", Message.INFO
	 */
	@Description(explanation = "Display how many jobs are loaded from database.")
	JEMC048I(48, "{0} jobs loaded for \"{1}\" queue", Message.INFO),

	/**
	 * "Affinity \"{0}\" loaded", Message.INFO
	 */
	@Description(explanation = "Display the affinity factory loaded.")
	JEMC049I(49, "AffinityFactory \"{0}\" loaded", Message.INFO),

	/**
	 * "{0}" ", Message.INFO
	 */
	@Description(explanation = "Display all information of configured and calculated execution environment.")
	JEMC050I(50, "{0}", Message.INFO),

	/**
	 * "JS policy file \"{0}\" executed", Message.INFO
	 */
	@Description(explanation = "Display script file used for policy by affinity factory.")
	JEMC051I(51, "Policy file \"{0}\" executed", Message.INFO),

	/**
	 * "{0} is added", Message.INFO
	 */
	@Description(explanation = "Display the common resource added.")
	JEMC052I(52, "Resource {0} is added", Message.INFO),

	/**
	 * "{0} is replaced", Message.INFO
	 */
	@Description(explanation = "Display the common resource removed.")
	JEMC053I(53, "Resource {0} is replaced", Message.INFO),

	/**
	 * "Unable to add or replace a resource", Message.ERROR
	 */
	@Description(explanation = "Internal error.")
	JEMC054E(54, "Unable to add or replace a resource", Message.ERROR),

	/**
	 * "{0} resources loaded for \"{1}\" queue", Message.INFO
	 */
	@Description(explanation = "Display how many resources are loaded from database.")
	JEMC055I(55, "{0} resources loaded for \"{1}\" queue", Message.INFO),

	/**
	 * "Missing required options:\n{0}, Message.INFO
	 */
	@Description(explanation = "Display all necessary options and how command usage, because the command has been launched with all mandatory options.<br>" + "Check command line")
	JEMC056I(56, "Missing required options:\n{0}", Message.INFO),

	/**
	 * ""Environment variable \"{0}\" is loaded. Value: \"{1}\"", Message.INFO
	 */
	JEMC057I(57, "Environment variable \"{0}\" is loaded. Value: \"{1}\"", Message.INFO),

	/**
	 * ""Environment variable \"{0}\" is missing but mandatory", Message.ERROR
	 */
	JEMC058E(58, "Environment variable \"{0}\" is missing but mandatory", Message.ERROR),

	/**
	 * "\"{0}\" successfully created see path: \"{1}\", Message.INFO
	 */
	JEMC059I(59, "\"{0}\" successfully created see path: \"{1}\"", Message.INFO),

	/**
	 * "The path \"{0}\" already exist cannot create node, Message.INFO
	 */
	JEMC060E(60, "The path \"{0}\" already exist cannot create node", Message.ERROR),

	/**
	 * "{0} is removed", Message.INFO
	 */
	JEMC061I(61, "Resource {0} is removed", Message.INFO),

	/**
	 * "{0} doesn't exist", Message.WARNING
	 */
	JEMC062W(62, "Resource {0} does not exist", Message.WARNING),

	/**
	 * Resource {0} properties:\n{1}, Message.INFO
	 */
	JEMC063I(63, "Resource {0} properties:\n{1}", Message.INFO),

	/**
	 * Resources list:\n{0}, Message.INFO
	 */
	JEMC064I(64, "Resources list:\n{0}", Message.INFO),

	/**
	 * "Error while substituting variable on string {0}", Message.ERROR
	 */
	JEMC065E(65, "Error while substituting variable on string {0}", Message.ERROR),

	/**
	 * "Missing required property {0}", Message.ERROR
	 */
	JEMC066E(66, "Missing required property {0}", Message.ERROR),

	/**
	 * "Folder {0} does not exist, you need to create it", Message.ERROR
	 */
	JEMC067E(67, "Folder {0} does not exist, you need to create it", Message.ERROR),

	/**
	 * "Component {0} is shutting down", Message.WARNING
	 */
	JEMC068W(68, "Component {0} is shutting down", Message.WARNING),

	/**
	 * "{0} shutdown is completed", Message.INFO
	 */
	JEMC069I(69, "{0} shutdown is completed", Message.INFO),

	/**
	 * "{0} shutdown is completed with exception", Message.ERROR
	 */
	JEMC070E(70, "{0} shutdown is completed with exception", Message.ERROR),

	/**
	 * "{0} starts shutdown procedure", Message.INFO
	 */
	JEMC071I(71, "{0} starts shutdown procedure", Message.INFO),

	/**
	 * "Job {0} is still running. Wait for 20 seconds to check again",
	 * Message.INFO
	 */
	JEMC072I(72, "Job {0} is still running. Wait for 20 seconds to check again", Message.INFO),

	/**
	 * "Error during creation of database for environment {0}." , Message.ERROR
	 */
	JEMC073E(73, "Error during creation of database for environment {0}.", Message.ERROR),

	/**
	 * "Job lifecycle listener error.", Message.ERROR
	 */
	@Description(explanation = "It occurs when there is a general problem inside the Job lifecycle listener execution.<br>Main Job lifecycle listener methods cannot throw exceptions, so errors are cached and this log is written.<br>No exception is thrown, but this error is logged.<br>Check the log because the real problem cause is surely written before this log, so it is possible to correct the error.")
	JEMC074E(74, "Job lifecycle listener error.", Message.ERROR),

	/**
	 * "Created the folder \"{0}\" to store statistics logs files", Message.INFO
	 */
	@Description(explanation = "It occurs when the statistics folder is missing but necessary to store the statistics log record so it creates it.")
	JEMC075I(75, "Created the folder \"{0}\" to store statistics logs files", Message.INFO),

	/**
	 * "The folder \"{0}\" is used to store statistics logs files", Message.INFO
	 */
	@Description(explanation = "It occurs when the statistics manager is starting and informs the folder used to store the statistics log record so it creates it.")
	JEMC076I(76, "The folder \"{0}\" is used to store statistics logs files", Message.INFO),

	/**
	 * "The statistics manager is stopped", Message.INFO
	 */
	@Description(explanation = "It occurs when shutdown hook is started to close the JEM node.")
	JEMC077I(77, "The statistics manager is stopped", Message.INFO),

	/**
	 * "The current statistics manager is managing statistics for whole JEM environment"
	 * , Message.INFO
	 */
	@Description(explanation = "It occurs when the node is the coordinator and works to extract from all nodes and store all statistics.")
	JEMC078I(78, "The current statistics manager is managing statistics for whole JEM environment", Message.INFO),

	/**
	 * "Unable to execute statistics remote command on all nodes", Message.ERROR
	 */
	@Description(explanation = "It occurs when receives an exception during the execution of future task on all node to extract statistics.<br>Check the showed exception.")
	JEMC079E(79, "Unable to execute statistics remote command on all nodes", Message.ERROR),

	/**
	 * "Unable to write statistics log record in \"{0}\" file", Message.ERROR
	 */
	@Description(explanation = "It occurs when receives an exception during the writing of statistics log records in the file.<br>Check the showed exception and if there's any file system error.")
	JEMC080E(80, "Unable to write statistics log record in \"{0}\" file", Message.ERROR),

	/**
	 * "Statistics management lasted \"{0}\" ms", Message.INFO
	 */
	@Description(explanation = "It occurs when the statistics manager ends its activities to extract and store the stat log records and it lasted more than 2 seconds.<br>It shows the elapsed time in milliseconds spent to complete the activities.")
	JEMC081W(81, "Statistics management lasted {0} ms", Message.WARNING),

	/**
	 * "Unable to write statistics log record in \"{0}\" file", Message.ERROR
	 */
	@Description(explanation = "It occurs to inform what file is used to store statistics log records.")
	JEMC082I(82, "The statistics log file used to store records is \"{0}\" file", Message.INFO),

	/**
	 * "Database for environment {0} already present.", Message.INFO
	 */
	@Description(explanation = "It occurs during the creation of a new environment in case the database is alreadty present, for example because you have installed the same environment in different machines.")
	JEMC083I(83, "Database for environment {0} already present.", Message.INFO),

	/**
	 * "File  {0} is been modified. {1}", Message.INFO
	 */
	@Description(explanation = "It occurs when a configuration file is been modified.")
	JEMC084I(84, "File  {0} is been modified. {1} ", Message.INFO),

	/**
	 * "Map {0} has size {1} KB", Message.INFO
	 */
	@Description(explanation = "It occurs when the cluster check if there are enough member to start up")
	JEMC085I(85, "Map {0} has size {1} KB", Message.INFO),

	/**
	 * "Cluster free memory= {0} KB is less than the needed memory= {1} KB. Wait for other nodes to join the claster."
	 * , Message.WARNING
	 */
	@Description(explanation = "It occurs at cluster start up when the are not enough nodes to bring the persisted queue in meory")
	JEMC086W(86, "Cluster free memory= {0} KB is less than the needed memory= {1} KB. Wait for other nodes to join the claster.", Message.WARNING),

	/**
	 * "The path {0} set for property {1} must not contain spaces.",
	 * Message.ERROR
	 */
	@Description(explanation = "It occurs when configuring a node if the path for the some folders of the gfs contains white spaces.")
	JEMC087E(87, "The path {0} set for property {1} must not contain spaces.", Message.ERROR),

	/**
	 * "The path {0} set for property {1} does not exist.", Message.ERROR
	 */
	@Description(explanation = "It occurs when configuring a node if the path for the some folders of the gfs does not exist.")
	JEMC088E(88, "The path {0} set for property {1} does not exist.", Message.ERROR),

	/**
	 * "Configured factory \"{0}\" is not an instance of AffinityLoader",
	 * Message.ERROR
	 */
	@Description(explanation = "Display the affintity loader that is not a loader.<br>Check configuration file and source code of affinity loader.")
	JEMC089E(89, "Configured factory \"{0}\" is not an instance of AffinityLoader", Message.ERROR),

	/**
	 * "Affinity \"{0}\" loaded", Message.INFO
	 */
	@Description(explanation = "Display the node implementation which is loaded.")
	JEMC090I(90, "Node \"{0}\" loaded", Message.INFO),

	/**
	 * "Configured node \"{0}\" is not an instance of NodeInfo", Message.ERROR
	 */
	@Description(explanation = "Display the node class that is not a NodeInfo.<br>Check configuration file and source code of Node.")
	JEMC091E(91, "Configured node \"{0}\" is not an instance of NodeInfo", Message.ERROR),

	/**
	 * Unable to read {0} object after javascript execution", Message.ERROR
	 */
	@Description(explanation = "Errors during retreving the object result after javascript execution.<br>Check the javascript policy file.")
	JEMC092E(92, "Unable to read {0} object after javascript execution", Message.ERROR),

	/**
	 * "Job \"{0}\" is in input queue", Message.INFO
	 */
	@Description(explanation = "Simple information from default job listener that job is inut queue.")
	JEMC093I(93, "Job \"{0}\" is in input queue", Message.INFO),

	/**
	 * "Job \"{0}\" is running", Message.INFO
	 */
	@Description(explanation = "Simple information from default job listener that job is running.")
	JEMC094I(94, "Job \"{0}\" is running", Message.INFO),

	/**
	 * "Job \"{0}\" is ended. Return-code={1}", Message.INFO
	 */
	@Description(explanation = "Simple information from default job listener that job is ended.")
	JEMC095I(95, "Job \"{0}\" is ended. Return-code={1}", Message.INFO),

	/**
	 * Hazelcast configuration file for the environment is null., Message.ERROR
	 */
	@Description(explanation = "It occurs when it's not possible to locate Hazelcast configuration file.<br>Check the location and path for JEM node installation.")
	JEMC096E(96, "Hazelcast configuration file \"{0}\" for the environment is null.", Message.ERROR),

	/**
	 * "JCL url is null.", Message.ERROR
	 */
	@Description(explanation = "It occurs when url, which represents JCL, is null.<br> Check where JCL is and its URL.")
	JEMC97E(97, "JCL url is null.", Message.ERROR),

	/**
	 * "{0} isn't a directory.", Message.ERROR
	 */
	JEMC098E(98, "{0} isn't a directory.", Message.ERROR),

	/**
	 * "{0} doesn't exist.", Message.ERROR
	 */
	JEMC099E(99, "{0} doesn't exist.", Message.ERROR),

	/**
	 * "{0} not found.", Message.ERROR
	 */
	JEMC100E(100, "{0} not found.", Message.ERROR),

	/**
	 * "{0} does not name a context.", Message.ERROR
	 */
	JEMC101E(101, "{0} does not name a context.", Message.ERROR),

	/**
	 * "Use rebind to override.", Message.ERROR
	 */
	JEMC102E(102, "Use rebind to override.", Message.ERROR),

	/**
	 * "Error while retrieving roles for user {0}", Message.ERROR
	 */
	JEMC103E(103, "Error while retrieving roles for user {0}", Message.ERROR),

	/**
	 * "{0} not allowed!", Message.ERROR
	 */
	JEMC104E(104, "{0} not allowed!", Message.ERROR),

	/**
	 * "User not allowed for permission {0}", Message.ERROR
	 */
	JEMC105E(105, "User not allowed for permission {0}", Message.ERROR),

	/**
	 * "Login denied.", Message.WARNING
	 */
	JEMC106W(106, "Login denied.", Message.WARNING),

	/**
	 * "Login denied for {0}", Message.WARNING
	 */
	JEMC107W(107, "Login denied for {0}", Message.WARNING),

	/**
	 * "Password in Hazelcast configuration must not be null or empty"" ,
	 * Message.ERROR
	 */
	@Description(explanation = "Password in Hazelcast configuration must not be null or empty")
	JEMC108E(108, "Password in Hazelcast configuration must not be null or empty", Message.ERROR),

	/**
	 * "Invalid request/response {0}", Message.WARNING
	 */
	JEMC109W(109, "Invalid request/response {0}", Message.WARNING),

	/**
	 * "Changed database encryption for environment {0}.", Message.INFO
	 */
	@Description(explanation = "It occurs during during the execution of the command ChangeDbEncryption if the database is already encrypted.")
	JEMC110I(110, "Changed database encryption for environment {0}.", Message.INFO),

	/**
	 * "Encrypted database for environment {0}.", Message.INFO
	 */
	@Description(explanation = "It occurs during during the execution of the command ChangeDbEncryption if the database is not encrypted.")
	JEMC111I(111, "Encrypted database for environment {0}.", Message.INFO),

	/**
	 * "Executor call back exception.", Message.ERROR
	 */
	@Description(explanation = "Internal error during a command. It may be called from user interface.")
	JEMC112E(112, "Executor call back exception.", Message.ERROR),

	/**
	 * "Executor call back result {0}.", Message.INFO
	 */
	@Description(explanation = "Internal error during a command. It may be called from user interface.")
	JEMC113I(113, "Executor call back result {0}.", Message.INFO),

	/**
	 * "Property \"{0}\" is not defined.", Message.ERROR
	 */
	@Description(explanation = "The JS affinity file is null.")
	JEMC114E(114, "Property \"{0}\" is not defined.", Message.ERROR),

	/**
	 * "{0} is shutting down.", Message.ERROR
	 */
	JEMC115E(115, "{0} is shutting down.", Message.ERROR),

	/**
	 * "Key is null.", Message.ERROR
	 */
	JEMC116E(116, "Key is null.", Message.ERROR),

	/**
	 * "Connection object instance is null.", Message.ERROR
	 */
	JEMC117E(117, "Connection object instance is null.", Message.ERROR),

	/**
	 * "Hashed valued is not equals to orginal cryted value.", Message.ERROR
	 */
	JEMC118E(118, "Hashed valued is not equals to orginal cryted value.", Message.ERROR),

	/**
	 * "Unable to lock \"{0}\" map in 10 seconds.", Message.ERROR
	 */
	JEMC119E(119, "Unable to lock \"{0}\" map in 10 seconds.", Message.ERROR),

	/**
	 * "User \"{0}\" have not got the {1}.", Message.ERROR
	 */
	JEMC120E(120, "User \"{0}\" have not got the {1}.", Message.ERROR),

	/**
	 * "User \"{0}\" is not authorized for \"{1}\".", Message.ERROR
	 */
	JEMC121E(121, "User \"{0}\" is not authorized for \"{1}\".", Message.ERROR),

	/**
	 * "\"{0}\" is not defined!", Message.ERROR
	 */
	JEMC122E(122, "{0} is not defined!", Message.ERROR),

	/**
	 * "Role {0} couldn't be changed.", Message.ERROR
	 */
	JEMC123E(123, "Role \"{0}\" couldn't be changed.", Message.ERROR),

	/**
	 * "Role \"{0}\" doesn't exist.", Message.ERROR
	 */
	JEMC124E(124, "Role \"{0}\" doesn't exist.", Message.ERROR),

	/**
	 * "Role \"{0}\" is not removable.", Message.ERROR
	 */
	JEMC125E(125, "Role \"{0}\" is not removable.", Message.ERROR),

	/**
	 * "Role \"{0}\" already exists.", Message.ERROR
	 */
	JEMC126E(126, "Role \"{0}\" already exists.", Message.ERROR),

	/**
	 * "Any override to a 'jem' property is not allowed!", Message.ERROR
	 */
	JEMC127E(127, "Any override to a 'jem' property is not allowed!", Message.ERROR),

	/**
	 * "Access to RMI listener is not allowed.", Message.ERROR
	 */
	JEMC128E(128, "Access to RMI listener is not allowed.", Message.ERROR),

	/**
	 * "Wildcard string cannot be null or empty. Make sure permission strings are properly formatted."
	 * , Message.ERROR
	 */
	JEMC129E(129, "Wildcard string cannot be null or empty. Make sure permission strings are properly formatted.", Message.ERROR),

	/**
	 * "Wildcard string cannot contain parts with only dividers. Make sure permission strings are properly formatted."
	 * , Message.ERROR
	 */
	JEMC130E(130, "Wildcard string cannot contain parts with only dividers. Make sure permission strings are properly formatted.", Message.ERROR),

	/**
	 * "Wildcard string cannot contain only dividers. Make sure permission strings are properly formatted."
	 * , Message.ERROR
	 */
	JEMC131E(131, "Wildcard string cannot contain only dividers. Make sure permission strings are properly formatted.", Message.ERROR),

	/**
	 * "Resource \"{0}\" is not a InputStream.", Message.ERROR
	 */
	JEMC132E(132, "Resource \"{0}\" is not a InputStream.", Message.ERROR),

	/**
	 * "Object is not a reference {0}.", Message.ERROR
	 */
	JEMC133E(133, "Object is not a reference {0}.", Message.ERROR),

	/**
	 * "Resource \"{0}\" is not a OutputStream.", Message.ERROR
	 */
	JEMC134E(134, "Resource \"{0}\" is not a OutputStream.", Message.ERROR),

	/**
	 * "DataSet INLINE is not allowed to open in output mode.", Message.ERROR
	 */
	JEMC135E(135, "DataSet INLINE is not allowed to open in output mode.", Message.ERROR),

	/**
	 * "Mandatory property \"{0}\" is missing.", Message.ERROR
	 */
	JEMC136E(136, "Mandatory property \"{0}\" is missing.", Message.ERROR),

	/**
	 * "The protocol must be 'ftp' or 'ftps' instead of \"{0}\".", Message.ERROR
	 */
	JEMC137E(137, "The protocol must be 'ftp' or 'ftps' instead of \"{0}\".", Message.ERROR),

	/**
	 * "Reply code \"{0}\", during connection, is not positive.", Message.ERROR
	 */
	JEMC138E(138, "Reply code \"{0}\", during connection, is not positive.", Message.ERROR),

	/**
	 * "Cannot bind empty name.", Message.ERROR
	 */
	JEMC139E(139, "Cannot bind empty name.", Message.ERROR),

	/**
	 * "getObjectInstance failed.", Message.ERROR
	 */
	JEMC140E(140, "getObjectInstance failed.", Message.ERROR),

	/**
	 * "System property \"{0}\" is not set.", Message.ERROR
	 */
	JEMC141E(141, "System property \"{0}\" is not set.", Message.ERROR),

	/**
	 * "Object \"{0}\" is missing.", Message.ERROR
	 */
	JEMC142E(142, "Object \"{0}\" is missing.", Message.ERROR),

	/**
	 * "Factory for {0} is missing.", Message.ERROR
	 */
	JEMC143E(143, "Factory for {0} is missing.", Message.ERROR),

	/**
	 * "User \"{0}\" does not have the \"{1}\" permission.", Message.ERROR
	 */
	JEMC144E(144, "User \"{0}\" does not have the \"{1}\" permission.", Message.ERROR),

	/**
	 * "Resource name is null.", Message.ERROR
	 */
	JEMC145E(145, "Resource name is null.", Message.ERROR),

	/**
	 * "Mode must be READ_MODE or WRITE_MODE value.", Message.ERROR
	 */
	JEMC146E(146, "Mode must be READ_MODE or WRITE_MODE value.", Message.ERROR),

	/**
	 * "Error while loading node.", Message.ERROR
	 */
	JEMC147E(147, "Error while loading node.", Message.ERROR),

	/**
	 * "File \"{0}\" is opened for reading", Message.INFO
	 */
	@Description(explanation = "It occurs when a statistic file is opened for processing.")
	JEMC148I(148, "File \"{0}\" is opened for reading", Message.INFO),

	/**
	 * "File \"{0}\" is removed", Message.INFO
	 */
	@Description(explanation = "It occurs when a statistic file is removed after processing.")
	JEMC149I(149, "File \"{0}\" is removed", Message.INFO),

	/**
	 * "File \"{0}\" is not removed", Message.WARNING
	 */
	@Description(explanation = "It occurs when a statistic file is not removed after processing because it is not able to delete the file from file system.")
	JEMC150W(150, "File \"{0}\" is not removed", Message.WARNING),

	/**
	 * "File \"{0}\" is not removed", Message.WARNING
	 */
	@Description(explanation = "It occurs when a statistic file is not removed after processing because it is not able to read all samples.")
	JEMC151W(151, "File \"{0}\" is not removed due to previuos error on samples", Message.WARNING),

	/**
	 * "The sample at line \"{0}\" is not correct. Exception: \"{1}\"",
	 * Message.ERROR
	 */
	@Description(explanation = "It occurs when a sample in the statistic file is not correct, usually not XML well-formed<br> Please check content of file and correct it.")
	JEMC152E(152, "The sample at line \"{0}\" is not correct. Exception: \"{1}\"", Message.ERROR),

	/**
	 * "Unable to create {0} directory.", Message.ERROR
	 */
	@Description(explanation = "It occurs when the path (indicated in JEM node configuration) doesn't exists.<br> Please check 'statistics-manager' element in JEM node configuration file.")
	JEMC153E(153, "Unable to create {0} directory. Disable statistics persistence", Message.ERROR),

	/**
	 * "Unable to rename the node folder", Message.ERROR);
	 */
	JEMC154E(154, "Unable to rename the node folder", Message.ERROR),

	/**
	 * "Unable to write JCL and message log for {0} job", Message.WARNING);
	 */
	JEMC155W(155, "Unable to write JCL and message log for {0} job, after job validation", Message.WARNING),

	/**
	 * "Error creating the HttpClient for HttpResource. Exception: \"{0}\"",
	 * Message.ERROR
	 */
	@Description(explanation = "It occurs when an error occurs creating the HttpClient for HttpResource.<br> Please check the configuration file of the HttpResource.")
	JEMC156E(156, "Error creating the HttpClient for HttpResource. Exception: \"{0}\"", Message.ERROR),

	/**
	 * "Error executing Http Request Method. Exception: \"{0}\"", Message.ERROR
	 */
	@Description(explanation = "It occurs when an error occurs executing Http Request Method.<br> Please check the log file for details.")
	JEMC157E(157, "Error executing Http Request Method. Exception: \"{0}\"", Message.ERROR),

	/**
	 * "Property {0} not found in the Context Environment executing Http Request Method."
	 * , Message.WARNING
	 */
	@Description(explanation = "It occurs executing Http Request Method when a property for is not found in the context Environment.")
	JEMC158W(158, "Property \"{0}\" not found in the Context Environment executing Http Request Method.", Message.WARNING),

	/**
	 * "Error executing login for HttpResource: {0} or {1} is mandatory.",
	 * Message.ERROR
	 */
	JEMC159E(159, "Error executing login for HttpResource: {0} or {1} is mandatory", Message.ERROR),

	/**
	 * "Error executing login for HttpResource: {0} is mandatory.",
	 * Message.ERROR
	 */
	JEMC160E(160, "Error executing login for HttpResource: {0} or is mandatory", Message.ERROR),

	/**
	 * "Statistic executor is still running from previous scheduling",
	 * Message.WARNING
	 */
	@Description(explanation = "It occurs executing statistic manager try to execute the collector task but it's still running.<br> Please check the log file for details and contact your JEM administrators.")
	JEMC161W(161, "Statistic executor is still running from previous scheduling", Message.WARNING),

	/**
	 * "Error during shutdown of {0} component", Message.ERROR
	 */
	@Description(explanation = "It occurs during shutdown phase and it wasn't able to shutdown a component.<br> Please check the log file for details and contact your JEM administrators.")
	JEMC162E(162, "Error during shutdown of {0} component", Message.ERROR),

	/**
	 * "Time occurs during polling on JCL queue", Message.ERROR
	 */
	@Description(explanation = "It occurs during JCL check. It means that node wasn't able to get the object from queue due to a lock on queue itself.<br> Please check the log file for details and contact your JEM administrators.")
	JEMC163E(163, "Time occurs during polling on JCL queue", Message.ERROR),

	/**
	 * "Unable to remove output job {0} on directory {1}", Message.WARNING
	 * Message.WARNING
	 */
	@Description(explanation = "It occurs when after purge command, output folder for job is not deleted.<br>Please check file system where otuput folder is stored.")
	JEMC164W(164, "Unable to remove output job {0} on directory {1}", Message.WARNING),

	/**
	 * "Unable to create the database connection for {0}", Message.ERROR);
	 */
	@Description(explanation = "It occurs during start up of node when opens the connection with database to persist Hazelcast structures.<br>Please check configuration file on database element and contact your JEM administrators.")
	JEMC165E(165, "Unable to create the database connection for {0}", Message.ERROR),

	/**
	 * "Unable to parse URL string parameter for the database connection: {0}",
	 * Message.ERROR);
	 */
	@Description(explanation = "It occurs during start up of node when it try to understand what is type of database.<br>Please check configuration file on database element and contact your JEM administrators.")
	JEMC166E(166, "Unable to parse URL string parameter for the database connection: {0}", Message.ERROR),

	/**
	 * "Unable to create the tables for the persistency", Message.ERROR);
	 */
	@Description(explanation = "It occurs during start up of node when tries to create the necessary tables to persist jobs.<br>Please check configuration file on database element and contact your JEM administrators.")
	JEMC167E(167, "Unable to create the tables for the persistency", Message.ERROR),

	/**
	 * "Unable to write job log record", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node tries to write the job in XML on output path.<br>Please check global file system availability and contact your JEM administrators.")
	JEMC168E(168, "Unable to write job object in xml format", Message.ERROR),

	/**
	 * "Unable to create statistic record", Message.WARNING
	 */
	@Description(explanation = "It occurs when the node is trying to create statistics but it gets an exception.<br>Please contact your JEM administrators.")
	JEMC169W(169, "Unable to create statistic record", Message.WARNING),

	/**
	 * "Unable to move job from checking map to input one", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to put a job in input map.<br>Please contact your JEM administrators.")
	JEMC170E(170, "Unable to move job {0} from checking map to input one", Message.ERROR),

	/**
	 * "Unable to move job {0} from checking map to output one", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to put a job in output map, after a validation of JCL.<br>Please contact your JEM administrators.")
	JEMC171E(171, "Unable to move job {0} from checking map to output one", Message.ERROR),

	/**
	 * ""Node is operational", Message.INFO
	 */
	@Description(explanation = "It occurs when the node, after applying all REDO statements, is again ready to work.")
	JEMC172I(172, "Node is operational", Message.INFO),

	/**
	 * ""Node is not operational", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to persist the job in queues on data base.<br>Please check database availability and contact your JEM administrators.")
	JEMC173E(173, "Node is not operational", Message.ERROR),

	/**
	 * ""Unable to store node information in memory data map", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to store its data in Hazelcast.<br>Please contact your JEM administrators.")
	JEMC174E(174, "Unable to store node information in memory data map", Message.ERROR),

	/**
	 * "Unable to move job {0} from running map to output one", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to put a job in output map, job execution.<br>Please contact your JEM administrators.")
	JEMC175E(175, "Unable to move job {0} from running map to output one", Message.ERROR),

	/**
	 * "Redo {0} applied", Message.INFO
	 */
	@Description(explanation = "It occurs when the node is able to redo all cached actions, after that database is available again.")
	JEMC176I(176, "Redo {0} applied", Message.INFO),

	/**
	 * "All Redo statemente ({0}) has been applied", Message.INFO
	 */
	@Description(explanation = "It occurs when the node applied all redo statements.")
	JEMC177I(177, "All Redo statemente ({0}) has been applied", Message.INFO),

	/**
	 * "Unable to apply redo statements", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to apply redo statement on data base.<br>Please check database availability and contact your JEM administrators.")
	JEMC178E(178, "Unable to apply redo statements", Message.ERROR),

	/**
	 * "Store a redo statement: {0}", Message.INFO
	 */
	@Description(explanation = "It occurs when the node is not able to persist data and save a redo statement in Hazelcast map.")
	JEMC179I(179, "Store a redo statement: {0}", Message.INFO),

	/**
	 * "Unable to store a redo statement", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to store redo statement in a Hazelcast map.<br>Please contact your JEM administrators.")
	JEMC180E(180, "Unable to store a redo statement", Message.ERROR),

	/**
	 * "Unable to check the free space of global file system", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is not able to read the freespace of OUTPUT folder.<br>Please check file system availability and contact your JEM administrators.")
	JEMC181E(181, "Unable to check the free space of global file system", Message.ERROR),

	/**
	 * "Not enough free space in global file system: {0}", Message.ERROR
	 */
	@Description(explanation = "It occurs when the node is aware that there is less than 10 MB free on global file system.<br>Please check file system usage and contact your JEM administrators.")
	JEMC182E(182, "Not enough free space in global file system: {0}", Message.ERROR),

	/**
	 * "Statistics persistence is disabled by configuration", Message.WARNING
	 */
	@Description(explanation = "It occurs when in JEM node configuration file the statistics manager is set with enable=false.<br/> No statistics record is saved on filesystem.")
	JEMC183W(183, "Statistics persistence is disabled by configuration", Message.WARNING),

	/**
	 * "Unable to extract jem version from jem.jar", Message.WARNING
	 */
	@Description(explanation = "It occurs when JEM node is unable to extract jem version from jem.jar")
	JEMC184W(184, "Unable to extract jem version from jem.jar", Message.WARNING),

	/**
	 * "Installed nodes has different release version, the behavior of the cluster may become unpredictable."
	 * , Message.WARNING
	 */
	@Description(explanation = "It occurs when JEM nodes inside the cluster have different release version")
	JEMC185W(185, "Installed nodes has different release version, the behavior of the cluster may become unpredictable", Message.WARNING),

	/**
	 * "Path \"{0}\" doesn't exist", Message.ERROR
	 */
	@Description(explanation = "It occurs when you're going thru global file system but you asked a path that doesn't exist! Please check inserted path.")
	JEMC186E(186, "Path \"{0}\" doesn't exist", Message.ERROR),

	/**
	 * "Path \"{0}\" is not a folder", Message.ERROR);
	 */
	@Description(explanation = "It occurs when you're going thru global file system but you asked a path that is not a path but a file! Please check inserted path.")
	JEMC187E(187, "Path \"{0}\" is not a folder", Message.ERROR),

	/**
	 * "Path \"{0}\" is not a file", Message.ERROR
	 */
	@Description(explanation = "It occurs when you're going thru global file system but you asked a file that is not a file but a path! Please check inserted file name.")
	JEMC188E(188, "Path \"{0}\" is not a file", Message.ERROR),

	/**
	 * "JEM is started in maintainance mode", Message.INFO
	 */
	@Description(explanation = "It occurs when the node is started in access maint, by the JEM property set to true.")
	JEMC189I(189, "JEM is started in maintainance mode", Message.INFO),

	/**
	 * "Job {0} was executing. ID: {1}", Message.WARNING
	 */
	@Description(explanation = "Display the job name and id of job which was executing and it was in ruuning queue due to a cluster failover. It moved on output queue. Please check the job if any action is needed to recover any inconsistent situation.")
	JEMC190W(190, "Job {0} was executing. ID: {1}", Message.WARNING),

	/**
	 * "Installed nodes has different release version", Message.WARNING
	 */
	@Description(explanation = "It occurs when JEM nodes inside the cluster have different release version")
	JEMC191E(191, "Installed nodes has different release version", Message.ERROR),

	/**
	 * "Installed nodes has different release version", Message.WARNING
	 */
	@Description(explanation = "Print jvm information")
	JEMC192I(192, "JVM version: {0}, home: {1}, vendor: {2}", Message.INFO),

	/**
	 * "Installed nodes has different release version", Message.WARNING
	 */
	@Description(explanation = "Print databe url on node startup")
	JEMC193I(193, "Database url: {0}", Message.INFO),

	/**
	 * "Platform not supported!", Message.ERROR
	 */
	@Description(explanation = "JEM is running in a platform not supported yet, to execute any job. Please contact your JEM administrator.")
	JEMC194E(194, "Platform not supported!", Message.ERROR),

	/**
	 * "Remote-object \"{0}\" is added", Message.INFO
	 */
	@Description(explanation = "Display the name of RMI EXTERNAL object, binded on registry. It's used for ANT utilities")
	JEMC195I(195, "External Remote-object \"{0}\" is added", Message.INFO),

	/**
	 * "Unable to add remote-object \"{0}\"", Message.WARNING
	 */
	@Description(explanation = "Display the name of RMI object which wasn't able to be binded on registry. See the exception. It's used for ANT utilities")
	JEMC196W(196, "Unable to add external remote-object by \"{0}\"", Message.WARNING),

	/**
	 * "{0} is been routed to environment {1}, cannot provide output.",
	 * Message.INFO
	 */
	@Description(explanation = "Inform user that the job was routed so it is not possible to get the log back.")
	JEMC197I(197, "{0} is been routed to environment {1}, cannot provide output.", Message.WARNING),

	/**
	 * "Arguments \"{0}\" and \"{1}\" are not compatible", Message.WARNING
	 */
	@Description(explanation = "Using the command line, you're using 2 arguments that are non comaptible. That means JEM is not able to apply both. Please check better command line. ")
	JEMC198W(198, "Arguments \"{0}\" and \"{1}\" are not compatible", Message.WARNING),

	/**
	 * "User private key is not protected by password", Message.WARNING
	 */
	@Description(explanation = "Inform user that the RSA private key supplied is not protected by password")
	JEMC199W(199, "User private key is not protected by password", Message.WARNING),

	/**
	 * "The certificate with alias {0} is not a X.509 certificate or is not present in the keystore"
	 * , Message.ERROR
	 */
	@Description(explanation = "Inform user that the provided certificate is not a valid X.509 certificate or is not present in the keystore.")
	JEMC200E(200, "The certificate with alias {0} is not a X.509 certificate or is not present in the keystore", Message.ERROR),

	/**
	 * "Certificate for user {0} has expired, you need to substitute it.",
	 * Message.ERROR
	 */
	@Description(explanation = "Inform user that the user certificate has expired.")
	JEMC201E(201, "Certificate for user {0} has expired on {1}, you need to substitute it.", Message.ERROR),

	/**
	 * "Using socket interceptor for autentication", Message.INFO
	 */
	@Description(explanation = "Inform that nodes are connecting via socket interceptor.")
	JEMC202I(202, "Using socket interceptor for autentication", Message.INFO),

	/**
	 * "request/response on login phase {0}", Message.INFO
	 */
	@Description(explanation = "print the protocol during login phase.")
	JEMC203I(203, "request/response on login phase\n {0}", Message.INFO),

	/**
	 * "Error while decrypting password", Message.ERROR
	 */
	@Description(explanation = "Error while decrypting password")
	JEMC204E(204, "Error while decrypting password", Message.ERROR),

	/**
	 * "User private key need password that has not been provided",
	 * Message.ERROR
	 */
	@Description(explanation = "Error while decrypting password")
	JEMC205E(205, "User private key need password that has not been provided", Message.ERROR),

	/**
	 * ""Unable to get FTp stream for file \"{0}\" with reply code \"{1}\"", Message.ERROR
	 */
	@Description(explanation = "It occurs when it's not able to have FTP stream. Please check if file exists.")
	JEMC206E(206, "Unable to get FTp stream for file \"{0}\" with reply code \"{1}\"", Message.ERROR),
	
	/**
	 * "Unable to load licenses information", Message.ERROR
	 */
	@Description(explanation = "It occurs when is not able to read the licenses, required by user interface. Please check licenses files.")
	JEMC207E(207, "Unable to load licenses information", Message.ERROR),

	/**
	 * "Inconsistent entity: current: {0}, modified {1}", Message.ERROR
	 */
	@Description(explanation = "It occurs when you are trying to modify an entity that someone else has already chanegd and saved.<br>reapply own updates to new instance.")
	JEMC208E(208, "Inconsistent entity: current: {0}, modified {1}", Message.ERROR);
	
	/**
	 * Specific message code identifier for <code>NodeMessage</code>. It will be
	 * displayed in the log.
	 */
	private final static String MESSAGE_ID = "C";

	/**
	 * The {@link Message} created in the constructor corresponding to an
	 * instance of <code>NodeMessage</code>.
	 * 
	 * @see Message
	 */
	private Message message;

	/**
	 * Constructor. It builds a <code>Message</code>. <br>
	 * This method uses the same parameter of the <code>Message</code>
	 * constructor and the specific ID: {@link #MESSAGE_ID}.
	 * 
	 * @param code identifier ID
	 * @param msg string to display. Could contain variables, resolved at
	 *            runtime
	 * @param level severity of log message
	 * @see Message
	 */
	private NodeMessage(int code, String messageContent, int level) {
		this.message = new Message(code, MESSAGE_ID, messageContent, level);
	}

	/**
	 * It returns the {@link Message} corresponding to an
	 * <code>NodeMessage</code> instance.
	 * 
	 * @return the {@link Message} corresponding to an <code>NodeMessage</code>
	 *         instance.
	 */
	@Override
	public Message toMessage() {
		return this.message;
	}
}
