/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.net.InetSocketAddress;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.jar.Attributes;
import java.util.jar.JarFile;

import org.apache.commons.lang3.StringUtils;
import org.hyperic.sigar.ProcCredName;
import org.hyperic.sigar.Sigar;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.configuration.ConfigKeys;
import org.pepstock.jem.node.configuration.ConfigurationException;
import org.pepstock.jem.node.executors.ExecutionResult;
import org.pepstock.jem.node.executors.GenericCallBack;
import org.pepstock.jem.node.executors.nodes.Drain;
import org.pepstock.jem.node.executors.nodes.Start;
import org.pepstock.jem.util.Parser;

import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.query.SqlPredicate;

/**
 * A set of methods to manage the shared map with all nodes of cluster
 * 
 * @author Andrea "Stock" Stocchero
 * 
 * @see org.pepstock.jem.node.NodeInfo
 */
public class NodeInfoUtility {
	
	private static Sigar SIGAR = new Sigar();

	/**
	 * Factory creates a NodeInfo copying a set of information from Member
	 * object of Hazelcast framework. NodeInfo will use Uuid of Member as the
	 * key.
	 * 
	 * @see org.pepstock.jem.node.NodeInfo
	 * @param member member object of Hazelcast framework
	 * @param info node info to load
	 * @throws Exception
	 */
	public static final void loadNodeInfo(Member member, NodeInfo info) throws Exception {
		JarFile jarFile = null;
		try {
			jarFile = new JarFile(new File(Main.class.getProtectionDomain().getCodeSource().getLocation().toURI()));
			Attributes at = (Attributes) jarFile.getManifest().getAttributes(ConfigKeys.JEM_MANIFEST_SECTION);
			String jemVersion = at.getValue(ConfigKeys.JEM_MANIFEST_VERSION);
			if (jemVersion != null)
				info.setJemVersion(jemVersion);
		} catch (IOException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} catch (URISyntaxException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} finally {
			if (jarFile != null) {
				try {
					jarFile.close();
				} catch (IOException e) {
				}
			}
		}
		// set uuid of member of hazelcast as key
		info.setKey(member.getUuid());
		// set status starting at the beginning
		info.setStatus(Status.STARTING);
		// no supernode (only web member is supernode)
		info.setSuperNode(member.isLiteMember());

		// for net info of member, loads all info inside of nodeinfo
		// port of RMI will be set later
		InetSocketAddress address = member.getInetSocketAddress();
		info.setPort(address.getPort());
		info.setIpaddress(address.getAddress().getHostAddress());

		// sets label to be displayed by GRS
		info.setLabel(info.getIpaddress() + ":" + info.getPort());

		// sets execution environment
		info.setExecutionEnvironment(Main.EXECUTION_ENVIRONMENT);
		// use JMX to extract current process id
		info.setProcessId(ManagementFactory.getRuntimeMXBean().getName());

		String hostname = StringUtils.substringAfter(info.getProcessId(), "@");
		// info.setHostname(address.getHostName());
		info.setHostname(hostname);

		OperatingSystemMXBean bean = ManagementFactory.getOperatingSystemMXBean();
		info.getNodeInfoBean().setSystemArchitecture(bean.getArch());
		info.getNodeInfoBean().setAvailableProcessors(bean.getAvailableProcessors());
		info.getNodeInfoBean().setSystemName(bean.getName());
		info.getNodeInfoBean().setTotalMemory(SIGAR.getMem().getTotal());
		
		ProcCredName cred = SIGAR.getProcCredName(SIGAR.getPid());
		info.setUser(cred.getUser());
		

		// checks if is super node. if yes, doesn't create a grs node
		if (!member.isLiteMember()) {
			// creates the request lock, necessary to ask for locking to GRS
			// add request to node info and then subscribe nodeinfo to GRS
			info.getRequest().setRequestorId(info.getKey());
			info.getRequest().setRequestorName(info.getLabel());
		}
		info.loaded();
	}

	/**
	 * 
	 * @param address
	 * @param info
	 * @throws ConfigurationException
	 * 
	 */
	public static synchronized void checkAndStoreNodeInfo(NodeInfo info) throws ConfigurationException {
		IMap<String, NodeInfo> members_map = Main.HAZELCAST.getMap(Queues.NODES_MAP);

		StringBuffer sb = new StringBuffer();
		sb.append("label = '").append(info.getLabel()).append("'");

		Collection<NodeInfo> nodeInfos = members_map.values(new SqlPredicate(sb.toString()));

		try {
			members_map.lock(info.getKey());
			if (!nodeInfos.isEmpty()) {
				for (NodeInfo prevNodeInfo : nodeInfos) {
					members_map.remove(prevNodeInfo.getKey());
				}
			}
			members_map.put(info.getKey(), info);

			boolean checkVersion = Parser.parseBoolean(System.getProperty(ConfigKeys.JEM_CHECK_VERSION), false);
			// check if node release version inside the cluster are different
			Collection<NodeInfo> allNodes = members_map.values();
			for (NodeInfo currNodeInfo : allNodes) {
				if (!currNodeInfo.isSuperNode() && !currNodeInfo.getJemVersion().equals(info.getJemVersion())) {
					if (checkVersion) {
						throw new ConfigurationException(NodeMessage.JEMC191E.toMessage().getFormattedMessage());
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC185W);
						break;
					}
				}
			}
		} catch (Exception ex) {
			if (ex instanceof ConfigurationException) {
				throw (ConfigurationException) ex;
			} else {
				LogAppl.getInstance().emit(NodeMessage.JEMC174E, ex);
			}
		} finally {
			members_map.unlock(info.getKey());
		}
	}

	/**
	 * 
	 * @param address
	 * @param info
	 */
	public static synchronized void storeNodeInfo(NodeInfo info) {
		IMap<String, NodeInfo> members_map = Main.HAZELCAST.getMap(Queues.NODES_MAP);

		try {
			members_map.lock(info.getKey());
			if (!members_map.containsKey(info.getKey()))
				members_map.put(info.getKey(), info);
			else
				members_map.replace(info.getKey(), info);
		} catch (Exception ex) {
			LogAppl.getInstance().emit(NodeMessage.JEMC174E, ex);
		} finally {
			members_map.unlock(info.getKey());
		}
	}

	/**
	 * 
	 * @param statusList the list of the status used to filtered the NODES_MAP
	 * @param notIn if set to true the filtering will be for status Not In the
	 *            list of status passed as parameter
	 * @param includeSupernode if set to true also the supernode will be
	 *            considered
	 * @return a List of NodeInfo present in the Queues.NODES_MAP (eventually an
	 *         empty list) filtered by status present in the statusList passed
	 *         as parameter. If the status List is null or empty return all the
	 *         NodeInfo present in the map
	 */
	public static List<NodeInfo> getNodesInfoByStatus(List<Status> statusList, boolean notIn, boolean includeSupernode) {
		List<NodeInfo> nodesInfo = new ArrayList<NodeInfo>();
		IMap<String, NodeInfo> members_map = Main.HAZELCAST.getMap(Queues.NODES_MAP);

		try {
			members_map.lockMap(10, TimeUnit.SECONDS);
			Iterator<NodeInfo> it = members_map.values().iterator();
			while (it.hasNext()) {
				NodeInfo currNodeInfo = it.next();
				if (includeSupernode && currNodeInfo.isSuperNode()) {
					nodesInfo.add(it.next());
				}
				if (!currNodeInfo.isSuperNode()) {
					nodesInfo.add(it.next());
				}
			}
		} catch (Exception ex) {
			LogAppl.getInstance().emit(NodeMessage.JEMC174E, ex);
		} finally {
			members_map.unlockMap();
		}

		if (statusList == null || statusList.isEmpty()) {
			return nodesInfo;
		}
		List<NodeInfo> nodesInfoToReturn = new ArrayList<NodeInfo>();
		for (int j = 0; j < nodesInfo.size(); j++) {
			NodeInfo currNodeInfo;
			currNodeInfo = nodesInfo.get(j);
			if (statusList.contains(currNodeInfo.getStatus()) && !notIn) {
				nodesInfoToReturn.add(currNodeInfo);
			}
			if (!statusList.contains(currNodeInfo.getStatus()) && notIn) {
				nodesInfoToReturn.add(currNodeInfo);
			}
		}
		return nodesInfoToReturn;
	}

	/**
	 * Drains the node
	 */
	public static void drain() {
		DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Drain(), Main.HAZELCAST.getCluster().getLocalMember());
		ExecutorService executorService = Main.HAZELCAST.getExecutorService();
		task.setExecutionCallback(new GenericCallBack());
		executorService.execute(task);
	}

	/**
	 * Drains the node
	 */
	public static void start() {
		DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Start(), Main.HAZELCAST.getCluster().getLocalMember());
		ExecutorService executorService = Main.HAZELCAST.getExecutorService();
		task.setExecutionCallback(new GenericCallBack());
		executorService.execute(task);
	}
}