/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
    
 Linking JEM, the BEE statically or dynamically with other modules is making a combined work based on JEM, the BEE. 
 Thus, the terms and conditions of the GNU General Public License cover the whole combination.

 As a special exception, the copyright holders of JEM, the BEE give you permission to combine JEM, the BEE program with 
 free software programs or libraries that are released under the GNU LGPL and with independent modules 
 that communicate with JEM, the BEE solely through the org.pepstock.jem.node.NodeInfo interface. 
 You may copy and distribute such a system following the terms of the GNU GPL for JEM, the BEE and the licenses 
 of the other code concerned, provided that you include the source code of that other code when and as 
 the GNU GPL requires distribution of source code and provided that you do not modify the 
 org.pepstock.jem.node.NodeInfo interface.

 Note that people who make modified versions of JEM, the BEE are not obligated to grant this special exception
 for their modified versions; it is their choice whether to do so. The GNU General Public License
 gives permission to release a modified version without this exception; this exception also makes it
 possible to release a modified version which carries forward this exception. If you modify the 
 org.pepstock.jem.node.NodeInfo interface, this exception does not apply to your modified version of 
 JEM, the BEE, and you must remove this exception when you distribute your modified version.

 This exception is an additional permission under section 7 of the GNU General Public License, version 3
 (GPLv3)  
   
*/
package org.pepstock.jem.node;

import java.io.Serializable;
import java.util.Date;
import java.util.Properties;
import java.util.concurrent.locks.ReentrantLock;

import org.pepstock.jem.Job;
import org.pepstock.jem.NodeInfoBean;

import com.google.gwt.user.client.rpc.GwtTransient;

/**
 * Represents a node of cluster. it uses many information of Hazelcast member
 * class. Extends GRS node so it's able to manage locking on global resource
 * system.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class NodeInfo implements Serializable {

	private static final long serialVersionUID = 1L;
	
	/**
	 * Default version if no one is found inside the manifest
	 */
	public static final String UNKNOWN_VERSION = "Unknown";

	private String hostname = null;

	private String ipaddress = null;

	private String key = null;

	private String label = null;

	private int port = Integer.MIN_VALUE;

	private int rmiPort = Integer.MIN_VALUE;

	private String processId = null;
	
	private String user = null;

	private Status status = Status.UNKNOWN;;

	private Job job = null;

	private boolean isSuperNode = false;

	private NodeInfoBean nodeInfoBean = new NodeInfoBean();

	private ExecutionEnvironment executionEnvironment = null;

	private Date startedTime = new Date();

	private String jemVersion = UNKNOWN_VERSION;

	@GwtTransient
	private RequestLock request = new DefaultRequestLock();

	private boolean isOperational = true;

	private final ReentrantLock lock = new ReentrantLock();

	/**
	 * Constructs the node info object
	 */
	public NodeInfo() {
	}

	/**
	 * Called before to load all data of node
	 * 
	 * @throws Exception if exception occurs
	 */
	public void loaded() throws Exception {

	}

	/**
	 * Called to initialize the node. A set of properties are passed, or a empty
	 * collection if the properties are not defined
	 * 
	 * @param properties properties
	 * @throws Exception if exception occurs
	 */
	public void init(Properties properties) throws Exception {

	}

	/**
	 * Gets unique key to identify the node inside the cluster.
	 * 
	 * @return return the key
	 */
	public final String getKey() {
		return key;
	}

	/**
	 * Sets unique key to identify the node inside the cluster.
	 * 
	 * @param key the key
	 */
	public final void setKey(String key) {
		this.key = key;
	}

	/**
	 * Gets readable name of node, used on displays or logs.
	 * 
	 * @return readable name of node
	 */
	public final String getLabel() {
		return label;
	}

	/**
	 * Sets readable name of node, used on displays or logs.
	 * 
	 * @param label readable name
	 */
	public final void setLabel(String label) {
		this.label = label;
	}

	/**
	 * Returns the ipaddress of machine where the node is running
	 * 
	 * @return ipaddress of machine
	 */
	public final String getIpaddress() {
		return ipaddress;
	}

	/**
	 * Sets the ipaddress and port number of machine where the node is running
	 * 
	 * @param ipaddress ipaddress of machine
	 */
	public final void setIpaddress(String ipaddress) {
		this.ipaddress = ipaddress;
	}

	/**
	 * Returns the port number what the node is listening on
	 * 
	 * @return the port number what the node is listening on
	 */
	public final int getPort() {
		return port;
	}

	/**
	 * Sets the port number what the node is listening on
	 * 
	 * @param port the port number what the node is listening on
	 */
	public final void setPort(int port) {
		this.port = port;
	}

	/**
	 * Returns the RMI port number what the node is listening on
	 * 
	 * @return the RMI port number what the node is listening on
	 */

	public final int getRmiPort() {
		return rmiPort;
	}

	/**
	 * Sets the RMI port number what the node is listening on
	 * 
	 * @param rmiPort the RMI port number what the node is listening on
	 */

	public void setRmiPort(int rmiPort) {
		this.rmiPort = rmiPort;
	}

	/**
	 * Returns the hostname of machine where the node is running
	 * 
	 * @return hostname of machine
	 */
	public final String getHostname() {
		return hostname;
	}

	/**
	 * Sets the hostname of machine where the node is running
	 * 
	 * @param hostname hostname of machine
	 */
	public final void setHostname(String hostname) {
		this.hostname = hostname;
	}

	/**
	 * Returns the current status of node
	 * 
	 * @see org.pepstock.jem.node.Status#Status(int, String)
	 * @return current status of node
	 */
	public final Status getStatus() {
		return status;
	}

	/**
	 * Sets the current status of node
	 * 
	 * @see org.pepstock.jem.node.Status#Status(int, String)
	 * @param status current status of node
	 */
	public final void setStatus(Status status) {
		this.status = status;
	}

	/**
	 * Returns the current job which is running managed by node
	 * 
	 * @see org.pepstock.jem.Job#Job()
	 * @return current job
	 */
	public final Job getJob() {
		return job;
	}

	/**
	 * Sets the current job which is running managed by node
	 * 
	 * @see org.pepstock.jem.Job#Job()
	 * @param job current job
	 */
	public final void setJob(Job job) {
		this.job = job;
	}

	/**
	 * Sets the super-node mode. This is the concept implemented by Hazelcast
	 * framework. Super Clients (super node for JEM) are members with no
	 * storage. The JVM will join the cluster as a 'super client' which will not
	 * be a 'data partition' (no data on that node) but will have super fast
	 * access to the cluster just like any regular member does. Inside JEM, only
	 * the web container is a super-node.
	 * 
	 * @param isSuperNode <code>true</code> if is JEM web container,
	 *            <code>false</code> otherwise
	 */
	public final void setSuperNode(boolean isSuperNode) {
		this.isSuperNode = isSuperNode;
	}

	/**
	 * Returns the supernode mode. This is the concept implemented by Hazelcast
	 * framework. A supernode is a member of cluster where no data are stored.
	 * Inside JEM, only the web container is a supernode.
	 * 
	 * @return <code>true</code> if is JEM web container, <code>false</code>
	 *         otherwise
	 */
	public final boolean isSuperNode() {
		return isSuperNode;
	}

	/**
	 * Returns the process id, created to execute the job. JMX method is used
	 * that returns the process id in PID@HOSTNAME format
	 * 
	 * @return process id in pid@name format
	 */
	public final String getProcessId() {
		return processId;
	}

	/**
	 * Sets the process id, created to execute the job. JMX method is used that
	 * returns the process id in PID@HOSTNAME format
	 * 
	 * @param processId process id in pid@name format
	 */
	public final void setProcessId(String processId) {
		this.processId = processId;
	}

	/**
	 * Returns the environment, domain and affinity information for the node
	 * 
	 * @return the executionEnviroment
	 */
	public final ExecutionEnvironment getExecutionEnvironment() {
		return executionEnvironment;
	}

	/**
	 * Sets the environment, domain and affinity information for the node
	 * 
	 * @param executionEnvironment the executionEnviroment to set
	 */
	public final void setExecutionEnvironment(ExecutionEnvironment executionEnvironment) {
		this.executionEnvironment = executionEnvironment;
	}

	/**
	 * Returns teh started time of node
	 * 
	 * @return the startedTime
	 */
	public final Date getStartedTime() {
		return startedTime;
	}

	/**
	 * Sets the started time of node
	 * 
	 * @param startedTime the startedTime to set
	 */
	public final void setStartedTime(Date startedTime) {
		this.startedTime = startedTime;
	}

	/**
	 * @return the request
	 */
	public RequestLock getRequest() {
		return request;
	}

	/**
	 * @param request the request to set
	 */
	public void setRequest(RequestLock request) {
		this.request = request;
	}

	/**
	 * @return the isOperational
	 */
	public boolean isOperational() {
		return isOperational;
	}

	/**
	 * @param isOperational the isOperational to set
	 */
	public void setOperational(boolean isOperational) {
		this.isOperational = isOperational;
	}

	/**
	 * Unlocks all resources previously locked in case of node failover.
	 */
	public void unlockForFailover() {
		if (request != null)
			getRequest().unlock();
	}

	/**
	 * Returns a string with a message that this feature is not supported.
	 * 
	 * @return all locks info
	 */
	public String displayRequestors() {
		return "No locks information is available with this kind of node.";
	}

	/**
	 * Returns a string with a message that this feature is not supported.
	 * 
	 * @param resourceKey resource name (key)
	 * @return all locks info for resource name
	 */
	public String displayRequestors(String resourceKey) {
		return displayRequestors();
	}


	/**
	 * @return the jemVersion present in the manifest of the jem.jar
	 */
	public final String getJemVersion() {
		return jemVersion;
	}

	/**
	 * @param jemVersion the jemVersion present in the manifest of the jem.jar
	 */
	public final void setJemVersion(String jemVersion) {
		this.jemVersion = jemVersion;
	}
	
	/**
	 * @return the user
	 */
	public final String getUser() {
		return user;
	}

	/**
	 * @param user the user to set
	 */
	public final void setUser(String user) {
		this.user = user;
	}

	/**
	 * @return the lock
	 */
	public ReentrantLock getLock() {
		return lock;
	}

	/**
	 * Returns the bean which represents the node. Necessary for serialization
	 * reasons to web interface. <br>
	 * 
	 * @return the nodeInfoBean
	 */
	public final NodeInfoBean getNodeInfoBean() {
		nodeInfoBean.setKey(this.getKey());
		nodeInfoBean.setLabel(this.getLabel());
		nodeInfoBean.setHostname(this.getHostname());
		nodeInfoBean.setIpaddress(this.getIpaddress());
		if (this.getJob() != null) {
			nodeInfoBean.setJobName(this.getJob().getName());
		}
		nodeInfoBean.setSuperNode(this.isSuperNode());
		nodeInfoBean.setPort(this.getPort());
		nodeInfoBean.setProcessId(this.getProcessId());
		nodeInfoBean.setRmiPort(this.getRmiPort());
		nodeInfoBean.setStatus(this.getStatus().getDescription());
		nodeInfoBean.setExecutionEnvironment(this.getExecutionEnvironment());
		nodeInfoBean.setStartedTime(startedTime);
		nodeInfoBean.setOperational(this.isOperational());
		nodeInfoBean.setJemVersion(this.getJemVersion());
		return nodeInfoBean;
	}


}