/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.HashMap;

import org.pepstock.jem.factories.JemFactory;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.affinity.AffinityLoader;
import org.pepstock.jem.node.configuration.ConfigKeys;
import org.pepstock.jem.node.configuration.ConfigurationException;
import org.pepstock.jem.node.events.JobLifecycleListenersSystem;
import org.pepstock.jem.node.resources.custom.ResourceDefinition;
import org.pepstock.jem.node.resources.custom.ResourceDefinitionsManager;
import org.pepstock.jem.node.tasks.platform.CurrentPlatform;
import org.pepstock.jem.util.Parser;
import org.pepstock.jem.util.rmi.RegistryContainer;

import com.hazelcast.config.FileSystemXmlConfig;
import com.hazelcast.core.HazelcastInstance;

/**
 * This is the main class of JEM node. Starts all sub component and stays in
 * wait. Contains all common static reference that all can use.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class Main {

	/**
	 * If <code>true</code>, this instance of node is the oldest one of cluster.<br>
	 * <br>
	 * Hazelcast cluster works as following:<br>
	 * 1. every member in the cluster has the same member list in the same
	 * order.<br>
	 * 2. first member is the oldest member so if the oldest member dies, second
	 * member in the list becomes the first member in the list and the new
	 * oldest member.<br>
	 * <br>
	 * The "COORDINATOR" role is important because if it's <code>true</code>, it
	 * does a lot of things for whole cluster.
	 */
	public static boolean IS_COORDINATOR = false;

	/**
	 * This is static reference of current execution environment of node
	 * instance.
	 * 
	 * @see org.pepstock.jem.node.ExecutionEnvironment
	 */
	public final static ExecutionEnvironment EXECUTION_ENVIRONMENT = new ExecutionEnvironment();
	
	/**
	 * This is static refenrece of affinity loader define in JEM config.
	 * Could be null, if there is any affinity loader configuration in JEM config.
	 * 
	 * @see org.pepstock.jem.node.affinity.AffinityLoader
	 */
	public static AffinityLoader AFFINITY_LOADER = null; 

	/**
	 * This is static reference of current input queue manager of node instance.
	 * 
	 * @see org.pepstock.jem.node.InputQueueManager
	 */
	public static InputQueueManager INPUT_QUEUE_MANAGER = null;

	/**
	 * This is static reference of current node information object.
	 * 
	 * @see org.pepstock.jem.node.NodeInfo
	 */
	public static NodeInfo NODE = null;

	/**
	 * This is static reference of current job task in execution, or null if
	 * node status is not ACTIVE or DRAINING
	 * 
	 * @see org.pepstock.jem.node.CancelableTask
	 * @see org.pepstock.jem.node.Status
	 */
	public static CancelableTask CURRENT_TASK = null;

	/**
	 * This is static reference of current output system
	 * 
	 * @see org.pepstock.jem.node.OutputSystem
	 */
	public static OutputSystem OUTPUT_SYSTEM = null;

	/**
	 * This is static reference of map where node has loaded all configured
	 * factories.<br>
	 * The key if the returned value of "getType" method of JemFactory
	 * 
	 * @see org.pepstock.jem.factories.JemFactory#getType()
	 */
	public final static HashMap<String, JemFactory> FACTORIES_LIST = new HashMap<String, JemFactory>();

	/**
	 * Contains all listeners configured which will be called when the job changes its status.
	 * 
	 * @see org.pepstock.jem.node.events.JobLifecycleListenersSystem
	 * @see org.pepstock.jem.node.events.JobLifecycleListener
	 */
	public static final JobLifecycleListenersSystem JOB_LIFECYCLE_LISTENERS_SYSTEM = new JobLifecycleListenersSystem();

	/**
	 * Contains all the custom resource definitions defined by the user, and has the method {@link ResourceDefinitionsManager#loadCustomResourceDefinition(org.pepstock.jem.node.configuration.CustomResourceDefinition, String)}
	 * to load them.
	 * 
	 * @see ResourceDefinitionsManager
	 * @see ResourceDefinition
	 */
	public static final ResourceDefinitionsManager CUSTOM_RESOURCE_DEFINITION_MANAGER = new ResourceDefinitionsManager();

	/**
	 * Saves Hazelcast instance to use on all classes which must access to Hazelcast objects. 
	 */
	public static HazelcastInstance HAZELCAST = null;

	/**
	 * Saves Hazelcast configuration because it must be read from DB managers for some keys to encrypt DB 
	 */
	public static FileSystemXmlConfig HAZELCAST_CONFIG = null;

	/**
	 * is the collector of all statistics. Acts only if node is the coordinator 
	 */
	public static StatisticsManager STATISTICS_MANAGER = null;

	/**
	 * Sets <code>true</code> if the node is started in maintainance.
	 */
	public static boolean IS_ACCESS_MAINT = false;
	
	/**
	 * Sets <code>true</code> if the hook shutdown thread is started, to shutdown the node.
	 */
	public static boolean IS_SHUTTING_DOWN = false;
	
	/**
	 * Count of JCL check done by this node.
	 */
	public static long NUMBER_OF_JCL_CHECK = 0;
	
	/**
	 * Count of JOB submission done by this node.
	 */
	public static long NUMBER_OF_JOB_SUBMITTED = 0;
	
	/**
	 * Count of JOB submission done by this node.
	 */
//	public static long START = 0;
	
	// all nodes of cluster must check and validate jcls and prejobs in
	// JCL_CHECKING queue
	private JclCheckingQueueManager jclChecker = new JclCheckingQueueManager();
	
	// submitter starts here
	private Submitter submitter = new Submitter();

	/**
	 * Constructs JEM node, initializing all sub components
	 */
	public Main() {
		try {
			CurrentPlatform.getInstance();
			
			// reads properties to check if access maint
			IS_ACCESS_MAINT = Parser.parseBoolean(System.getProperty(ConfigKeys.JEM_ACCESS_MAINT), false);
			if (IS_ACCESS_MAINT){
				LogAppl.getInstance().emit(NodeMessage.JEMC189I);
			}
			
			// starts the configuration loading and sub component instantiation
			StartUpSystem.run();
			// starts a RMI regirty container for objects necessary from job
			// execution
			// uses the TCP port defined for Hazelcast cluster, adding a
			// constant number 200
			int objectPort = Main.NODE.getPort() + 200;
			LogAppl.getInstance().emit(NodeMessage.JEMC013I, String.valueOf(objectPort));
			RegistryContainer.createInstance(objectPort);
			LogAppl.getInstance().emit(NodeMessage.JEMC014I);
			Main.NODE.setRmiPort(objectPort);
	
			Main.STATISTICS_MANAGER.init();
			
			NodeInfoUtility.checkAndStoreNodeInfo(Main.NODE);
			
			// input manager starts here
			INPUT_QUEUE_MANAGER = new InputQueueManager();
		
			// sets shutdown hook passing the threads to interrupt
			Runtime.getRuntime().addShutdownHook(new ShutDownHandler(jclChecker, submitter));

			// and at the end, wait this thread...
			waitState();

		} catch (ConfigurationException e1) {
			// occurs when we have some misconfiguration. Node ends
			LogAppl.getInstance().emit(NodeMessage.JEMC006E, e1);
			System.exit(12);
		} catch (RemoteException e1) {
			// occurs when RMI registry is not able to start. Node ends
			LogAppl.getInstance().emit(NodeMessage.JEMC007E, e1);
			System.exit(12);
		} catch (Exception e1) {
			// occurs when the platform is not supported
			LogAppl.getInstance().emit(NodeMessage.JEMC194E, e1);
			System.exit(12);
		}
	}

	/**
	 * Checks if is running inside of Eclipse (to do it, set a RUNNING_IN_ECLIPSE env variable to true), otherwise
	 * stays in wait mode
	 */
	public synchronized void waitState() {
		// Starts really here jcl checking
		jclChecker.start();
	
		// Starts really here submitter
		submitter.start();

		// checks if the variable is set understanding if is in Eclipse or not
		if (Boolean.parseBoolean(System.getenv("RUNNING_IN_ECLIPSE")) == true) {
			// writes a message
    		System.err.println("You're using Eclipse; click in this console and	" +
    						"press ENTER to call System.exit() and run the shutdown routine.");
    		try {
    			// opens a input stream waiting for an input
    			System.in.read();
    		} catch (IOException e) {
    			e.printStackTrace();
    		}
    	} else {
    		try {
    			// waits 
    			while(true){
    				wait();
    			}
    		} catch (InterruptedException e) {
    		}
    	}
		// here starts shutdown hook
		System.exit(0);
	}

	/**
	 * Main method to start!
	 * 
	 * @param args null at the moment
	 */
	public static void main(String[] args) {
		System.setProperty(ConfigKeys.IPV4, Boolean.TRUE.toString());
		new Main();
	}

}
