/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.locks.Lock;

import org.pepstock.jem.Jcl;
import org.pepstock.jem.Job;
import org.pepstock.jem.log.LogAppl;

import com.hazelcast.core.EntryEvent;
import com.hazelcast.core.EntryListener;
import com.hazelcast.core.IMap;

/**
 * Manages all activities related to input queue. It's able to listen when new
 * jobs are put in input queue and checks (using environment, domain and
 * affinity) if these jobs could be executed by itself.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class InputQueueManager implements EntryListener<String, Job>, ShutDownInterface{
	
	private LinkedBlockingQueue<Job> queue = null;

	private JobComparator comparator = new JobComparator();

	private InputQueuePredicate predicate = new InputQueuePredicate();
	/**
	 * Constructs the manager. It creates a internal queue on which it moves the
	 * job to execute. It defines itself as a listener of input queue.
	 * Furthermore it creates the right SQLPredicate to extract the job which
	 * could be executed here. Where conditions are:<br>
	 * <p>
	 * 1. Hold not true<br>
	 * 2. Environment equals to
	 * org.pepstock.jem.node.ExecutionEnvironment.getEnvironment()<br>
	 * 3. Domain equals to
	 * org.pepstock.jem.node.ExecutionEnvironment.getDomain() or Job.DEFAULT_DOMAIN<br>
	 * 4. Affinity equals to
	 * org.pepstock.jem.node.ExecutionEnvironment.getAffinity() or Job.DEFAULT_AFFINITY<br>
	 * </p>
	 * It sorts by priority on descending mode.
	 * 
	 * @see org.pepstock.jem.node.ExecutionEnvironment#getEnvironment()
	 * @see org.pepstock.jem.node.ExecutionEnvironment#getDomain()
	 * @see org.pepstock.jem.node.ExecutionEnvironment#getAffinity()
	 * @see org.pepstock.jem.Job#isHold()
	 * @see org.pepstock.jem.Job#getPriority()
	 */
	public InputQueueManager() {
		queue = new LinkedBlockingQueue<Job>();
		IMap<String, Job> inputQueue = Main.HAZELCAST.getMap(Queues.INPUT_QUEUE);
		inputQueue.addEntryListener(this, true);

		predicate.setExecutionEnviroment(Main.EXECUTION_ENVIRONMENT);
	}

	/**
	 * Returns the job object move on internal queue and remove on it. This job
	 * must be executed.
	 * 
	 * @return job instance to execute
	 * @throws InterruptedException if interrupted while waiting
	 */
	public Job takeJob() throws InterruptedException {
		return queue.take();
	}

	/**
	 * Checks if there is any job on input queue to execute.
	 */
	public synchronized void checkJobToSubmit() {
		
		if (Main.IS_ACCESS_MAINT){
			if (!Main.NODE.getStatus().equals(Status.DRAINED)){
				LogAppl.getInstance().emit(NodeMessage.JEMC189I);
				NodeInfoUtility.drain();
			}
			return;
		}
		
		Main.NODE.getLock().lock();
		try {
			// if node is drained or draining, it doesn't execute anything
			if (Main.NODE.getStatus().equals(Status.DRAINED) || Main.NODE.getStatus().equals(Status.DRAINING) 
					|| Main.NODE.getStatus().equals(Status.ACTIVE))
				return;

			// set Active status and store teh info about the node on Hazelcast map
			Main.NODE.setStatus(Status.ACTIVE);
			NodeInfoUtility.storeNodeInfo(Main.NODE);
			LogAppl.getInstance().emit(NodeMessage.JEMC030I, Main.NODE.getStatus());
		} finally {
			Main.NODE.getLock().unlock();
		}

		// get jobs which could be executed by this node, using teh SQL
		// predicate prepared on constructor
		IMap<String, Job> inputQueue = Main.HAZELCAST.getMap(Queues.INPUT_QUEUE);
		// TODO Mettere READ/WRITE locks se server
		Collection<Job> jobs = inputQueue.values(predicate);
		// it does if there is at least a job compliant with Sql predicate
		if (!jobs.isEmpty()) {
			// creates a new collection and sort with job comparator which uses
			// the priority
			ArrayList<Job> queuedJobs = new ArrayList<Job>(jobs);
			Collections.sort(queuedJobs, comparator);
			for (Job job : queuedJobs) {

				// set mustReturn because I'm not sure if some other node has
				// already change the list of jobs
				// set to true when it has teh job to execute
				boolean mustReturn = false;

				// locks Input queue of hazelcast to avoid multiple access to
				// the map
				try {
					inputQueue.lock(job.getId());
					// checks if the job is still in input queue (I'm not
					// sure if some other node has already change the list
					// of jobs)
					if (inputQueue.containsKey(job.getId())) {
						IMap<String, Job> runningQueue = Main.HAZELCAST.getMap(Queues.RUNNING_QUEUE);

						// set member key and label (this node), and started date into job
						job.setMemberId(Main.NODE.getKey());
						job.setMemberLabel(Main.NODE.getLabel());
						job.setStartedTime(new Date());

						job.setRunningStatus(Job.RUNNING);
						// move the job to internal queue and from INPUT
						// queue to RUNNING queue
						// TODO Mettere READ/WRITE locks se server
						try{
							runningQueue.lock(job.getId());
							Job storedJob = inputQueue.remove(job.getId());

							// if storedJob is null means
							// that job is not longer in QUEUE. Why?
							// see issue #270. Could be a tryLock problem (hashcode for KEY?)
							if (storedJob != null){
								queue.put(job);
								runningQueue.put(job.getId(), job);
								// job found! I can return
								mustReturn = true;
								LogAppl.getInstance().emit(NodeMessage.JEMC018I, job.toString());
							} 
						} catch (Exception ex){
							LogAppl.getInstance().emit(NodeMessage.JEMC017E, ex);
						} finally {
							runningQueue.unlock(job.getId());
						}
					}
				} finally {
					inputQueue.unlock(job.getId());
				}
				// if I have the job to execute, then return otherwise try with
				// another job, if there is
				if (mustReturn)
					return;
			}
		}
		// change status of node in Inactive. means that there is not nay job in
		// queue to execute by this node
		Main.NODE.setStatus(Status.INACTIVE);
		NodeInfoUtility.storeNodeInfo(Main.NODE);
		LogAppl.getInstance().emit(NodeMessage.JEMC030I, Main.NODE.getStatus());
	}

	/**
	 * Checks and move the job on routing queue because the environment
	 * attribute is not equals to cluster definition
	 * 
	 * @param job job instance to check
	 * @return <code>true</code> if moved on routing queue, <code>false</code>
	 *         otherwise
	 */
	public synchronized boolean isRouted(Job job) {
		boolean isRouted = false;

		// check the environment value of job with environment of node
		// if they are not equals, job is moved to routing queue
		Jcl jcl = job.getJcl();
		if (!jcl.getEnvironment().equalsIgnoreCase(Main.EXECUTION_ENVIRONMENT.getEnvironment())) {

			// locks Input queue of hazelcast to avoid multiple access to the
			// map
			// TODO Mettere READ/WRITE locks se server
			Lock lock = Main.HAZELCAST.getLock(Queues.INPUT_QUEUE);
			lock.lock();
			try {

				// checks if the job is still in input queue (I'm not sure if
				// some other node has already change the list of jobs)
				IMap<String, Job> inputQueue = Main.HAZELCAST.getMap(Queues.INPUT_QUEUE);
				if (inputQueue.containsKey(job.getId())) {
					IMap<String, Job> routingQueue = Main.HAZELCAST.getMap(Queues.ROUTING_QUEUE);

					// move the job from INPUT queue to ROUTING queue
					inputQueue.remove(job.getId());
					routingQueue.put(job.getId(), job);

					// set return value
					isRouted = true;
					LogAppl.getInstance().emit(NodeMessage.JEMC019I, job.toString(), jcl.getEnvironment());
				}
			} finally {
				lock.unlock();
			}
		}
		return isRouted;
	}

	/**
	 * Listen if there is new job in input queue Checks if routed otherwise
	 * checks to submit by this node
	 */
	@Override
	public void entryAdded(EntryEvent<String, Job> event) {
		// if shutting down or access maint, do nothing
		if (Main.IS_SHUTTING_DOWN || Main.IS_ACCESS_MAINT)
			return;
		
		Job job = event.getValue();

		// if not routed and status is INACTIVE (no job is executing), it can
		// check if new job could be executed by this node
		if (!isRouted(job)) {
			if (Main.NODE.getStatus().equals(Status.INACTIVE)) {
				checkJobToSubmit();
			}
		}
	}

	/**
	 * Not implemented because not necessary
	 */
	@Override
	public void entryEvicted(EntryEvent<String, Job> event) {
	}

	/**
	 * Not implemented because not necessary
	 */
	@Override
	public void entryRemoved(EntryEvent<String, Job> event) {
	}

	/**
	 * Listen if there is a update on job in input queue The users could change
	 * the environment, domain, affinity or hold job attributes which can affect
	 * this node Checks if routed otherwise checks to submit by this node
	 */
	@Override
	public void entryUpdated(EntryEvent<String, Job> event) {
		// if shutting down or access maint, do nothing
		if (Main.IS_SHUTTING_DOWN || Main.IS_ACCESS_MAINT)
			return;
	
		Job job = event.getValue();

		// if not routed and status is INACTIVE (no job is executing), it can
		// check if job could be executed by this node
		if (!isRouted(job)) {
			if (Main.NODE.getStatus().equals(Status.INACTIVE)) {
				checkJobToSubmit();
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.node.ShutDownInterface#shutdown()
	 */
	@Override
	public void shutdown() throws Exception {
		queue.put(ShutDownJob.JOB);
	}
}