/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.node;

import java.io.Serializable;
import java.util.LinkedList;

import org.pepstock.jem.Jcl;

/**
 * Contains the environment, domain and affinity information for the node. They
 * are configured in JEM configuration file inside the
 * <code>&lt;environment&gt;</code>, <code>&lt;domain&gt;</code> and
 * <code>&lt;affinity&gt;</code> elements.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class ExecutionEnvironment implements Serializable {

	private static final long serialVersionUID = 1L;

	private String environment = null;

	private String domain = Jcl.DEFAULT_DOMAIN;
	
	private LinkedList<String> staticAffinities = new LinkedList<String>();
	
	private LinkedList<String> dynamicAffinities = new LinkedList<String>();
	
	private int memory = Jcl.DEFAULT_MEMORY;
	
	/**
	 * Empty constructor
	 */
	public ExecutionEnvironment() {
	}

	/**
	 * Returns the name of cluster (see Hazelcast "group" definition) that the
	 * node uses to submit jobs.
	 * 
	 * @return the name of cluster
	 */
	public String getEnvironment() {
		return environment;
	}

	/**
	 * Sets the name of cluster that the node uses to submit jobs.
	 * 
	 * @param environment the name of cluster
	 */
	public void setEnvironment(String environment) {
		this.environment = environment;
	}

	/**
	 * Returns the name of domain. Domain is a subset of nodes that the node
	 * uses to submit jobs. Default is Job.DEFAULT_DOMAIN
	 * 
	 * @see org.pepstock.jem.Job#DEFAULT_DOMAIN
	 * @return the name of domain
	 */
	public String getDomain() {
		return domain;
	}

	/**
	 * Sets the name of domain that the node uses to submit jobs.
	 * 
	 * @param domain the name of domain
	 */
	public void setDomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Returns the collections of names of affinity that the node uses to submit jobs
	 * 
	 * @return the list of names name of affinity
	 */
	public LinkedList<String> getAllAffinities() {
		LinkedList<String> allAffinities = new LinkedList<String>();
		allAffinities.addAll(staticAffinities);
		allAffinities.addAll(dynamicAffinities);
		return allAffinities;
	}
	
	/**
	 * @return the staticAffinities
	 */
	public LinkedList<String> getStaticAffinities() {
		return staticAffinities;
	}

	/**
	 * @param staticAffinities the staticAffinities to set
	 */
	public void setStaticAffinities(LinkedList<String> staticAffinities) {
		this.staticAffinities = staticAffinities;
	}

	/**
	 * @return the dynamicAffinities
	 */
	public LinkedList<String> getDynamicAffinities() {
		return dynamicAffinities;
	}

	/**
	 * @param dynamicAffinities the dynamicAffinities to set
	 */
	public void setDynamicAffinities(LinkedList<String> dynamicAffinities) {
		this.dynamicAffinities = dynamicAffinities;
	}

	/**
	 * Returns the memory which will be used by node to start the process
	 * 
	 * @return the memory in MB
	 */
	public int getMemory() {
		return memory;
	}

	/**
	 * Sets the memory which will be used by node to start the process
	 * 
	 * @param memory the memory in MB to set
	 */
	public void setMemory(int memory) {
		this.memory = memory;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "ExecutionEnvironment [environment=" + environment + ", domain=" + domain + ", staticAffinities=" + staticAffinities + ", dynamicAffinities=" + dynamicAffinities + ", memory=" + memory + "]";
	}
}