/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console.commands.resources;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.console.HttpManager;
import org.pepstock.jem.console.JemConsole;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.Resources;
import org.pepstock.jem.node.resources.XmlUtil;

/**
 * Add or replace a resource into JEM.<br>
 * <br>
 * <b>setResources -source ... </b><br>
 * <br>
 * Is possible to have help from command line by <code>-help</code> argument.<br>
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class Set extends ResourceCommand{
	
	private static final String ACTION = org.pepstock.jem.console.commands.resources.Set.class.getName();

	private static final String KEY = "setResources";
	
	private static final String DESCRIPTION = "Adds and updates a resource defintion by resource definition xml file";
	
	/**
	 * Key for the url of resource definition 
	 */
	private static String SOURCE = "source";

	/**
	 * Empty constructor
	 */
	public Set(){
		super(ACTION);
	}
	
	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#execute(java.lang.String)
	 */
	@SuppressWarnings("static-access")
	@Override
	public void execute(String userid, String groupid) throws Exception {
		// -source mandatory arg
		Option source = OptionBuilder.withArgName(SOURCE).hasArg().withDescription("use given url resource definition").create(SOURCE);
		source.setRequired(true);
		ArgumentsParser parser = new ArgumentsParser(this.getClass().getName(), source);
		
		Properties props = parser.parseArg(getArguments());

		// gets resource definition URL
		String source_file = props.getProperty(SOURCE);
		
		// gets URL of JCL content, reads and loads it into Prejob object, setting the JCL type
		URL url = null;
		try {
			url = new URL(source_file);
		} catch (MalformedURLException ex){
			// if it's not a URL, try as a file
			File propsFile = new File(source_file);
			if (!propsFile.isAbsolute()){
				propsFile = new File(JemConsole.CURRENT_DIRECTORY, source_file);
			}
			url = propsFile.toURI().toURL();
		}

		// creates the resource using Xstream to load!
		Object data = XmlUtil.getXStream().fromXML(url.openStream());
		if ((data instanceof Resource) || (data instanceof Resources)){
			Object object = HttpManager.getInstance().execute(COMMON_RESOURCES_QUERY_STRING, getParameters(), data);
			// checks if is added or updated
			Boolean isAdded = (Boolean)object;
			LogAppl.getInstance().emit((isAdded) ? NodeMessage.JEMC052I : NodeMessage.JEMC053I, data.toString());
		} else {
			throw new Exception("Source file doesn't contain any resource or resources object but '"+data.getClass().getName()+"'");
		}
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getCommandKey()
	 */
	@Override
	public String getKey() {
		return KEY;
	}
	
	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getDescription()
	 */
	@Override
	public String getDescription() {
		return DESCRIPTION;
	}
}