/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console.commands.resources;

import java.io.File;
import java.io.FileOutputStream;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.console.HttpManager;
import org.pepstock.jem.console.JemConsole;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.Resources;
import org.pepstock.jem.node.resources.XmlUtil;

import com.thoughtworks.xstream.XStream;

/**
 * Gets a resource into JEM.<br>
 * Is a command (to execute by command line) which accepts 2 arguments: <br>
 * <code>-file [url]</code>, optional argument, indicates the properties file where store resource definition. Default is STD output.<br>
 * <b>GetResourcesList [-file ...]</b><br>
 * <br>
 * Is possible to have help from command line by <code>-help</code> argument.<br>
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class GetList extends ResourceCommand{

	private static final String ACTION = org.pepstock.jem.console.commands.resources.GetList.class.getName();
	
	private static final String KEY = "getResourcesList";
	
	private static final String DESCRIPTION = "Returns the defined resources list";
	
	/**
	 * Key for the properties of resource
	 */
	private static String FILE = "file";
	
	/**
	 * Key for the name of resource 
	 */
	private static String NAME = "name";


	/**
	 * Empty constructor
	 */
	public GetList(){
		super(ACTION);
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#execute(java.lang.String)
	 */
	@SuppressWarnings("static-access")
	@Override
	public void execute(String userid, String groupid) throws Exception {
		// -file mandatory arg
		Option properties = OptionBuilder.withArgName(FILE).hasArg().withDescription("use given resource definition file").create(FILE);
		
		// -name mandatory arg
		Option name = OptionBuilder.withArgName(NAME).hasArg().withDescription("use given resource name").create(NAME);

		ArgumentsParser parser = new ArgumentsParser(GetList.class.getName(), properties, name);
		
		Properties props = parser.parseArg(getArguments());

		// gets resource properties URL
		String resource_file = props.getProperty(FILE);
		// gets resource name
		String resource_name = props.getProperty(NAME);

		// gets File of properties file
		File propsFile = null;
		if (resource_file != null){
			propsFile = new File(resource_file);
			if (!propsFile.isAbsolute()){
				propsFile = new File(JemConsole.CURRENT_DIRECTORY, resource_file);
			}
		}
		
		Object object = HttpManager.getInstance().execute(COMMON_RESOURCES_QUERY_STRING, getParameters(), (resource_name != null) ? resource_name : "*");
		if (object instanceof Resources){
			Resources resources = (Resources)object;
			// serialize the resource in XML format
			StringBuffer buffer = new StringBuffer();
			for (Resource res : resources.getResources()){
				buffer.append(res.toString()).append('\n');
			}
			LogAppl.getInstance().emit(NodeMessage.JEMC064I, buffer.toString());
			if (propsFile != null){
				XStream xs = XmlUtil.getXStream(); 
				xs.toXML(resources, new FileOutputStream(propsFile));
			}
		} else {
			throw new Exception("Source file doesn't contain any resource or resources object but '"+object.getClass().getName()+"'");
		}
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getCommandKey()
	 */
	@Override
	public String getKey() {
		return KEY;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getDescription()
	 */
	@Override
	public String getDescription() {
		return DESCRIPTION;
	}
	
}