/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console.commands.resources;

import java.io.File;
import java.io.FileOutputStream;
import java.io.StringWriter;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.console.HttpManager;
import org.pepstock.jem.console.JemConsole;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.resources.XmlUtil;

import com.thoughtworks.xstream.XStream;

/**
 * Gets a resource into JEM.<br>
 * Is a command (to execute by command line) where 1 argument is mandatory and 1 is optional: <br>
 * <code>-name [name]</code> indicates the name of resource.<br>
 * <code>-file [url]</code> indicates the properties file where store resource definition<br>
 * <br>
 * <b>GetResource -name ... [-file ...]</b><br>
 * <br>
 * Is possible to have help from command line by <code>-help</code> argument.<br>
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class Get extends ResourceCommand{

	private static final String ACTION = org.pepstock.jem.console.commands.resources.Get.class.getName();

	private static final String KEY = "getResource";
	
	private static final String DESCRIPTION = "Returns the resource defintion by resource name";

	/**
	 * Key for the file of resource
	 */
	private static String FILE = "file";

	/**
	 * Key for the name of resource 
	 */
	private static String NAME = "name";


	/**
	 * Empty constructor
	 */
	public Get(){
		super(ACTION);
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#execute(java.lang.String)
	 */
	@SuppressWarnings("static-access")
	@Override
	public void execute(String userid, String groupid) throws Exception {
		// -file mandatory arg
		Option properties = OptionBuilder.withArgName(FILE).hasArg().withDescription("use given resource definition file").create(FILE);

		// -name mandatory arg
		Option name = OptionBuilder.withArgName(NAME).hasArg().withDescription("use given resource name").create(NAME);
		name.setRequired(true);
		ArgumentsParser parser = new ArgumentsParser(this.getClass().getName(), name, properties);
		Properties props = parser.parseArg(getArguments());

		// gets resource properties URL
		String resource_file = props.getProperty(FILE);
		// gets resource name
		String resource_name = props.getProperty(NAME);

		// gets File of properties file
		File propsFile = null;
		if (resource_file != null){
			propsFile = new File(resource_file);
			if (!propsFile.isAbsolute()){
				propsFile = new File(JemConsole.CURRENT_DIRECTORY, resource_file);
			}
		}

		Object object = HttpManager.getInstance().execute(COMMON_RESOURCES_QUERY_STRING, getParameters(), resource_name);
		if (object !=  null){
			if (object instanceof Resource){
				// puts the resource in the map
				Resource resource = (Resource)object;
				XStream xs = XmlUtil.getXStream(); 
				StringWriter writer = new StringWriter();
				xs.toXML(resource, writer);
				LogAppl.getInstance().emit(NodeMessage.JEMC063I, resource.toString(), writer.getBuffer().toString());
				if (propsFile != null){
					xs.toXML(resource, new FileOutputStream(propsFile));
				}
				return;
			}
		}
		LogAppl.getInstance().emit(NodeMessage.JEMC062W, resource_name);
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getCommandKey()
	 */
	@Override
	public String getKey() {
		return KEY;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getDescription()
	 */
	@Override
	public String getDescription() {
		return DESCRIPTION;
	}
}