/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console.commands;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.Job;
import org.pepstock.jem.PreJob;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.commands.util.Factory;
import org.pepstock.jem.console.Command;
import org.pepstock.jem.console.HttpManager;
import org.pepstock.jem.console.JemConsole;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;

/**
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class Submit extends Command {

	private static final String KEY = "submit";
	
	private static final String DESCRIPTION = "Submits a job described by job control language file";

	/**
	 * Query string to submit a job
	 */
	private static final String SUBMIT_QUERY_STRING = "/servlet/submit";
	
	/**
	 * Key for the url for jcl
	 */
	private static String JCL = "jcl";
	
	/**
	 * Key for the type of jcl
	 */
	private static String TYPE = "type";

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#execute(java.lang.String)
	 */
	@Override
	public void execute(String userid, String groupid) throws Exception {
		// parses args
		// -jcl mandatory arg
		@SuppressWarnings("static-access")
		Option jcl = OptionBuilder.withArgName(JCL).hasArg().withDescription("use given jcl url/file").create(JCL);
		jcl.setRequired(true);
		// -type mandatory arg
		@SuppressWarnings("static-access")
		Option type = OptionBuilder.withArgName(TYPE).hasArg().withDescription("use given jcl type").create(TYPE);
		type.setRequired(true);
		
		ArgumentsParser parser = new ArgumentsParser(this.getClass().getName(), jcl, type);
		
		Properties props = parser.parseArg(getArguments());

		// gets resource properties URL
		String jcl_url = props.getProperty(JCL);
		// gets resource name
		String jcl_type = props.getProperty(TYPE);

		submitJob(jcl_url, jcl_type, userid, groupid);
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getKey()
	 */
	@Override
	public String getKey() {
		return KEY;
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.console.Command#getDescription()
	 */
	@Override
	public String getDescription() {
		return DESCRIPTION;
	}

	
	/**
	 * Submits the job, connecting to Hazelcast, reading JCL content and creating new Pre job object, 
	 * adds itself as listener to a hazelcast topic 
	 * 
	 * @throws Exception if errors occur
	 */
	private void submitJob(String jclUrl, String jclType, String userid, String groupid) throws Exception{
		
		// gets URL of JCL content, reads and loads it into Prejob object, setting the JCL type
		URL url = null; 
		try {
			url = new URL(jclUrl);	
		} catch (MalformedURLException ex){
			// if it's not a URL, try as a file
			File jcl = new File(jclUrl);
			if (!jcl.isAbsolute()){
				jcl = new File(JemConsole.CURRENT_DIRECTORY, jclUrl);
			}
			url = jcl.toURI().toURL();
		}
		
		PreJob preJob = Factory.createPreJob(url);
		preJob.setJclType(jclType);
		
		// creates a job asking to Hazelcast for a new long value
		// Pads the value with "0"
		Job job = new Job();
		// sets user using System.properties
		//job.setUser(System.getProperty("user.name"));
		job.setUser(userid);
		job.setOrgUnit(groupid);
		// loads all line arguments (the -D properties).
		// could be useful to factories, listeners and during job execution to job itself
		job.setInputArguments(ManagementFactory.getRuntimeMXBean().getInputArguments());
		
		// loads prejob with job
		preJob.setJob(job);

		//submit the prejob by http
		
		Object object = HttpManager.getInstance().execute(SUBMIT_QUERY_STRING, null, preJob);

		if (object != null){
			if (object instanceof String ){
				String jobId = (String) object;
				job.setId(jobId);
				LogAppl.getInstance().emit(NodeMessage.JEMC020I, job.toString());
			}
		}
	}

}