/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console;

import java.io.File;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.commands.UserIDCommand;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.console.commands.ChangeDirectory;
import org.pepstock.jem.console.commands.ListDirectory;
import org.pepstock.jem.console.commands.Submit;
import org.pepstock.jem.console.commands.ViewFile;
import org.pepstock.jem.console.commands.resources.Get;
import org.pepstock.jem.console.commands.resources.GetList;
import org.pepstock.jem.console.commands.resources.Remove;
import org.pepstock.jem.console.commands.resources.Set;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.util.CmdConsole;

/**
 * This is a console to operate via command line to JEM.<br>
 * Uses the HTTP protocol, so the JEM web app must work otherwise an error occurs.<br>
 * Needs arguments to work:<br>
 * <br>
 * <code>-url [http address]</code> indicates the address of JEM web application<br>
 * <code>-user [userid]</code>, optionally indicates the user id to use to log in JEM. It missing, uses the user of machine<br>
 * <br>
 * <br>
 * <b>JemConsole -host ... [-user ...]</b><br>
 * <br>
 * The command asks the password of the user, as following:<br>
 * <br>
 * [user]'s password:<br>
 * <br>
 * Password is the same used by web application, using Apache Shiro authentication realm.<br>
 * <br>
 * The prompt is always <code>JEM [jem-group-name]--&gt;</code>.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class JemConsole extends UserIDCommand{
	
	/**
	 * Key for the host of JEM web app 
	 */
	private static String HOST = "host";

	/**
	 * Key for the user id passed by arg 
	 */
	private static String USER = "user";

	private static final String HEADER_COMMAND = "Command";
	private static final String HEADER_DESCRIPTION = "Description";
	
	private static StringBuffer HELP_STRING = new StringBuffer();
	
	private Command[] commands = new Command[]{new GetList(), new Get(), new Set(), new Remove(), new Submit(), new ChangeDirectory(), new ListDirectory(),
			new ViewFile()};
	

	/**
	 * Console instance, used outside of this class for writer 
	 */
	public static final CmdConsole CONSOLE = new CmdConsole(); 
	
	/**
	 * The current directory
	 */
	public static File CURRENT_DIRECTORY;
	
	/**
	 * Constructs and creates the help format, with all commands
	 */
	public JemConsole() {
		super();
		loadtHelpFormat();
		File file = new File(".");
		CURRENT_DIRECTORY = new File(FilenameUtils.normalize(file.getAbsolutePath()));
	}
	

	/**
	 * Returns the command by key word
	 * 
	 * @param key keyword of command
	 * @return command command instance
	 */
	public Command getCommand(String key){
		for (int i=0; i<commands.length; i++){
			if (commands[i].getKey().equalsIgnoreCase(key))
				return commands[i];
		}
		return null;
	}
	
	
	/**
	 * Creates the help format to display every time a command is unknown
	 */
	private void loadtHelpFormat(){
		// calculates the max len of key word
		int maxLen = 0;
		for (int i=0; i<commands.length; i++){
			maxLen = Math.max(maxLen, commands[i].getKey().length());
		}
		// calculates number od tabs (1 tab = 8 chars)
		int tabCounts = ((int)Math.floor(maxLen / (double)8)) + 1;
	
		// builds Header
		int underlineLen = HEADER_COMMAND.length() + HEADER_DESCRIPTION.length();
		HELP_STRING.append(HEADER_COMMAND);
		int hdrTabCounts = ((int)Math.floor(HEADER_COMMAND.length() / (double)8));
		for (int k=hdrTabCounts; k<tabCounts; k++){
			HELP_STRING.append('\t');
			underlineLen += 8;
		}
		HELP_STRING.append(HEADER_DESCRIPTION);
		
		// builds underline with =
		HELP_STRING.append('\n').append(StringUtils.repeat("=", underlineLen));
		
		// adds all keys and descriptions for all commands
		for (int i=0; i<commands.length; i++){
			HELP_STRING.append('\n').append(commands[i].getKey());
			int myTabCounts = ((int)Math.floor(commands[i].getKey().length() / (double)8));
			for (int k=myTabCounts; k<tabCounts; k++){
				HELP_STRING.append('\t');
			}
			HELP_STRING.append(commands[i].getDescription());
		}
	}

	/**
	 * Main method! Parses the arguments and creates the console, everything by HTTP
	 * 
	 * @param args command-line arguments
	 */
	@SuppressWarnings("static-access")
	public static void main(String[] args){
		LogAppl.getInstance(CONSOLE.getWriter());
		JemConsole jemconsole= new JemConsole();
		// sets return code
		int rc = 0;
		try {
			//-host arguments
			Option host = OptionBuilder.withArgName(HOST).hasArg().withDescription("use given host and port to JEM http server").create(HOST);
			
			Option user = OptionBuilder.withArgName(USER).hasArg().withDescription("use given userid to log in JEM").create(USER);
			//host.setRequired(true);
			ArgumentsParser parser = new ArgumentsParser(JemConsole.class.getName(), host, user);
			Properties properties = parser.parseArg(args);
			String http_host = null;
			// checks if the URL is argument or not
			if (properties.containsKey(HOST)){
				http_host = properties.getProperty(HOST);	
			} else {
				boolean waitForHost = true;
				while(waitForHost){
					String[] value = CONSOLE.readCommand("JEM HTTP host: ");
					if (value != null){
						if (value.length > 0){
							http_host = value[0];
							waitForHost = false; 
						}
					}
				}
			}
			
			HttpManager.createInstance(http_host);

			// checks if user is passed by arg
			if (properties.containsKey(USER)){
				jemconsole.setUserID(properties.getProperty(USER));	
			}
			
			String password = CONSOLE.readPassword(jemconsole.getUserID());
			HttpManager.getInstance().login(jemconsole.getUserID(), password);
			
			String prompt = "JEM ["+HttpManager.getInstance().getGroupName()+"]--> ";
			CONSOLE.print("Current directory: "+ FilenameUtils.normalize(JemConsole.CURRENT_DIRECTORY.getAbsolutePath()));
			
			boolean exit = false;
			while(!exit){
				try{
					String[] arguments = CONSOLE.readCommand(prompt);

					if (arguments != null){
						if (arguments[0].equalsIgnoreCase("exit")){
							exit = true;
						} else if (arguments[0].equalsIgnoreCase("help")){
							CONSOLE.print(HELP_STRING.toString());
						} else {
							Command command = jemconsole.getCommand(arguments[0]);
							if (command != null){
								command.setArguments(arguments);
								command.execute(jemconsole.getUserID(), jemconsole.getGroupID());
							} else {
								LogAppl.getInstance().emit(NodeMessage.JEMC050I, "Command '"+arguments[0]+"' not found");
								CONSOLE.print(HELP_STRING.toString());
							}
						}
					}
					CONSOLE.getWriter().flush();
				} catch (Exception e){
					e.printStackTrace(CONSOLE.getWriter());
					CONSOLE.getWriter().flush();
				}
			}
		} catch (Exception e) {
			CONSOLE.print(e.getMessage());
			rc = 1;
		} 
		try {
			HttpManager.getInstance().logout();
		} catch (Exception e) {
		}
		System.exit(rc);		
	}
}