/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.console;

import java.io.IOException;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Properties;

import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;

import com.thoughtworks.xstream.XStream;

/**
 * Utility class to call JEM by HTTP to extract group name and members list and to submit a job by http.
 * @author Andrea "Stock" Stocchero
 *
 */
public class HttpManager {
	
	/**
	 * Key used to store user id
	 */
	public static final String USER_PROPERTY_KEY = "jem.command.user";
	
	/**
	 * Key used to store user password
	 */
	public static final String PASSWORD_PROPERTY_KEY = "jem.command.password";
	/**
	 * Query string to get cluster group name of Hazelcast
	 */
	private static final String NAME_QUERY_STRING = "/servlet/getClusterGroupName";
	
	/**
	 * Query string to log in
	 */
	private static final String LOGIN_QUERY_STRING = "/servlet/login";

	/**
	 * Query string to log out
	 */
	private static final String LOGOUT_QUERY_STRING = "/servlet/logout";

	
	private static HttpManager MANAGER = null;
	
	private URI uri = null;
	
	private HttpClient client = null;

	/**
	 * Private constructor of singleton. Creates a http client (in SSL configuration if URL is HTTPS) which will be mantained 
	 * for all commands till logout 
	 * 
	 * @param url JEM web app url
	 * 
	 * @throws URISyntaxException if URL is not correct
	 * @throws KeyStoreException if keystore has errors
	 * @throws NoSuchAlgorithmException if keystore has errors 
	 * @throws UnrecoverableKeyException if keystore has errors
	 * @throws KeyManagementException if keystore has errors
	 */
	private HttpManager(String url) throws URISyntaxException, KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
		// creates a HTTP client
        client = new DefaultHttpClient();
        // creates a URI builder
        URIBuilder builder = new URIBuilder(url);
        // creates URI checking if is syntaxly correct 
        uri = builder.build();
        // configures SSL transport if necessary
        configureHttpClient();
	}

	/**
	 * Public singleton method to create the instance.
	 * 
	 * @param url JEM web app url
	 * @return http manager instance
	 * 
	 * @throws URISyntaxException if URL is not correct
	 * @throws KeyStoreException if keystore has errors
	 * @throws NoSuchAlgorithmException if keystore has errors 
	 * @throws UnrecoverableKeyException if keystore has errors
	 * @throws KeyManagementException if keystore has errors
	 */
	public static HttpManager createInstance(String url) throws URISyntaxException, KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException{
		if (MANAGER == null){
			MANAGER = new HttpManager(url);
		}
		return getInstance();
	}
	
	/**
	 * Public singleton method to return the instance.
	 * @return http manager instance
	 */
	public static HttpManager getInstance(){
		return MANAGER;
	}
	
	
	/**
	 * Calls a http node of JEM to get group name of Hazelcast cluster.
	 * 
	 * @param url http URL to call
	 * @return group name of Hazelcast cluster
	 * @throws Exception if errors occur
	 */
	public String getGroupName() throws Exception {
		// creates URI builder
		URIBuilder builder = new URIBuilder(uri);
		// adds path for group name request
		builder.setPath(HttpManager.NAME_QUERY_STRING);
		// prepares GET request and basic response handler
		HttpGet httpget = new HttpGet(builder.build());
		
		ResponseHandler<String> responseHandler = new BasicResponseHandler();
		// executes and no parsing
		// result must be only a string
		String responseBody = client.execute(httpget, responseHandler);
		return responseBody.trim();
	}

	/**
	 * Execute a generic path via HTTP/HTTPS, passing parameters and object to serialize in the 
	 * HTTP content
	 * 
	 * @param path http path to call (usually a servlet)
	 * @param parameters parameters to add to URL
	 * @param parm object to serialize in XML inside teh HTTP content
	 * @return deserialize object from response (to be cast)
	 * @throws Exception if errors occur
	 */
	public Object execute(String path, Properties parameters, Object parm) throws Exception {
		// creates XMl engine
		XStream streamer = new XStream();

		// creates URI builder
		URIBuilder builder = new URIBuilder(uri);
		// sets path to call
		builder.setPath(path);
		// if there is at least a parameter, adds all parameters to URI
		if (parameters != null){
			if (!parameters.isEmpty()){
				for (Object key : parameters.keySet()){
					String value = parameters.getProperty(key.toString());
					builder.setParameter(key.toString(), value);	
				}
			}
		}
		// prepares POST request and basic response handler
		HttpPost httppost = new HttpPost(builder.build());
		// if there is a object to serialize, adds to content in XML format
		if (parm != null){
			StringEntity entity = new StringEntity(streamer.toXML(parm), ContentType.create("text/xml", "UTF-8"));
			httppost.setEntity(entity);
		}
		ResponseHandler<String> responseHandler = new BasicResponseHandler();

		// executes and XML parsing
		// result is a generic object to be cast
		String responseBody = client.execute(httppost, responseHandler);
		String result = responseBody.trim();
		try {
			return streamer.fromXML(result);	
		} catch (Exception ex){
			return result;
		}
	}

	/**
	 * Performs the login using user and password
	 * 
	 * @param user user to authenticate
	 * @param password password to authenticate
	 * @param url http URL to call
	 * @param httpclient hhtp client already created
	 * @throws ClientProtocolException if any errors occurs on calling the servlet
	 * @throws IOException if I/O error occurs
	 * @throws URISyntaxException if uri syntax is wrong
	 */
	public void login(String user, String password) throws ClientProtocolException, IOException, URISyntaxException{
        // account info in a properties
        Properties properties = new Properties();
        properties.setProperty(USER_PROPERTY_KEY, user);
        properties.setProperty(PASSWORD_PROPERTY_KEY, password);
        StringWriter writer = new StringWriter();
        properties.store(writer, "Account info");
        // login 
        
        // concats URL with query string
		URIBuilder builder = new URIBuilder(uri);
		builder.setPath(HttpManager.LOGIN_QUERY_STRING);

        StringEntity entity = new StringEntity(writer.toString(), ContentType.create("text/plain", "UTF-8"));        

        // prepares POST request and basic response handler
        HttpPost httppost = new HttpPost(builder.build());
        httppost.setEntity(entity);
        ResponseHandler<String> responseHandler = new BasicResponseHandler();

        // executes and no parsing
        // result must be only a string
        client.execute(httppost, responseHandler);
	}

	/**
	 * Performs logout from JEM
	 * 
	 * @throws ClientProtocolException if any errors occurs on calling the servlet
	 * @throws IOException if I/O error occurs
	 * @throws URISyntaxException if uri syntax is wrong 
	 */
	public void logout() throws ClientProtocolException, IOException, URISyntaxException{
        // concats URL with query string
		URIBuilder builder = new URIBuilder(uri);
		builder.setPath(HttpManager.LOGOUT_QUERY_STRING);

        // prepares POST request and basic response handler
        HttpPost httppost = new HttpPost(builder.build());
        ResponseHandler<String> responseHandler = new BasicResponseHandler();
        // executes and no parsing
        client.execute(httppost, responseHandler);
        // with logout, closes the HTTP client
        client.getConnectionManager().shutdown();
	}

	/**
	 * Configures HTTP client in SSL mode if the scheme is HTTPS, otherwise do nothing
	 * 
	 * @throws KeyStoreException if keystore has errors
	 * @throws NoSuchAlgorithmException if keystore has errors 
	 * @throws UnrecoverableKeyException if keystore has errors
	 * @throws KeyManagementException if keystore has errors
	 */
	private void configureHttpClient() throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException{
		// checks if is HTTPS
		if (uri.getScheme().equalsIgnoreCase("https")){
			// creates SSL socket factory
			SSLSocketFactory sf = buildSSLSocketFactory();
			// creates new Scheme and registers it to be used
			Scheme https = new Scheme(uri.getScheme(), uri.getPort(), sf);
			SchemeRegistry sr = client.getConnectionManager().getSchemeRegistry();
			sr.register(https);
		}
	}
 
	/**
	 * Creates a SSL socket factory
	 * 
	 * @return socket factory for SSL mode
	 * @throws KeyStoreException if keystore has errors
	 * @throws NoSuchAlgorithmException if keystore has errors 
	 * @throws UnrecoverableKeyException if keystore has errors
	 * @throws KeyManagementException if keystore has errors
	 */
	private SSLSocketFactory buildSSLSocketFactory() throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
		TrustStrategy ts = new TrustStrategy() {
			@Override
			public boolean isTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
				// always true then TRUSTED
				return true;
			}
		};
		// build socket factory with hostname verification turned off
		return  new SSLSocketFactory(ts, SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
	}
	
}