/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.commands.util;

import java.io.IOException;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Properties;

import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;
import org.pepstock.jem.PreJob;
import org.pepstock.jem.node.resources.ResourcesUtil;

import com.thoughtworks.xstream.XStream;

/**
 * Utility class to call JEM by HTTP to extract group name and members list and to submit a job by http.
 * @author Andrea "Stock" Stocchero
 *
 */
public class HttpUtil {
	
	/**
	 * Key used to store user id
	 */
	public static final String USER_PROPERTY_KEY = "jem.command.user";
	
	/**
	 * Key used to store user password
	 */
	public static final String PASSWORD_PROPERTY_KEY = "jem.command.password";
	/**
	 * Query string to get cluster group name of Hazelcast
	 */
	private static final String NAME_QUERY_STRING = "/servlet/getClusterGroupName";
	
	/**
	 * Query string to get all member of group name of Hazelcast
	 */
	private static final String MEMBERS_QUERY_STRING = "/servlet/getClusterMembers";

	/**
	 * Query string to log in
	 */
	private static final String LOGIN_QUERY_STRING = "/servlet/login";

	/**
	 * Query string to log out
	 */
	private static final String LOGOUT_QUERY_STRING = "/servlet/logout";

	/**
	 * Query string to submit a job
	 */
	private static final String SUBMIT_QUERY_STRING = "/servlet/submit";
	
	/**
	 * Query string to get a job by id
	 */
	private static final String JOBID_QUERY_STRING = "/servlet/getJobById";

	/**
	 * Query string to get a ended job by id
	 */
	private static final String ENDE_JOBID_QUERY_STRING = "/servlet/getEndedJobById";

	/**
	 * Query string to common resource management
	 */
	private static final String COMMON_RESOURCES_QUERY_STRING = "/servlet/resources";

	/**
	 * Calls a http node of JEM to get all members of group, necessary to client to connect to JEM.
	 * 
	 * @param url http URL to call
	 * @return Arrays with all members of Hazelcast cluster
	 * @throws Exception if errors occur
	 */
	public static String[] getMembers(String url) throws Exception {
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the result
        String nodes[] = null;
        try {
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.MEMBERS_QUERY_STRING;
        	// prepares GET request and basic response handler
            HttpGet httpget = new HttpGet(completeUrl);
            ResponseHandler<String> responseHandler = new BasicResponseHandler();

            // executes and parse the results
            // result must be [ipaddress:port],[ipaddress:port],[ipaddress:port],....[ipaddress:port]
            String responseBody = httpclient.execute(httpget, responseHandler);
            nodes = responseBody.trim().split(",");

		} finally {
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return nodes;
	}
	
	/**
	 * Calls a http node of JEM to get group anme of Hazelcast cluster, necessary to client to connect to JEM.
	 * 
	 * @param url http URL to call
	 * @return group name of Hazelcast cluster
	 * @throws Exception if errors occur
	 */
	public static String getGroupName(String url) throws Exception {
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the result
        String groupName = null;
        try {
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.NAME_QUERY_STRING;
        	// prepares GET request and basic response handler
            HttpGet httpget = new HttpGet(completeUrl);
            ResponseHandler<String> responseHandler = new BasicResponseHandler();
            
            // executes and no parsing
            // result must be only a string
            String responseBody = httpclient.execute(httpget, responseHandler);
            groupName = responseBody.trim();

		} finally {
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return groupName;
	}

	/**
	 * Calls a http node of JEM to submit a job.
	 * 
	 * @param user user to authenticate
	 * @param password password to authenticate
	 * @param url http URL to call
	 * @param prejob job instance to submit
	 * @return job id
	 * @throws Exception if errors occur
	 */
	public static String submit(String user, String password, String url, PreJob prejob) throws Exception {
		String jobId = null;
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the entity to send via HTTP
        XStream streamer = new XStream();
        String content = streamer.toXML(prejob);
        
        boolean loggedIn = false;
        try {
        	login(user, password, url, httpclient);
        	loggedIn = true;
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.SUBMIT_QUERY_STRING;
        	StringEntity entity = new StringEntity(content, ContentType.create("text/xml", "UTF-8"));        

        	// prepares POST request and basic response handler
        	HttpPost httppost = new HttpPost(completeUrl);
        	httppost.setEntity(entity);
            ResponseHandler<String> responseHandler = new BasicResponseHandler();
            
            // executes and no parsing
            // result must be only a string
            String responseBody = httpclient.execute(httppost, responseHandler);
            jobId = responseBody.trim();
		} finally {
			if (loggedIn){
				try {
					logout(url, httpclient);
				} catch (Exception e){
					//TODO message
				}
			}
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return jobId;
	}

	/**
	 * 
	 * @param user
	 * @param password
	 * @param url
	 * @param jobId
	 * @param queueName
	 * @return Object
	 * @throws Exception
	 */
	public static Object getJobByID(String user, String password, String url, String jobId, String queueName) throws Exception {
		Object returnedObject = null;
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the entity to send via HTTP
        XStream streamer = new XStream();
        
        boolean loggedIn = false;
        try {
        	login(user, password, url, httpclient);
        	loggedIn = true;
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.JOBID_QUERY_STRING+"?jobId="+jobId+"&queueName="+queueName;
        	// prepares POST request and basic response handler
            HttpGet httpget = new HttpGet(completeUrl);
            ResponseHandler<String> responseHandler = new BasicResponseHandler();
            
            // executes and no parsing
            // result must be only a string
            String responseBody = httpclient.execute(httpget, responseHandler);
            returnedObject = streamer.fromXML(responseBody.trim());
		} finally {
			if (loggedIn){
				try {
					logout(url, httpclient);
				} catch (Exception e){
					//TODO message
				}
			}
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return returnedObject;
	}

	/**
	 * 
	 * @param user
	 * @param password
	 * @param url
	 * @param jobId
	 * @param queueName
	 * @return Object
	 * @throws Exception
	 */
	public static Object getEndedJobByID(String user, String password, String url, String jobId) throws Exception {
		Object returnedObject = null;
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the entity to send via HTTP
        XStream streamer = new XStream();
        
        boolean loggedIn = false;
        try {
        	login(user, password, url, httpclient);
        	loggedIn = true;
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.ENDE_JOBID_QUERY_STRING+"?jobId="+jobId;
        	// prepares POST request and basic response handler
            HttpGet httpget = new HttpGet(completeUrl);
            ResponseHandler<String> responseHandler = new BasicResponseHandler();
            
            // executes and no parsing
            // result must be only a string
            String responseBody = httpclient.execute(httpget, responseHandler);
            returnedObject = streamer.fromXML(responseBody.trim());
		} finally {
			if (loggedIn){
				try {
					logout(url, httpclient);
				} catch (Exception e){
					e.printStackTrace();
				}
			}
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return returnedObject;
	}

	
	/**
	 * Calls a HTTP node to manage the common resources
	 * 
	 * @param user user to authenticate
	 * @param password password to authenticate
	 * @param url http URL to call
	 * @param action the class name of the command which is calling this method.
	 * @param parm object to pass to servlet in the body of http request
	 * @return a object which can be of different types, depending on caller of this method
	 * @throws Exception if any exception occurs
	 */
	public static Object resource(String user, String password, String url, @SuppressWarnings("rawtypes") Class action, Object parm) throws Exception {
		Object returnedObject = null;
		// creates a HTTP client
        HttpClient httpclient = new DefaultHttpClient();
        configureHttpClient(httpclient, url);
        // prepares the entity to send via HTTP
        XStream streamer = new XStream();
        String content = null;
        if (parm != null)
        	content = streamer.toXML(parm);
        
        boolean loggedIn = false;
        try {
        	login(user, password, url, httpclient);
        	loggedIn = true;
        	// concats URL with query string
        	String completeUrl = url+HttpUtil.COMMON_RESOURCES_QUERY_STRING;

        	URIBuilder builder = new URIBuilder(completeUrl);
        	builder.setParameter(ResourcesUtil.ACTION_PARM, action.getName());
        	URI uri = builder.build();
        	// prepares POST request and basic response handler
        	HttpPost httppost = new HttpPost(uri);
        	// if there's a content, sets it (XML format useing xStream)
        	if (content != null){
        		StringEntity entity = new StringEntity(content, ContentType.create("text/xml", "UTF-8"));
        		httppost.setEntity(entity);
        	}
            ResponseHandler<String> responseHandler = new BasicResponseHandler();
            
            // executes 
            String responseBody = httpclient.execute(httppost, responseHandler);
            // parses from XML format, by XStream
            returnedObject = streamer.fromXML(responseBody.trim());
		} finally {
			// if logged, try to logout
			if (loggedIn){
				try {
					logout(url, httpclient);
				} catch (Exception e){
				}
			}
			// close http client
            httpclient.getConnectionManager().shutdown();
        }
		return returnedObject;
	}

	/**
	 * Performs the login using user and password
	 * 
	 * @param user user to authenticate
	 * @param password password to authenticate
	 * @param url http URL to call
	 * @param httpclient hhtp client already created
	 * @throws ClientProtocolException if any errors occurs on calling the servlet
	 * @throws IOException if I/O error occurs
	 */
	private static void login(String user, String password, String url, HttpClient httpclient) throws ClientProtocolException, IOException{
        // account info in a properties
        Properties properties = new Properties();
        properties.setProperty(USER_PROPERTY_KEY, user);
        properties.setProperty(PASSWORD_PROPERTY_KEY, password);
        StringWriter writer = new StringWriter();
        properties.store(writer, "Account info");
        // login 
        
        // concats URL with query string
        String completeUrl = url+HttpUtil.LOGIN_QUERY_STRING;
        StringEntity entity = new StringEntity(writer.toString(), ContentType.create("text/plain", "UTF-8"));
        
        // prepares POST request and basic response handler
        HttpPost httppost = new HttpPost(completeUrl);
        httppost.setEntity(entity);
        ResponseHandler<String> responseHandler = new BasicResponseHandler();

        // executes and no parsing
        // result must be only a string
		httpclient.execute(httppost, responseHandler);
	}

	/**
	 * Logs out from HTTP server.
	 * 
	 * @param url http URL to call
	 * @param httpclient http client already created
	 * @throws ClientProtocolException if any errors occurs on calling the servlet
	 * @throws IOException if I/O error occurs	 */
	private static void logout(String url, HttpClient httpclient) throws ClientProtocolException, IOException{
        String completeUrl = url+HttpUtil.LOGOUT_QUERY_STRING;
        // prepares POST request and basic response handler
        HttpPost httppost = new HttpPost(completeUrl);
        ResponseHandler<String> responseHandler = new BasicResponseHandler();
        // executes and no parsing
        httpclient.execute(httppost, responseHandler);
	}
	
	/**
	 * Configures SSL HTTP client, if necessary
	 * 
	 * @param client http client already created
	 * @param uri http URI to call
	 * @throws URISyntaxException 
	 * @throws KeyManagementException
	 * @throws UnrecoverableKeyException
	 * @throws NoSuchAlgorithmException
	 * @throws KeyStoreException
	 */
	private static void configureHttpClient(HttpClient client, String uri) throws URISyntaxException, KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException{
		URI uriObject = URI.create(uri);
		HttpUtil.configureHttpClient(client, uriObject);
	}
	
	/**
	 * Configures SSL HTTP client, if necessary
	 * 
	 * @param client http client already created
	 * @param uri http URI to call
	 * @throws KeyManagementException
	 * @throws UnrecoverableKeyException
	 * @throws NoSuchAlgorithmException
	 * @throws KeyStoreException
	 */
	private static void configureHttpClient(HttpClient client, URI uri) throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException{
		if (uri.getScheme() != null){
			// sets SSL ONLY if teh scheme is HTTP
			if (uri.getScheme().equalsIgnoreCase("https")){
				SSLSocketFactory sf = buildSSLSocketFactory();
				int port=uri.getPort();
				if(port==-1){
					port=443;
				}
				Scheme https = new Scheme(uri.getScheme(), port, sf);
				SchemeRegistry sr = client.getConnectionManager().getSchemeRegistry();
				sr.register(https);
			}
		}
	}
 
	private static SSLSocketFactory buildSSLSocketFactory() throws KeyManagementException, UnrecoverableKeyException, NoSuchAlgorithmException, KeyStoreException {
		TrustStrategy ts = new TrustStrategy() {
			@Override
			public boolean isTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
				// always true to avoif certicateunknow excpetion
				return true;
			}
		};
		/* build socket factory with hostname verification turned off. */
		return  new SSLSocketFactory(ts, SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
	}
	
}