/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.pepstock.jem.commands;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.Job;
import org.pepstock.jem.OutputFileContent;
import org.pepstock.jem.PreJob;
import org.pepstock.jem.Result;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.commands.util.Factory;
import org.pepstock.jem.commands.util.HazelcastUtil;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.executors.jobs.GetMessagesLog;

import com.hazelcast.core.Cluster;
import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.IQueue;
import com.hazelcast.core.ITopic;
import com.hazelcast.core.IdGenerator;
import com.hazelcast.core.Member;
import com.hazelcast.core.Message;
import com.hazelcast.core.MessageListener;

/**
 * Submits JCL into JEm.<br>
 * Is a command (to execute by command line) where 4 arguments are mandatory and
 * 3 are optional: <br>
 * 
 * *
 * <ul>
 * <li><code>-jcl [url]</code> mandatory, indicates the JCL which describes the
 * job. Must be a valid URL where to read JCL content</li>
 * <li><code>-type [type]</code> mandatory indicates the type of JCL. Must be a
 * valid value, defined from JEM factories implemented.</li>
 * <li><code>-host [http address]</code> mandatory indicates the address of a
 * "web node" to receive the list of members and cluster information.</li>
 * <li><code>-password [String password]</code> mandatory indicates hazelcast
 * password needed to connect to the environment.</li>
 * <li><code>-nowait</code> optional allows to end the process to submit without
 * waiting for end of job.</li>
 * <li><code>-printOutput</code> optional in case you choose no wait it will
 * print on standard output the job log</li>
 * <li><code>-privateKey [String path of the private key]</code> optional
 * indicates the pem private key to use in case JEM was installed with login
 * protocol enabled.</li>
 * </ul>
 * <p>
 * <b>Submit -jcl ... -type ... -host ... -password [-nowait] [-printOutput]
 * [-privateKey ...]</b>
 * <p>
 * <br>
 * The "submitter": <br>
 * <ul>
 * <li>asks to JEM cluster a unique ID, used like JOB ID, using Hazelcast
 * features to create a unique ID in the cluster</li>
 * <li>puts the JOB in PRE-INPUT queue</li>
 * <li>waits for notification of the end of job (if -nowait is not set), using
 * Hazelcast features of publish/subscribe (topic) pattern</li>
 * </ul>
 * <br>
 * Is possible to have help from command line by <code>-help</code> argument.
 * <br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class Submit extends UserIDCommand implements MessageListener<Job> {
	/**
	 * Key for the url for jcl
	 */
	public final static String JCL = "jcl";

	/**
	 * Key for the local environment to connect to
	 */
	public final static String PASSWORD = "password";

	/**
	 * Key for the type of jcl
	 */
	public final static String TYPE = "type";

	/**
	 * Key for the url of JEM web app
	 */
	public final static String HOST = "host";
	/**
	 * Key for the no wait
	 */
	public final static String NOWAIT = "nowait";

	/**
	 * Key for the printOutput
	 */
	public final static String PRINT_OUTPUT = "printOutput";

	/**
	 * Key for the private key to supply in case of socketInterceptor enable
	 */
	public final static String PRIVATE_KEY = "privateKey";
											   
	/**
	 * Password for the private key to supply in case of socketInterceptor
	 * enable
	 */
	public final static String PRIVATE_KEY_PWD = "privateKeyPwd";

	private String jclUrl = null;

	private String password = null;

	private String jclType = "ant";

	private String hostForJemWeb = null;

	private boolean noWait = false;

	private boolean printOutput = false;

	private HazelcastInstance client = null;

	private Object lock = new Object();

	private Job job = null;

	private String privateKey = null;

	private String privateKeyPassword = null;

	/**
	 * Empty constructor
	 */
	public Submit() {
	}

	/**
	 * Returns the object used to wait the end of job. When receives the
	 * notification about the end, notifies on this object.
	 * 
	 * @return object used to wait
	 */
	public Object getLock() {
		return lock;
	}

	/**
	 * Is called from Hazelcast engine when a job is put in OUTPUT queue, so is
	 * ended. Checks if same job that has sumbitted. If yes, notifies the thread
	 * to end
	 * 
	 * @param endedJob job ended
	 */
	@Override
	public void onMessage(Message<Job> endedJob) {
		// check if it was a routed job
		if (endedJob.getMessageObject().getRoutingInfo().getId() != null) {
			// if is not the job submitted, does nothing
			if (!job.getId().equals(endedJob.getMessageObject().getRoutingInfo().getId()))
				return;
			// save job ended, overriding the previous instance beacause routing
			// info has been added
			job = endedJob.getMessageObject();
			// if nowait is false remove job from ROUTED QUEUE
			if (!job.isNowait()) {
				IMap<String, Job> routedQueue = client.getMap(Queues.ROUTED_QUEUE);
				routedQueue.remove(job.getRoutingInfo().getId());
			}
		} else {
			// if is not the job submitted, does nothing
			if (!job.equals(endedJob.getMessageObject()))
				return;
			// save job ended, overriding the previous instance (no necessary
			// now
			// anymore)
			job = endedJob.getMessageObject();
		}
		// notify that job is ended
		synchronized (getLock()) {
			getLock().notify();
		}
	}

	/**
	 * Submits the job, connecting to Hazelcast, reading JCL content and
	 * creating new Pre job object, adds itself as listener to a hazelcast topic
	 * 
	 * @throws Exception if errors occur
	 */
	public void submitJob() throws Exception {
		System.setProperty("hazelcast.logging.type", "log4j");
		// creates a new Client instance of Hazelcast
		client = HazelcastUtil.getInstance(hostForJemWeb, password, privateKey, privateKeyPassword, getUserID());

		// gets URL of JCL content, reads and loads it into Prejob object,
		// setting the JCL type
		URL url = null;
		try {
			url = new URL(this.jclUrl);
		} catch (MalformedURLException ex) {
			// if it's not a URL, try as a file
			File jcl = new File(this.jclUrl);
			url = jcl.toURI().toURL();
		}

		PreJob preJob = Factory.createPreJob(url);
		preJob.setJclType(this.jclType);

		// creates a job asking to Hazelcast for a new long value
		// Pads the value with "0"
		job = new Job();
		// sets user using System.properties
		// JOB.setUser(System.getProperty("user.name"));
		job.setUser(getUserID());
		job.setOrgUnit(getGroupID());

		IdGenerator generator = client.getIdGenerator(Queues.JOB_ID_GENERATOR);
		long id = generator.newId();

		String jobId = Factory.createJobId(job, id);
		job.setId(jobId);
		// loads all line arguments (the -D properties).
		// could be useful to factories, listeners and during job execution to
		// job itself
		job.setInputArguments(ManagementFactory.getRuntimeMXBean().getInputArguments());
		// set nowait
		job.setNowait(noWait);
		// loads prejob with job
		preJob.setJob(job);

		// gets topic object and adds itself as listener
		ITopic<Job> topic = client.getTopic(Queues.ENDED_JOB_TOPIC);
		topic.addMessageListener(this);

		// puts the pre job in a queue for validating and miving to right QUEUE
		// (input if is correct, output if is wrong)
		IQueue<PreJob> jclCheckingQueue = client.getQueue(Queues.JCL_CHECKING_QUEUE);
		jclCheckingQueue.put(preJob);
	}

	/**
	 * Parses the arguments, creates the client, submits job and wait the end.
	 * 
	 * @param args the arguments for the commands
	 * 
	 * @return SubmitResult bean that contains:
	 *         <p>
	 *         the return code, 0 success 1 failure
	 *         <p>
	 *         the jobId
	 */
	@SuppressWarnings("static-access")
	public static SubmitResult executeCommand(String[] args) {
		LogAppl.getInstance();
		// sets return code
		int rc = 0;
		Submit cc = new Submit();
		try {
			// parses args
			ArgumentsParser parser = null;
			synchronized (OptionBuilderLock.getLock()) {
				// -jcl mandatory arg
				Option jcl = OptionBuilder.withArgName(JCL).hasArg().withDescription("use given jcl file").create(JCL);
				jcl.setRequired(true);
				// -password mandatory
				Option passwd = OptionBuilder.withArgName(PASSWORD).hasArg().withDescription("the hazelcast password for the environment").create(PASSWORD);
				passwd.setRequired(true);
				// -type mandatory arg
				Option type = OptionBuilder.withArgName(TYPE).hasArg().withDescription("use given jcl type").create(TYPE);
				type.setRequired(true);
				// -url mandatory arg
				Option host = OptionBuilder.withArgName(HOST).hasArg().withDescription("use given host to extract members cluster and group name").create(HOST);
				host.setRequired(true);
				// -nowait optional arg
				Option wait = OptionBuilder.withArgName(NOWAIT).withDescription("use given do not wait for end of job").create(NOWAIT);
				// -printOutput optional arg
				Option print_Output = OptionBuilder.withArgName(PRINT_OUTPUT).withDescription("use given to print std output and error of job").create(PRINT_OUTPUT);
				// -privateKey optional arg
				Option pk = OptionBuilder.withArgName(PRIVATE_KEY).hasArg().withDescription("the private key path to supply in case of socket interceptor enabled").create(PRIVATE_KEY);
				// -privateKeyPwd optional arg
				Option pkPwd = OptionBuilder.withArgName(PRIVATE_KEY_PWD).hasArg().withDescription("the private key password").create(PRIVATE_KEY_PWD);
				parser = new ArgumentsParser(Submit.class.getName(), jcl, type, host, wait, print_Output, pk, passwd, pkPwd);
			}
			Properties properties = parser.parseArg(args);
			cc.jclUrl = properties.getProperty(JCL);
			cc.hostForJemWeb = properties.getProperty(HOST);
			cc.jclType = properties.getProperty(TYPE);
			cc.privateKey = properties.getProperty(PRIVATE_KEY);
			cc.privateKeyPassword = properties.getProperty(PRIVATE_KEY_PWD);
			cc.password = properties.getProperty(PASSWORD);
			if ("".equals(properties.getProperty(NOWAIT))) {
				cc.noWait = true;
			}
			if ("".equals(properties.getProperty(PRINT_OUTPUT))) {
				cc.printOutput = true;
			}

			if (cc.noWait && cc.printOutput) {
				LogAppl.getInstance().emit(NodeMessage.JEMC198W, NOWAIT, PRINT_OUTPUT);
			}

			// JCL_URL = properties.get()
			// submits job
			cc.submitJob();
			// if doesn't have to wait, ends in CC 0
			if (!cc.noWait) {
				// waits for ending ot job execution
				synchronized (cc.getLock()) {
					boolean wait = true;
					while (wait) {
						try {
							// wait on object
							cc.getLock().wait();
							wait = false;
							// checks if job is not null
							if (cc.job != null) {
								// logs return code and exception if exists
								LogAppl.getInstance().emit(NodeMessage.JEMC021I, cc.job.toString(), String.valueOf(cc.job.getResult().getReturnCode()));
								if (cc.job.getResult().getReturnCode() != Result.SUCCESS) {
									LogAppl.getInstance().emit(NodeMessage.JEMC021I, cc.job.toString(), cc.job.getResult().getExceptionMessage());
								}

							}
							// sets return code to exit
							if (cc.job != null) {
								// sets return code to exit
								Result res = cc.job.getResult();
								if (res != null) {
									rc = cc.job.getResult().getReturnCode();
								} else {
									rc = 1;
								}
							} else {
								rc = 1;
							}
							// printOutput
							if (cc.job != null) {
								if (cc.printOutput) {
									printOutput(cc);
								}
							}
						} catch (InterruptedException e1) {
							// sets return code to error
							LogAppl.getInstance().emit(NodeMessage.JEMC017E);
							rc = 1;
						}
					}
				}
			}
		} catch (Exception e) {
			// with any exception, ends in CC 1
			LogAppl.getInstance().emit(NodeMessage.JEMC033E, e);
			// sets return code to error
			rc = 1;
		} finally {
			// shutdown hazelcast client
			if (cc.client != null) {
				cc.client.getLifecycleService().shutdown();
			}
		}
		SubmitResult sr = new SubmitResult();
		if (cc.job != null) {
			sr.setJobId(cc.job.getId());
		}
		sr.setRc(rc);
		return sr;
	}

	private static void printOutput(Submit cc) throws Exception {
		Cluster cluster = cc.client.getCluster();
		Set<Member> set = cluster.getMembers();
		for (Member member : set) {
			if (!member.isLiteMember()) {
				DistributedTask<OutputFileContent> task = new DistributedTask<OutputFileContent>(new GetMessagesLog(cc.job), member);
				ExecutorService executorService = cc.client.getExecutorService();
				executorService.execute(task);
				OutputFileContent content;
				try {
					content = task.get();
					System.out.println(content.getContent());
				} catch (InterruptedException e) {
					throw new Exception(e.getMessage(), e);
				} catch (ExecutionException e) {
					throw new Exception(e.getMessage(), e);
				}
				return;
			}
		}
	}

	/**
	 * Main method! Parses the arguments, creates the client, submits job and
	 * wait the end.
	 * 
	 * @param args command-line arguments
	 */
	public static void main(String[] args) {
		SubmitResult sr = executeCommand(args);
		System.exit(sr.getRc());
	}
}
