/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.commands;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.pepstock.jem.Job;
import org.pepstock.jem.PreJob;
import org.pepstock.jem.Result;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.commands.util.Factory;
import org.pepstock.jem.commands.util.HttpUtil;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.util.CmdConsole;
import org.pepstock.jem.util.TimeUtils;

/**
 * Submits JCL into JEM, by http.<br>
 * Is a command (to execute by command line) which accepts 3 arguments: <br>
 * <code>-jcl [url]</code> indicates the JCL which describes the job. Must be a
 * valid URL where to read JCL content<br>
 * <code>-type [type]</code> indicates the type of JCL. Must be a valid value,
 * defined from JEM factories implemented<br>
 * <code>-host [http address]</code> indicates the address of a "node" to submit
 * the job<br>
 * <code>-user [user]</code> indicates the userid to use connecting
 * to JEM<br>
 * <code>-password [pwd]</code> indicates optionally the password of userid to
 * use connecting to JEM<br>
 * <br>
 * <b>HttpSubmit -jcl ... -type ... -host ... -user ... -password ... -wait</b><br>
 * <br>
 * The command asks the password of the user, as following:<br>
 * <br>
 * [user]'s password:<br>
 * <br>
 * Password is the same used by web application, using Apache Shiro
 * authentication realm.<br>
 * <br>
 * Is possible to have help from command line by <code>-help</code> argument.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class HttpSubmit {
	/**
	 * Key for the url for jcl
	 */
	public final static String JCL = "jcl";

	/**
	 * Key for the type of jcl
	 */
	public final static String TYPE = "type";

	/**
	 * Key for the host of JEM web app
	 */
	public final static String HOST = "host";

	/**
	 * Key for the password of the user
	 */
	public final static String PASSWORD = "password";

	/**
	 * Key for the userid of the user
	 */
	public final static String USER = "user";

	/**
	 * Key for the no wait
	 */
	public final static String WAIT = "wait";
	
	private String jclUrl = null;

	private String jclType = "ant";

	private String hostForJemWeb = null;

	private String passwordValue = null;

	private String userValue = null;
	
	private boolean wait = false;

	/**
	 * Empty constructor
	 */
	public HttpSubmit() {
	}

	/**
	 * Submits the job, connecting to JEM by HTTP, reading JCL content and
	 * creating new Pre job object.
	 * @return return code 
	 * 
	 * @throws Exception if errors occur
	 */
	public int submitJob() throws Exception {

		int rc = 0;
		// gets URL of JCL content, reads and loads it into Prejob object,
		// setting the JCL type
		URL url = null;
		try {
			url = new URL(this.jclUrl);
		} catch (MalformedURLException ex) {
			// if it's not a URL, try as a file
			File jcl = new File(this.jclUrl);
			url = jcl.toURI().toURL();
		}

		// creates Prejob loading from URL
		PreJob preJob = Factory.createPreJob(url);
		// sets JCL type whcich wa an argument
		preJob.setJclType(this.jclType);

		// creates a empty job
		Job job = new Job();
		// gets user id
		job.setUser(userValue);
		// loads all line arguments (the -D properties).
		// could be useful to factories, listeners and during job execution to
		// job itself
		job.setInputArguments(ManagementFactory.getRuntimeMXBean().getInputArguments());

		// loads prejob with job
		preJob.setJob(job);

		// submit the prejob by http. The HTTP call returns the JOB ID
		String jobId = HttpUtil.submit(userValue, passwordValue, hostForJemWeb, preJob);
		job.setId(jobId);

		LogAppl.getInstance().emit(NodeMessage.JEMC020I, job.toString());
		
		if (this.wait){
			Job resultJob = null;
			while(resultJob == null){
				Thread.sleep(30 * TimeUtils.SECOND);
				Object obj = HttpUtil.getEndedJobByID(userValue, passwordValue, hostForJemWeb, jobId);
				if (obj instanceof Job){
					resultJob = (Job)obj;
					// logs return code and exception if exists
					LogAppl.getInstance().emit(NodeMessage.JEMC021I, resultJob.toString(), String.valueOf(resultJob.getResult().getReturnCode()));
					if (resultJob.getResult().getReturnCode() != Result.SUCCESS) {
						LogAppl.getInstance().emit(NodeMessage.JEMC021I, resultJob.toString(), resultJob.getResult().getExceptionMessage());
					}
					// sets return code to exit
					Result res = resultJob.getResult();
					if (res != null) {
						rc = resultJob.getResult().getReturnCode();
					} else {
						rc = 1;
					}
				}
			}
		}
		return rc;
	}

	/**
	 * Parses the arguments, submits job by HTTP.
	 * 
	 * @param args the arguments for the command
	 * @return the return code, 0 success 1 failure
	 */
	@SuppressWarnings("static-access")
	public static int executeCommand(String[] args) {
		LogAppl.getInstance();
		// sets return code
		int rc = 0;
		// creates the submitter
		HttpSubmit cc = new HttpSubmit();
		try {
			ArgumentsParser parser = null;
			synchronized (OptionBuilderLock.getLock()) {
				// parses args
				// -jcl mandatory arg
				Option jcl = OptionBuilder.withArgName(JCL).hasArg().withDescription("use given jcl file").create(JCL);
				jcl.setRequired(true);
				// -type mandatory arg
				Option type = OptionBuilder.withArgName(TYPE).hasArg().withDescription("use given jcl type").create(TYPE);
				type.setRequired(true);
				// -host mandatory arg
				Option host = OptionBuilder.withArgName(HOST).hasArg().withDescription("use given host to submit the job").create(HOST);
				host.setRequired(true);
				// -nowait optional arg
				Option wait = OptionBuilder.withArgName(WAIT).withDescription("use given to wait for end of job").create(WAIT);
				// -user optional arg
				Option user = OptionBuilder.withArgName(USER).hasArg().withDescription("use given userid to login into JEM").create(USER);
				host.setRequired(true);
				// -password optional arg
				Option pwd = OptionBuilder.withArgName(PASSWORD).hasArg().withDescription("use given password to login into JEM").create(PASSWORD);
				// parses all arguments
				parser = new ArgumentsParser(HttpSubmit.class.getName(), jcl, type, host, user, pwd, wait);
			}

			// saves all arguments in common variables
			Properties properties = parser.parseArg(args);
			cc.jclUrl = properties.getProperty(JCL);
			cc.hostForJemWeb = properties.getProperty(HOST);
			cc.jclType = properties.getProperty(TYPE);
			cc.passwordValue = properties.getProperty(PASSWORD);
			cc.userValue = properties.getProperty(USER);
			cc.wait = properties.containsKey(WAIT);

			if (cc.passwordValue == null) {
				// creates console to read password
				CmdConsole console = new CmdConsole();
				cc.passwordValue = console.readPassword(cc.userValue);
			}
			// submits job
			rc = cc.submitJob();
		} catch (Exception e) {
			// with any exception, ends in CC 1
			LogAppl.getInstance().emit(NodeMessage.JEMC033E, e);
			// sets return code to error
			rc = 1;
		}
		return rc;
	}

	/**
	 * Main method! Parses the arguments, submits job by HTTP, exit with return
	 * code, 0 for success 1 for error
	 * 
	 * @param args command-line arguments
	 */
	public static void main(String[] args) {
		int rc = executeCommand(args);
		System.exit(rc);
	}
}