/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.ant.tasks.utilities;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Properties;

import javax.naming.InitialContext;

import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.tools.ant.BuildException;
import org.pepstock.jem.ant.AntMessage;
import org.pepstock.jem.ant.tasks.DataDescription;
import org.pepstock.jem.ant.tasks.StepJava;
import org.pepstock.jem.ant.tasks.utilities.sort.DefaultComparator;
import org.pepstock.jem.commands.util.ArgumentsParser;
import org.pepstock.jem.node.tasks.jndi.ContextUtils;

/**
 * Is a utility (both a task ANT and a main program) that sort data.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class SortTask extends StepJava {
	
	private static int DEFAULTMAXTEMPFILES = 1024;

	private static final String INPUT_DATA_DESCRIPTION_NAME = "INPUT";
	
	private static final String OUTPUT_DATA_DESCRIPTION_NAME = "OUTPUT";

	/**
	 * Key for the class to load to transform and load data  
	 */
	private static String CLASS = "class";

	/**
	 * Empty constructor
	 */
	public SortTask() {
	}

	/**
	 * Sets itself as main program and calls <code>execute</code> method of
	 * superclass (StepJava).<br>
	 * Checks the necessary data descriptions are defined otherwise an exception will occur
	 * 
	 * @throws BuildException occurs if an error occurs
	 */
	@Override
	public void execute() throws BuildException {
		// checks before execution if has INPUT and OUTPUT
		// data description
		boolean input = false;
		boolean output = false;
		for (DataDescription dd : super.getDataDescriptions()){
			if (dd.getName().equalsIgnoreCase(INPUT_DATA_DESCRIPTION_NAME))
				input = true;
			if (dd.getName().equalsIgnoreCase(OUTPUT_DATA_DESCRIPTION_NAME))
				output = true;
		}
		
		if (!input){
			throw new BuildException(AntMessage.JEMA018E.toMessage().getFormattedMessage(INPUT_DATA_DESCRIPTION_NAME));
		}
		if (!output){
			throw new BuildException(AntMessage.JEMA018E.toMessage().getFormattedMessage(OUTPUT_DATA_DESCRIPTION_NAME));
		}
		
		super.setClassname(SortTask.class.getName());
		
		/**
		 * Issue 168
		 */
//		Path classPath =super.createClasspath();
//		String classPathString = System.getProperty("java.class.path");
//		classPath.setPath(classPathString);
//		super.setClasspath(classPath);

		super.execute();
	}

	
	/**
	 * Divides the file into small blocks. If the blocks
	 * are too small, we shall create too many temporary files.
	 * If they are too big, we shall be using too much memory.
	 * 
	 * @param filetobesorted
	 * @param maxtmpfiles
	 * @return block size
	 * @throws IOException 
	 */
	public static long estimateBestSizeOfBlocks(FileInputStream filetobesorted, int maxtmpfiles) throws IOException {
		long sizeoffile = filetobesorted.getChannel().size() * 2;
		/**
		 * We multiply by two because later on someone insisted on counting the
		 * memory usage as 2 bytes per character. By this model, loading a file
		 * with 1 character will use 2 bytes.
		 */
		// we don't want to open up much more than maxtmpfiles temporary files,
		// better run
		// out of memory first.
		long blocksize = sizeoffile / maxtmpfiles + (sizeoffile % maxtmpfiles == 0 ? 0 : 1);

		// on the other hand, we don't want to create many temporary files
		// for naught. If blocksize is smaller than half the free memory, grow
		// it.
		long freemem = Runtime.getRuntime().freeMemory();
		if (blocksize < freemem / 2) {
			blocksize = freemem / 2;
		}
		return blocksize;
	}

	/**
	 * This will simply load the file by blocks of x rows, then sort them
	 * in-memory, and write the result to temporary files that have to be merged
	 * later.
	 * 
	 * @param fileInput some flat file
	 * @param cmp string comparator
	 * @return a list of temporary flat files
	 * @throws IOException 
	 */
	public static List<File> sortInBatch(FileInputStream fileInput, Comparator<String> cmp) throws IOException {
		return sortInBatch(fileInput, cmp, DEFAULTMAXTEMPFILES, Charset.defaultCharset(), null);
	}

	/**
	 * This will simply load the file by blocks of x rows, then sort them
	 * in-memory, and write the result to temporary files that have to be merged
	 * later. You can specify a bound on the number of temporary files that will
	 * be created.
	 * 
	 * @param fileInput some flat file
	 * @param cmp string comparator
	 * @param maxtmpfiles maximal number of temporary files
	 * @param cs 
	 * @param Charset character set to use (can use Charset.defaultCharset())
	 * @param tmpdirectory location of the temporary files (set to null for
	 *            default location)
	 * @return a list of temporary flat files
	 * @throws IOException 
	 */
	public static List<File> sortInBatch(FileInputStream fileInput, Comparator<String> cmp, int maxtmpfiles, Charset cs, File tmpdirectory) throws IOException {
		List<File> files = new ArrayList<File>();
		BufferedReader fbr = new BufferedReader(new InputStreamReader(fileInput, cs));
		long blocksize = estimateBestSizeOfBlocks(fileInput, maxtmpfiles);// in bytes

		try {
			List<String> tmplist = new ArrayList<String>();
			String line = "";
			try {
				while (line != null) {
					long currentblocksize = 0;// in bytes
					while ((currentblocksize < blocksize) && ((line = fbr.readLine()) != null)) { // as
																									// long
																									// as
																									// you
																									// have
																									// enough
																									// memory
						tmplist.add(line);
						currentblocksize += line.length() * 2; // java uses 16
																// bits per
																// character?
					}
					files.add(sortAndSave(tmplist, cmp, cs, tmpdirectory));
					tmplist.clear();
				}
			} catch (EOFException oef) {
				if (tmplist.size() > 0) {
					files.add(sortAndSave(tmplist, cmp, cs, tmpdirectory));
					tmplist.clear();
				}
			}
		} finally {
			fbr.close();
		}
		return files;
	}

	/**
	 * Sort a list and save it to a temporary file
	 * 
	 * @return the file containing the sorted data
	 * @param tmplist data to be sorted
	 * @param cmp string comparator
	 * @param cs charset to use for output (can use Charset.defaultCharset())
	 * @param tmpdirectory location of the temporary files (set to null for
	 *            default location)
	 * @throws IOException 
	 */
	public static File sortAndSave(List<String> tmplist, Comparator<String> cmp, Charset cs, File tmpdirectory) throws IOException {
		Collections.sort(tmplist, cmp);
		File newtmpfile = File.createTempFile("sortInBatch", "flatfile", tmpdirectory);
		newtmpfile.deleteOnExit();
		BufferedWriter fbw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(newtmpfile), cs));
		try {
			for (String r : tmplist) {
				fbw.write(r);
				fbw.newLine();
			}
		} finally {
			fbw.close();
		}
		return newtmpfile;
	}

	/**
	 * This merges a bunch of temporary flat files
	 * 
	 * @param files
	 * @param fileOutput 
	 * @param cmp 
	 * @param output file
	 * @return The number of lines sorted.
	 * @throws IOException 
	 */
	public static int mergeSortedFiles(List<File> files, FileOutputStream fileOutput, final Comparator<String> cmp) throws IOException {
		return mergeSortedFiles(files, fileOutput, cmp, Charset.defaultCharset());
	}

	/**
	 * This merges a bunch of temporary flat files
	 * 
	 * @param files
	 * @param fileOutput 
	 * @param cmp 
	 * @param cs 
	 * @param output file
	 * @param Charset character set to use to load the strings
	 * @return The number of lines sorted.
	 * @throws IOException 
	 */
	public static int mergeSortedFiles(List<File> files, FileOutputStream fileOutput, final Comparator<String> cmp, Charset cs) throws IOException {
		PriorityQueue<BinaryFileBuffer> pq = new PriorityQueue<BinaryFileBuffer>(11, new Comparator<BinaryFileBuffer>() {
			public int compare(BinaryFileBuffer i, BinaryFileBuffer j) {
				return cmp.compare(i.peek(), j.peek());
			}
		});
		for (File f : files) {
			BinaryFileBuffer bfb = new BinaryFileBuffer(f, cs);
			pq.add(bfb);
		}
		BufferedWriter fbw = new BufferedWriter(new OutputStreamWriter(fileOutput, cs));
		int rowcounter = 0;
		try {
			while (pq.size() > 0) {
				BinaryFileBuffer bfb = pq.poll();
				String r = bfb.pop();
				fbw.write(r);
				fbw.newLine();
				++rowcounter;
				if (bfb.empty()) {
					bfb.fbr.close();
					boolean isDeleted = bfb.originalfile.delete();// we don't need you anymore
	            	if (!isDeleted){
	            		// nop
	            	}
				} else {
					pq.add(bfb); // add it back
				}
			}
		} finally {
			fbw.close();
			for (BinaryFileBuffer bfb : pq)
				bfb.close();
		}
		return rowcounter;
	}

	/**
	 * Main program, called by StepJava class. It reads from dd defined as INPUT and writes in OUTPUT one
	 * 
	 * @param args has a optional comaprator class to use to sort
	 * @throws Exception if data description doesn't exists, if an
	 *             error occurs during copying
	 */
	@SuppressWarnings({ "resource", "static-access", "unchecked" })
	public static void main(String[] args) throws Exception {
		// -class mandatory arg
		Option classArg = OptionBuilder.withArgName(CLASS).hasArg().withDescription("class of Comparator<String> to invoke reading the objects").create(CLASS);
		classArg.setRequired(false);

		// parses all arguments
		ArgumentsParser parser = new ArgumentsParser(SortTask.class.getName(), classArg);
		
		// saves all arguments in common variables
		Properties properties = parser.parseArg(args);
		
		String classParam = properties.getProperty(CLASS);

		Comparator<String> comparator = null;
		if (classParam !=null) {
			Object objectTL = Class.forName(classParam).newInstance();
			if (objectTL instanceof Comparator<?>) {
				System.out.println(AntMessage.JEMA042I.toMessage().getFormattedMessage(objectTL.getClass().getName()));
				comparator = (Comparator<String>) objectTL;
			} else {
				System.out.println(AntMessage.JEMA043I.toMessage().getFormattedMessage());
				comparator = new DefaultComparator();
				
			}
		} else {
			System.out.println(AntMessage.JEMA043I.toMessage().getFormattedMessage());
			comparator = new DefaultComparator();
		}
		
		int maxtmpfiles = DEFAULTMAXTEMPFILES;
		Charset cs = Charset.defaultCharset();

		// new initial context to access by JNDI to COMMAND DataDescription
		InitialContext ic = ContextUtils.getContext();

		// gets inputstream
		Object input = (Object) ic.lookup(INPUT_DATA_DESCRIPTION_NAME);
		// gets outputstream
		Object output = (Object) ic.lookup(OUTPUT_DATA_DESCRIPTION_NAME);

		FileInputStream istream = null;
		FileOutputStream ostream = null;

		// checks if object is a inputstream otherwise error
		if (input instanceof FileInputStream){
			istream = (FileInputStream) input;
		} else {
			throw new Exception(AntMessage.JEMA017E.toMessage().getFormattedMessage(INPUT_DATA_DESCRIPTION_NAME, input.getClass().getName()));
		}
		// checks if object is a outputstream otherwise error
		if (output instanceof FileOutputStream){
			ostream = (FileOutputStream) output;
		} else {
			throw new Exception(AntMessage.JEMA017E.toMessage().getFormattedMessage(OUTPUT_DATA_DESCRIPTION_NAME, output.getClass().getName()));
		}



		if (istream.getChannel().size() > 0){
			List<File> l = sortInBatch(istream, comparator, maxtmpfiles, cs, null);
			mergeSortedFiles(l, ostream, comparator, cs);
		}

	}
}

/**
 * Buffer which contains data to sort
 */
class BinaryFileBuffer {
	public static int BUFFERSIZE = 2048;
	public BufferedReader fbr;
	public File originalfile;
	private String cache;
	private boolean empty;

	/**
	 * 
	 * @param f
	 * @param cs
	 * @throws IOException
	 */
	public BinaryFileBuffer(File f, Charset cs) throws IOException {
		originalfile = f;
		fbr = new BufferedReader(new InputStreamReader(new FileInputStream(f), cs), BUFFERSIZE);
		reload();
	}

	/**
	 * @return true if empty, otherwise false
	 */
	public boolean empty() {
		return empty;
	}

	/**
	 * 
	 * @throws IOException
	 */
	private void reload() throws IOException {
		try {
			if ((this.cache = fbr.readLine()) == null) {
				empty = true;
				cache = null;
			} else {
				empty = false;
			}
		} catch (EOFException oef) {
			empty = true;
			cache = null;
		}
	}

	/**
	 * 
	 * @throws IOException
	 */
	public void close() throws IOException {
		fbr.close();
	}

	/**
	 * @return null if is empty otherwise cache string
	 */
	public String peek() {
		if (empty())
			return null;
		return cache.toString();
	}

	/**
	 * @return peek value string
	 * @throws IOException
	 */
	public String pop() throws IOException {
		String answer = peek();
		reload();
		return answer;
	}

}