/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.ant.tasks.utilities;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;

import javax.naming.InitialContext;

import org.apache.tools.ant.BuildException;
import org.pepstock.jem.ant.AntMessage;
import org.pepstock.jem.ant.tasks.DataDescription;
import org.pepstock.jem.ant.tasks.StepJava;
import org.pepstock.jem.node.tasks.jndi.ContextUtils;

/**
 * Is a utility (both a task ANT and a main program) that copy data from and to a data description.<br>
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class CopyTask extends StepJava {
	
	private static final int BUF_SIZE = 8192;

	private static final String INPUT_DATA_DESCRIPTION_NAME = "INPUT";
	
	private static final String OUTPUT_DATA_DESCRIPTION_NAME = "OUTPUT";

	/**
	 * Empty constructor
	 */
	public CopyTask() {
	}

	/**
	 * Sets itself as main program and calls <code>execute</code> method of
	 * superclass (StepJava).<br>
	 * Checks the necessary data descriptions are defined otherwise an exception will occur
	 * 
	 * @throws BuildException occurs if an error occurs
	 */
	@Override
	public void execute() throws BuildException {
		// checks before execution if has INPUT and OUTPUT
		// data description
		boolean input = false;
		boolean output = false;
		for (DataDescription dd : super.getDataDescriptions()){
			if (dd.getName().equalsIgnoreCase(INPUT_DATA_DESCRIPTION_NAME))
				input = true;
			if (dd.getName().equalsIgnoreCase(OUTPUT_DATA_DESCRIPTION_NAME))
				output = true;
		}
		
		if (!input){
			throw new BuildException(AntMessage.JEMA018E.toMessage().getFormattedMessage(INPUT_DATA_DESCRIPTION_NAME));
		}
		if (!output){
			throw new BuildException(AntMessage.JEMA018E.toMessage().getFormattedMessage(OUTPUT_DATA_DESCRIPTION_NAME));
		}
		
		super.setClassname(CopyTask.class.getName());

		/**
		 * Issue 168
		 */
//		Path classPath =super.createClasspath();
//		String classPathString = System.getProperty("java.class.path");
//		classPath.setPath(classPathString);
//		super.setClasspath(classPath);

		super.execute();
	}

	/**
	 * Main program, called by StepJava class. It reads from dd defined as INPUT and writes in OUTPUT one
	 * 
	 * @param args not used
	 * @throws Exception if data description data description doesn't exists, if an
	 *             error occurs during copying
	 */
	@SuppressWarnings("resource")
	public static void main(String[] args) throws Exception {
		// new initial context to access by JNDI to COMMAND DataDescription
		InitialContext ic = ContextUtils.getContext();

		// gets inputstream
		Object input = (Object) ic.lookup(INPUT_DATA_DESCRIPTION_NAME);
		// gets outputstream
		Object output = (Object) ic.lookup(OUTPUT_DATA_DESCRIPTION_NAME);

		InputStream istream = null;
		OutputStream ostream = null;

		// checks if object is a inputstream otherwise error
		if (input instanceof InputStream){
			istream = (InputStream) input;
		} else {
			throw new Exception(AntMessage.JEMA017E.toMessage().getFormattedMessage(INPUT_DATA_DESCRIPTION_NAME, input.getClass().getName()));
		}
		// checks if object is a outputstream otherwise error
		if (output instanceof OutputStream){
			ostream = (OutputStream) output;
		} else {
			throw new Exception(AntMessage.JEMA016E.toMessage().getFormattedMessage(OUTPUT_DATA_DESCRIPTION_NAME, output.getClass().getName()));
		}

		// if the input stream is a fileinput stream it uses the channel (faster!) 
		if (istream instanceof FileInputStream){
			FileChannel ichannel = ((FileInputStream)istream).getChannel();
			ichannel.transferTo(0, ichannel.size(), Channels.newChannel(ostream));
			ichannel.close();
		} else if (ostream instanceof FileOutputStream){ 
			// if the output stream is a fileoutput stream it uses the channel (faster!)
			FileChannel ochannel = ((FileOutputStream)ostream).getChannel();
			ochannel.transferFrom(Channels.newChannel(istream), 0, Long.MAX_VALUE);
			ochannel.close();
		} else {
			// otherwise normal read and write
			byte[] buffer = new byte[BUF_SIZE];
			int count = 0;
			do {
				ostream.write(buffer, 0, count);
				count = istream.read(buffer, 0, buffer.length);
			} while (count != -1);
			ostream.close();
			istream.close();
		}
	}
}