/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.ant.tasks.managers;

import java.text.MessageFormat;
import java.util.HashMap;

import org.apache.commons.lang3.StringUtils;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.PropertyHelper;
import org.pepstock.jem.ant.DataDescriptionStep;

/**
 * Is singleton object. Only one instance of this container must instantiated.<br>
 * Contains all return code for all step which are able to manage the RC.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.4
 * 
 */
public class ReturnCodesContainer {

	private static final String level_separator = ".";
	
	private static final int level_without_id = 2;
	
	private static final int level_with_id = 3;
	
	// format used for searching by target, task  and id
	private static final MessageFormat mf = new MessageFormat("{0}.{1}.{2}");

	private static ReturnCodesContainer CONTAINER = null;

	private HashMap<String, Integer> mapReturnCodes = null;

	/**
	 * Is singleton static method to retrieve the object. Only one instance of
	 * this container must instantiated.<br>
	 * If static instance is null, creates a new one.
	 * 
	 * @return container instance
	 */
	public static ReturnCodesContainer getInstance() {
		if (CONTAINER == null) {
			CONTAINER = new ReturnCodesContainer();
		}
		return CONTAINER;
	}

	/**
	 * Private constructor which screates the map to save the references of data
	 * description
	 */
	private ReturnCodesContainer() {
		mapReturnCodes = new HashMap<String, Integer>();
	}

	/**
	 * Returns <code>true</code> if container contains the step, otherwise
	 * <code>false</code>.
	 * 
	 * @param step representation
	 * @return <code>true</code> if container contains the step
	 */
	public boolean hasReturnCode(DataDescriptionStep step) {
		return hasReturnCode(createKey(step));
	}

	/**
	 * Returns return found by reference, otherwise <code>null</code>.
	 * 
	 * @param step step representation
	 * @return return code instance
	 */
	public Integer getReturnCode(DataDescriptionStep step) {
		return getReturnCode(createKey(step));
	}

	/**
	 * Returns <code>true</code> if container contains the reference, otherwise
	 * <code>false</code>.
	 * 
	 * @param reference string reference representation
	 * @return <code>true</code> if container contains the reference
	 */
	public boolean hasReturnCode(String reference) {
		String ref = normalizeReference(reference);
		return (ref != null) ? mapReturnCodes.containsKey(ref.toLowerCase()) : false;	
	}

	/**
	 * Returns return found by reference, otherwise <code>null</code>.
	 * 
	 * @param reference reference string representation
	 * @return return code instance
	 */
	public Integer getReturnCode(String reference) {
		String ref = normalizeReference(reference);
		return (ref != null) ? mapReturnCodes.get(ref.toLowerCase()) : null;	
	}

	/**
	 * Adds new data description implementation, defined for passed target and
	 * task.
	 * 
	 * @param project ANT project
	 * @param item task in executing
	 * @param rc 
	 * @param task task name
	 * @param dd data description implementation
	 */
	public void setReturnCode(Project project, DataDescriptionStep item, Integer rc) {
		// create a key using message format defined for reference
		String key = createKey(item);
		mapReturnCodes.put(key, rc);
		
		PropertyHelper.getPropertyHelper(project).setNewProperty(key, rc);
	}

	/**
	 * Creates a key using the format defined for searching
	 * 
	 * @see ReturnCodesContainer#mf
	 * @param target target name
	 * @param task task name
	 * @return the key of map (always lower-case)
	 */
	private String createKey(DataDescriptionStep item) {
		String key;
		key = mf.format(new Object[] { item.getTargetName(), item.getTaskName(), item.getId() }, new StringBuffer(), null).toString();
		return key.toLowerCase();
	}
	
	/**
	 * 
	 * @param reference
	 * @return
	 */
	private String normalizeReference(String reference){
		String[] levels = StringUtils.split(reference, level_separator);
		if (levels != null){
			if (levels.length == level_with_id){
				return reference;
			} else if (levels.length == level_without_id){
				reference = levels[0] + level_separator + // target
						levels[1] + level_separator + // task
						DataDescriptionStep.DEFAULT_ID; // id
				return reference;		
			}
		}
		return null;
	}

	/**
	 * Returns the string representation of data description container (uses
	 * HaspMap to string method).
	 * 
	 * @see java.util.HashMap#toString()
	 * @return the string representation of data description container
	 */
	@Override
	public String toString() {
		return mapReturnCodes.toString();
	}

}